package cn.com.duiba.cloud.log.client.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.log.client.BusinessLoggerMethod;
import cn.com.duiba.cloud.log.client.integration.BusinessLoggerInterceptor;
import cn.com.duiba.cloud.log.client.service.context.BusinessLoggerStackFrame;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.core.annotation.AnnotationUtils;

import javax.annotation.Resource;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Objects;

/**
 * 日志切面
 *
 * @author liuyao
 */
@Slf4j
@Aspect
public class BusinessLoggerProxy {

    public static final String RESPONSE = "response";

    private final LocalVariableTableParameterNameDiscoverer discoverer = new LocalVariableTableParameterNameDiscoverer();

    @Resource
    private BusinessLoggerService businessLoggerService;
    @Autowired(required = false)
    private List<BusinessLoggerInterceptor> businessLoggerInterceptors;

    @Around("@annotation(cn.com.duiba.cloud.log.client.BusinessLoggerMethod)")
    public Object logger(ProceedingJoinPoint point) throws Throwable {
        MethodSignature methodSignature = (MethodSignature) point.getSignature();
        Method method = methodSignature.getMethod();
        BusinessLoggerMethod loggerMethod = AnnotationUtils.findAnnotation(method, BusinessLoggerMethod.class);
        String template = Objects.requireNonNull(loggerMethod).template();
        String group = loggerMethod.group();
        if (StringUtils.isBlank(template)) {
            throw new IllegalArgumentException("操作日志: " + group + ", template不能为空");
        }
        BusinessLoggerStackFrame frame = new BusinessLoggerStackFrame();
        frame.setTemplate(template);
        frame.setGroup(group);
        businessLoggerService.initStackContext(frame);

        if (loggerMethod.request()) {
            Object[] arguments = point.getArgs();
            String[] paramNames = discoverer.getParameterNames(method);
            for (int i = 0; i < Objects.requireNonNull(paramNames).length; i++) {
                String paramName = paramNames[i];
                Object value = arguments[i];
                frame.addParam(paramName, value);
            }
        }
        boolean persistence = doInterceptor(frame);
        Object result;
        try {
            result = point.proceed();
            if (loggerMethod.response()) {
                frame.addParam(RESPONSE, result);
            }
            return result;
        } catch (Exception e) {
            //记录为异常日志
            Throwable exception = e;
            while (exception instanceof InvocationTargetException) {
                exception = ((InvocationTargetException) exception).getTargetException();
            }
            frame.setErrorLog(!(exception instanceof BizException));
            //将业务异常msg替代为RESPONSE塞入
            if (Objects.nonNull(e) && StringUtils.isNotBlank(e.getMessage())) {
                String msg = e.getMessage().length() > 1000 ? e.getMessage().substring(0, 1000) : e.getMessage();
                frame.addParam(RESPONSE, msg);
            }
            throw e;
        } finally {
            businessLoggerService.flushThreadContext(persistence);
        }
    }

    private boolean doInterceptor(BusinessLoggerStackFrame frame) {
        for (BusinessLoggerInterceptor interceptor : businessLoggerInterceptors) {
            boolean persistence = interceptor.doHandler(frame);
            if (!persistence) {
                return false;
            }
        }
        return true;
    }


}