package cn.com.duiba.cloud.duiba.openapi.service.api.config;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.duiba.openapi.service.api.param.ApiRegisterParam;
import cn.com.duiba.cloud.duiba.openapi.service.api.remoteservice.RemoteApiConfigService;
import cn.com.duiba.cloud.openapi.service.base.annotation.OpenApi;
import cn.com.duiba.cloud.openapi.service.base.annotation.OpenPath;
import cn.hutool.extra.spring.SpringUtil;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.Map;
import java.util.Optional;

/**
 * 开放接口注册配置
 * 在程序启动成功之后执行
 *
 * @author huangguosheng@duiba.com.cn
 * @date 2021/12/20 11:08 上午
 **/
@Slf4j
@Configuration
@EnableConfigurationProperties(OpenApiProperty.class)
@ConditionalOnProperty(prefix = "openapi", name = "enable", havingValue = "true")
public class OpenApiConfiguration implements ApplicationRunner {

    @Autowired
    private OpenApiProperty openApiProperty;

    @Autowired
    private RemoteApiConfigService remoteApiConfigService;

    @Value("${spring.application.name}")
    private String serviceName;

    @Override
    public void run(ApplicationArguments args) throws Exception {
        this.initOpenApi();
    }

    public void initOpenApi() throws BizException, ClassNotFoundException {
        // 获取OpenApi注解的类型
        final Map<String, Object> beansWithAnnotation = SpringUtil.getApplicationContext().getBeansWithAnnotation(OpenApi.class);
        for (Map.Entry<String, Object> entry : beansWithAnnotation.entrySet()) {
            Object obj = entry.getValue();
            if (!entry.getKey().endsWith("Impl")) {
                continue;
            }
            OpenApi openApi = AnnotationUtils.findAnnotation(obj.getClass(), OpenApi.class);
            assert openApi != null;
            String prefix = this.dealPath(openApi.prefix().trim());
            Method[] methods = ReflectionUtils.getDeclaredMethods(obj.getClass());
            for (Method method : methods) {
                // 获取指定方法上的注解的属性
                final OpenPath openPath = AnnotationUtils.findAnnotation(method, OpenPath.class);
                if (null == openPath) {
                    continue;
                }

                Class<?>[] parameterTypes = method.getParameterTypes();
                if (parameterTypes.length > 1) {
                    throw new IllegalArgumentException(MessageFormat.format("{0}#{1}参数必须封装成简单对象，否则无法解析", obj.getClass().getSimpleName(), method.getName()));
                }

                // 验证必要的注解的属性
                String apiPath = Optional.ofNullable(openPath.apiPath()).orElseThrow(() -> new IllegalArgumentException("请设置ApiPath路径 例如：/test"));
                apiPath = prefix + "/" + this.dealPath(apiPath.trim());


                log.info("成功读取OpenApi路径：{}", apiPath);

                //  注册到开放平台
                ApiRegisterParam registerParam = new ApiRegisterParam();
                registerParam.setApiPath(apiPath);
                registerParam.setApiDesc(openPath.apiDesc());
                registerParam.setServiceName(serviceName);
                String className = obj.getClass().getTypeName();
                Class<?>[] interfaces = Class.forName(className.substring(0, className.lastIndexOf("Impl")) + "Impl").getInterfaces();
                registerParam.setClassName(interfaces.length == 0 ? className : interfaces[0].getTypeName());
                registerParam.setMethod(method.getName());
                if (1 == parameterTypes.length) {
                    registerParam.setMethodParamClass(parameterTypes[0].getTypeName());
                }
                registerParam.setRouteClass(openApiProperty.getRouteClass());
                try {
                    remoteApiConfigService.register(registerParam);
                    log.info("OpenApi注册成功：{}", JSON.toJSONString(registerParam));
                } catch (Exception e) {
                    log.error("OpenApi注册失败：{}", JSON.toJSONString(registerParam), e);
                }

            }
        }
    }

    /**
     * 路径处理
     *
     * @param apiPath
     * @return
     */
    private String dealPath(String apiPath) {
        // 处理去头部斜杆
        if (0 == apiPath.indexOf("/")) {
            return apiPath.substring(1);
        }
        return apiPath;
    }


}
