package cn.com.duiba.cloud.duiba.openapi.web.api.config;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.duiba.openapi.web.api.constant.AutoLoginConstant;
import cn.com.duiba.cloud.duiba.openapi.web.api.dto.ConsumerCookieDTO;
import cn.com.duiba.cloud.duiba.openapi.web.api.dto.LoginCookieDTO;
import cn.com.duiba.cloud.duiba.openapi.web.api.param.AutoLoginAllParam;
import cn.com.duiba.cloud.duiba.openapi.web.api.param.DecryptDataParam;
import cn.com.duiba.cloud.duiba.openapi.web.api.remoteservice.RemoteAutoLoginService;
import cn.com.duiba.cloud.duiba.openapi.web.api.utils.CookieUtil;
import cn.hutool.core.util.EscapeUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

/**
 * 开放请求，试用于开放平台的免登接口
 *
 * @author huangguosheng@duiba.com.cn
 * @date 2022/3/22 3:42 下午
 **/
@Slf4j
public class OpenRequest {
    /**
     * 获取当前用户信息
     * 注意点：
     * 1、用户信息获取失败会返回null，接入方更具情况对外抛出异常
     * 2、有可能会出现窜数据情况（相同浏览器，appA留下的数据，appB来），接入方需要对appId进行判断
     *
     * @return
     * @throws BizException
     */
    public static ConsumerCookieDTO getConsumerInfo() throws BizException {
        String timestamp = "";
        String encodeConsumerData = "";
        // 获取cookie中的数据
        Cookie[] cookies = CookieUtil.getCookies();
        for (Cookie cookie : cookies) {
            if (AutoLoginConstant.CookieName.LOGIN_TIME_COOKIE.equals(cookie.getName())) {
                timestamp = cookie.getValue();
            } else if (AutoLoginConstant.CookieName.LOGIN_CONSUMER_COOKIE.equals(cookie.getName())) {
                encodeConsumerData = cookie.getValue();
            }
        }

        if (StringUtils.isBlank(timestamp) || StringUtils.isBlank(encodeConsumerData)) {
            return null;
        }

        // 解密数据
        try {
            RemoteAutoLoginService remoteAutoLoginService = SpringUtil.getApplicationContext().getBean(RemoteAutoLoginService.class);
            DecryptDataParam decryptDataParam = DecryptDataParam.builder().data(encodeConsumerData).timestamp(Long.valueOf(timestamp)).build();
            String dataJson = remoteAutoLoginService.cookieDataDecrypt(decryptDataParam);
            return JSON.parseObject(dataJson, ConsumerCookieDTO.class);
        } catch (Exception e) {
            log.error("用户信息获取失败,可能登陆信息已过期或者假数据", e);
            return null;
        }
    }

    /**
     * 获取Transfer信息
     *
     * @return
     */
    public static String getTransfer() {
        return CookieUtil.getCookieValue(AutoLoginConstant.CookieName.TRANSFER);
    }

    /**
     * 获取dcustom信息
     *
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String getDCustom() throws UnsupportedEncodingException {
        String dCustomValue = CookieUtil.getCookieValue(AutoLoginConstant.CookieName.DCUSTOM);
        if (StringUtils.isBlank(dCustomValue)) {
            return null;
        }
        return URLDecoder.decode(dCustomValue, "utf-8");
    }


    /**
     * 老兑吧免登时用
     * 初始化用户信息及cookie信息
     *
     * @param param
     * @throws UnsupportedEncodingException
     */
    public static void initConsumerAndCookie(AutoLoginAllParam param) throws UnsupportedEncodingException {
        // 免登前置处理，保存用户信息，加密用户数据
        RemoteAutoLoginService remoteAutoLoginService = SpringUtil.getApplicationContext().getBean(RemoteAutoLoginService.class);
        LoginCookieDTO loginCookieDTO = remoteAutoLoginService.autoLoginPre(param);

        // 设置登陆cookie
        setLoginCookie(loginCookieDTO);
    }

    /**
     * 设置cookie
     * 1、要存储哪些cookie内容(用户信息、开发者传递信息)  2、对cookie内容做加密 3、获取cookie内容的方法
     *
     * @throws UnsupportedEncodingException
     */
    public static void setLoginCookie(LoginCookieDTO param) throws UnsupportedEncodingException {
        // 登录请求，用于把用户信息放入cookie(包括open_consumer_data、open_timestamp两个cookie)
        Long loginTime = System.currentTimeMillis();
        String domain = param.getDomain();
        // 1、用户信息
        CookieUtil.setCookie(AutoLoginConstant.CookieName.LOGIN_CONSUMER_COOKIE, param.getUserEncryptCookieVal());

        // 2、登录时间
        CookieUtil.setCookie(AutoLoginConstant.CookieName.LOGIN_TIME_COOKIE, loginTime.toString());

        // 3、是否是未登录用户
        CookieUtil.setCrossCookieFor24Hour(AutoLoginConstant.CookieName.IS_NOT_LOGIN_USER,
                String.valueOf(isNotLoginUser(param.getPartnerUserId())), domain);


        // 4、transfer额外处理。开发者写入的参数，需要在扣积分时携带给开发者
        String transfer = param.getTransfer();
        if (StringUtils.isNotBlank(transfer) && transfer.length() < 100) {
            CookieUtil.setCrossCookieFor24Hour(AutoLoginConstant.CookieName.TRANSFER, transfer, domain);
        } else {
            CookieUtil.deleteCookie(AutoLoginConstant.CookieName.TRANSFER, domain);
        }

        // 5、开发者自己的参数
        String dcustom = param.getDcustom();
        if (StringUtils.isNotBlank(dcustom)) {
            dcustom = EscapeUtil.unescapeHtml4(dcustom);
            CookieUtil.setCrossCookieFor24Hour(AutoLoginConstant.CookieName.DCUSTOM, URLEncoder.encode(dcustom, "utf-8"), domain);
        } else {
            CookieUtil.deleteCookie(AutoLoginConstant.CookieName.DCUSTOM, domain);
        }
    }


    private static boolean isNotLoginUser(String partnerUserId) {
        return StringUtils.isNotBlank(partnerUserId)
                && (AutoLoginConstant.NOT_LOGIN_USER_ID.equals(partnerUserId)
                || partnerUserId.startsWith(AutoLoginConstant.NOT_LOGIN_USER_ID_GEN_NEW)
                || AutoLoginConstant.PREVIEW_USER.equals(partnerUserId));
    }

}
