package cn.com.duiba.cloud.stock.service.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.cloud.stock.service.api.dto.stock.ConsumeStockResultDTO;
import cn.com.duiba.cloud.stock.service.api.dto.stock.StockDTO;
import cn.com.duiba.cloud.stock.service.api.dto.stock.StockInfoDTO;
import cn.com.duiba.cloud.stock.service.api.dto.stock.StockLotDTO;
import cn.com.duiba.cloud.stock.service.api.dto.stock.StockOccupyGeneralInfoDTO;
import cn.com.duiba.cloud.stock.service.api.dto.stock.StockOccupyInfoDTO;
import cn.com.duiba.cloud.stock.service.api.enums.stock.StockOccupyEnum;
import cn.com.duiba.cloud.stock.service.api.param.stock.ConsumeStockParam;
import cn.com.duiba.cloud.stock.service.api.param.stock.StockOccupyGeneralInfoParam;
import cn.com.duiba.cloud.stock.service.api.param.stock.UpdateOccupyStockParam;
import cn.com.duiba.cloud.stock.service.api.param.stock.UpdateStockStatusParam;
import cn.com.duiba.wolf.entity.PageResponse;

import java.util.List;

/**
 * 库存中心remote接口
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 1.0
 * @date 2021/11/20 1:33 下午
 **/
@AdvancedFeignClient
public interface RemoteStockService {
    /**
     * 通过skuId批量获取库存信息
     *
     * @param skuIdList skuId列表
     * @param userId    用户id
     * @param userType  用户类型(1:平台;2:租户;3:用户)
     * @return 库存信息 {@link StockInfoDTO}
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>请求参数有误</li>
     *                          <li>不支持多级分销</li>
     *                      </ul>
     */
    List<StockInfoDTO> batchGetStockInfo(List<Long> skuIdList, Long userId, Integer userType) throws BizException;

    /**
     * 获取库存信息<br>
     * 只局限于自建库存和预采购库存
     *
     * @param skuId    skuId
     * @param userId   创建商品的用户id
     * @param userType 创建商品的用户类型
     * @return 库存信息
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>请求参数有误</li>
     *                      </ul>
     */
    List<StockDTO> getStockInfo(Long skuId, Long userId, Integer userType) throws BizException;

    /**
     * 根据库存id查询库存批次列表
     *
     * @param stockId 库存id
     * @return 库次年批次列表
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>库存id不能为空</li>
     *                      </ul>
     */
    List<StockLotDTO> listStockLotInfo(Long stockId) throws BizException;

    /**
     * 编辑预占库存<br>
     * 新增、减少、删除预占库存都支持
     * <p>
     * 场景:<br>
     * 新增sku:{@link UpdateOccupyStockParam#updateOccupyStockSubParamList}在原有的预占列表上新增,
     * 同时{@link UpdateOccupyStockParam.UpdateOccupyStockSubParam#dispatchValue}必修大于等于0<br>
     * 原有sku新增:{@link UpdateOccupyStockParam.UpdateOccupyStockSubParam#changeValue}必须大于0<br>
     * 原有sku减少:{@link UpdateOccupyStockParam.UpdateOccupyStockSubParam#changeValue}必须小于0<br>
     * 删除sku，即将预占库存全部回退到库存中心:{@link UpdateOccupyStockParam#updateOccupyStockSubParamList}
     * 在原有的预占列表上减少,如果要删除所有预占库存，传空数组即可
     * </p>
     *
     * @param updateOccupyStockParam 编辑预占库存请求参数
     * @return true/false
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>调度值和变更值不能同时为空</li>
     *                          <li>新增预占库存时，调度值不能为空或不能小于0</li>
     *                          <li>通过skuId:[{}]无法获取库存信息</li>
     *                          <li>db异常</li>
     *                          <li>修改预占库存时，变更值不能为空</li>
     *                          <li>根据库存id:[{}]无法获取库存</li>
     *                          <li>编辑预占库存请求子参数列表不能为空</li>
     *                      </ul>
     */
    Boolean updateOccupyStock(UpdateOccupyStockParam updateOccupyStockParam) throws BizException;

    /**
     * 批量消耗库存
     *
     * @param consumeStockParamList 消耗库存请求参数列表
     * @return 变更库存结果 {@link ConsumeStockResultDTO}
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>预占场景id和预占类型不能为空</li>
     *                          <li>通过预占场景id:[{}]、skuId:[{}]无法获取预占库存信息</li>
     *                          <li>消耗预占库存，通过计量id无法匹配预占库存</li>
     *                          <li>通过库存id:[{}]无法获取批次库存信息</li>
     *                          <li>db异常</li>
     *                          <li>通过skuId:[{}]无法获取库存信息</li>
     *                          <li>消耗销售库存，通过计量id无法匹配库存</li>
     *                      </ul>
     */
    List<ConsumeStockResultDTO> consumeStock(List<ConsumeStockParam> consumeStockParamList) throws BizException;

    /**
     * 批量修改库存状态
     *
     * @param updateStockStatusParamList 修改库存状态
     * @return true/false
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>批量修改库存状态不能为冻结</li>
     *                          <li>通过业务单号:[{}],业务类型:[{}]无法获取库存流水</li>
     *                          <li>db异常</li>
     *                          <li>通过库存id:[{}]无法获取库存信息</li>
     *                          <li>通过库存批次id:[{}]无法获取批次库存信息</li>
     *                      </ul>
     */
    Boolean batchUpdateStockStatus(List<UpdateStockStatusParam> updateStockStatusParamList) throws BizException;

    /**
     * 根据skuId、预占场景id批量获取预占库存信息
     *
     * @param skuIdList   skuId列表
     * @param userId      用户id
     * @param userType    用户类型(1:平台;2:租户;3:用户)
     * @param generalId   预占场景id 如活动id
     * @param generalType 预占类型 {@link StockOccupyEnum}
     * @return 预占库存信息 {@link StockOccupyInfoDTO}
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>查询预占库存请求参数有误</li>
     *                          <li>通过skuId:[{}]无法获取预占库存信息</li>
     *                          <li>库存id列表不能为空</li>
     *                          <li>bizNoList不能为空</li>
     *                      </ul>
     */
    List<StockOccupyInfoDTO> batchGetStockOccupyInfo(List<Long> skuIdList, Long userId, Integer userType, Long generalId, Integer generalType) throws BizException;

    /**
     * 分页查询预占列表库存信息
     *
     * @param stockOccupyGeneralInfoParam 请求参数 {@link StockOccupyGeneralInfoParam}
     * @return 预占列表库存信息
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>分页参数不能为空</li>
     *                          <li>库存id列表不能为空</li>
     *                          <li>bizNoList不能为空</li>
     *                      </ul>
     */
    PageResponse<StockOccupyGeneralInfoDTO> getStockOccupyGeneralInfo(StockOccupyGeneralInfoParam stockOccupyGeneralInfoParam) throws BizException;
}