package cn.com.duiba.cloud.stock.service.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.cloud.stock.service.api.dto.channel.SkuChannelDTO;
import cn.com.duiba.cloud.stock.service.api.dto.channel.SpuChannelDTO;
import cn.com.duiba.cloud.stock.service.api.dto.channel.SpuChannelUsableDTO;
import cn.com.duiba.cloud.stock.service.api.param.channel.ChannelParam;
import cn.com.duiba.cloud.stock.service.api.param.channel.SkuChannelQueryParam;
import cn.com.duiba.cloud.stock.service.api.param.channel.SpuChannelQueryParam;
import cn.com.duiba.wolf.entity.PageResponse;

import java.util.List;

/**
 * 库存渠道remote接口
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 1.0
 * @date 2021/11/20 8:45 下午
 **/
@AdvancedFeignClient
public interface RemoteStockChannelService {
    /**
     * 通过条件分页查询spu渠道列表
     *
     * @param spuChannelQueryParam 查询spu渠道列表条件
     * @return spu渠道列表
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>分页参数不能为空</li>
     *                      </ul>
     */
    PageResponse<SpuChannelDTO> listSpuChannel(SpuChannelQueryParam spuChannelQueryParam) throws BizException;

    /**
     * 通过条件分页查询sku渠道列表
     *
     * @param skuChannelQueryParam 查询sku渠道列表条件
     * @return sku渠道列表
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>分页参数有误</li>
     *                          <li>通过spuId【{}】无法获取spu渠道</li>
     *                          <li>bizNoList不能为空</li>
     *                          <li>根据库存id【{}】无法获取库存</li>
     *                      </ul>
     */
    PageResponse<SkuChannelDTO> listSkuChannel(SkuChannelQueryParam skuChannelQueryParam) throws BizException;

    /**
     * 根据spuId列表查询spu渠道是否可用
     *
     * @param spuIdList spuId列表
     * @return spu渠道是否可用列表
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>spuId列表不能为空</li>
     *                      </ul>
     */
    List<SpuChannelUsableDTO> listSpuChannelUsable(List<Long> spuIdList) throws BizException;

    /**
     * 创建渠道和库存<br>
     * 只局限于自建商品的场景
     * <ol>
     *     <li>step1 创建渠道</li>
     *     <li>step2 创建库存</li>
     * </ol>
     *
     * @param channelParam spu渠道请求参数 {@link ChannelParam}
     * @return spu渠道id
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>商品类型有误</li>
     *                          <li>db异常</li>
     *                          <li>库存参数skuId、stockType不能为空</li>
     *                          <li>批次库存请求参数列表或库存id不能为空</li>
     *                          <li>通过批次库存id:[{}]无法获取批次库存</li>
     *                          <li>新增批次库存不能为空或小于0</li>
     *                          <li>总库存和批次库存之和不相等</li>
     *                      </ul>
     */
    Long createChannel(ChannelParam channelParam) throws BizException;

    /**
     * 修改渠道和库存<br>
     * 只局限于自建商品的场景
     * <ol>
     *     <li>step1 修改渠道</li>
     *     <li>step2 修改库存</li>
     * </ol>
     *
     * @param channelParam spu渠道请求参数 {@link ChannelParam}
     * @return true/false
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>商品类型有误</li>
     *                          <li>通过spuId:[{}"]无法获取spu渠道</li>
     *                          <li>库存参数skuId、stockType不能为空</li>
     *                          <li>通过库存id:[{}]无法获取库存信息</li>
     *                          <li>批次库存请求参数列表或库存id不能为空</li>
     *                          <li>通过批次库存id:[{}]无法获取批次库存</li>
     *                          <li>db异常</li>
     *                          <li>库存变更值和批次库存变更值之和不匹配</li>
     *                          <li>新增批次库存不能为空或小于0</li>
     *                          <li>总库存和批次库存之和不相等</li>
     *                      </ul>
     */
    Boolean updateChannel(ChannelParam channelParam) throws BizException;

    /**
     * spu渠道状态开关
     *
     * @param id       spu渠道id
     * @param isUsable 是否可用 1：是，0：否
     * @return true/false
     * @throws BizException 业务异常
     *                      <ul>
     *                          <li>spu渠道id不能为空</li>
     *                          <li>isUsable参数错误</li>
     *                          <li>db异常</li>
     *                      </ul>
     */
    Boolean spuChannelStatusSwitch(Long id, Integer isUsable) throws BizException;
}