package cn.com.duiba.cloud.stock.service.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.cloud.stock.service.api.dto.occupy.MonopolizeStockDTO;
import cn.com.duiba.cloud.stock.service.api.dto.occupy.OccupyStockDTO;
import cn.com.duiba.cloud.stock.service.api.dto.stock.StockDTO;
import cn.com.duiba.cloud.stock.service.api.exception.OccupyError;
import cn.com.duiba.cloud.stock.service.api.exception.StockError;
import cn.com.duiba.cloud.stock.service.api.param.occupy.RemoteBatchFindMonopolizeOnlyValueParam;
import cn.com.duiba.cloud.stock.service.api.param.occupy.RemoteBatchFindOccupyOnlyValueParam;
import cn.com.duiba.cloud.stock.service.api.param.occupy.RemoteFindMonopolizeOnlyValueParam;
import cn.com.duiba.cloud.stock.service.api.param.occupy.RemoteFindOccupyOnlyValueParam;
import cn.com.duiba.cloud.stock.service.api.param.stock.RemoteCompleteSecKillParam;
import cn.com.duiba.cloud.stock.service.api.param.stock.RemoteSecKillParam;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import java.util.List;

/**
 * 库存高性能接口
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.1
 * @date 2022-07-13 10:06
 * @since 0.1.0
 **/
@AdvancedFeignClient
public interface RemoteStockHighPerformanceService {
    /**
     * 高性能接口，通过skuId获取可用库存（不包含独占库存）
     *
     * @param skuId skuId
     * @return 可用库存数，无库存数据为null
     */
    Long findStockNumberOnlyValue(@NotNull(message = "skuId不能为空") Long skuId);

    /**
     * 高性能接口，通过skuId列表批量获取可用库存（不包含独占库存）
     *
     * @param skuIdList skuId列表
     * @return skuId和可用库存数映射，无库存数据，不会加入列表，需要上层自己判断是否存在
     */
    List<StockDTO> batchFindStockNumberOnlyValue(@NotNull(message = "skuId列表不能为空")
                                                 @Size(min = 1, max = 100, message = "skuId列表大小必须在1-100之间") List<Long> skuIdList);

    /**
     * 高性能接口，通过appId、skuId获取独占可用库存
     *
     * @param remoteFindMonopolizeOnlyValueParam 仅查询独占库存值请求参数
     * @return 独占可用库存数
     */
    Long findMonopolizeOnlyValue(RemoteFindMonopolizeOnlyValueParam remoteFindMonopolizeOnlyValueParam);

    /**
     * 高性能接口，通过appId、skuIds批量获取独占可用库存
     *
     * @param remoteBatchFindMonopolizeOnlyValueParam 批量仅查询独占库存值请求参数
     * @return skuId和独占可用库存数映射，无库存数据，不会加入列表，需要上层自己判断是否存在
     */
    List<MonopolizeStockDTO> batchFindMonopolizeOnlyValue(
            RemoteBatchFindMonopolizeOnlyValueParam remoteBatchFindMonopolizeOnlyValueParam);

    /**
     * 高性能接口，通过generalId、generalType获取预占可用库存
     *
     * @param remoteFindOccupyOnlyValueParam 仅获取预占可用库存请求参数
     * @return 预占可用库存数，无库存数据为null
     * @throws BizException {@link StockError#PARAM_IS_ERROR}
     */
    Long findOccupyOnlyValue(RemoteFindOccupyOnlyValueParam remoteFindOccupyOnlyValueParam) throws BizException;

    /**
     * 高性能接口，通过generalId列表、generalType批量获取预占可用库存
     *
     * @param remoteBatchFindOccupyOnlyValueParam 批量仅获取预占可用库存请求参数
     * @return generalId和预占可用库存数映射，无库存数据，不会加入列表，需要上层自己判断是否存在
     * @throws BizException {@link StockError#PARAM_IS_ERROR}
     */
    List<OccupyStockDTO> batchFindOccupyOnlyValue(
            RemoteBatchFindOccupyOnlyValueParam remoteBatchFindOccupyOnlyValueParam) throws BizException;

    /**
     * 秒杀
     *
     * @param remoteSecKillParam 秒杀请求参数
     * @return 剩余库存
     * @throws BizException 计量达到下线、预占不存在
     */
    Long secKillConsume(RemoteSecKillParam remoteSecKillParam) throws BizException;

    /**
     * 标记完成秒杀
     *
     * @param remoteCompleteSecKillParam 标记完成秒杀请求参数
     * @return true/false
     * @throws BizException 业务异常 {@link StockError},{@link OccupyError}
     */
    Boolean completeSecKill(RemoteCompleteSecKillParam remoteCompleteSecKillParam) throws BizException;
}
