package cn.com.duiba.cloud.manage.service.api.utils;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.manage.service.api.model.constant.autologin.AutoLoginConstant;
import cn.com.duiba.cloud.manage.service.api.model.dto.autologin.ConsumerCookieDTO;
import cn.com.duiba.cloud.manage.service.api.remoteservice.autologin.RemoteAutoLoginService;
import cn.com.duiba.wolf.cache.RedisCacheClient;
import cn.com.duiba.wolf.utils.SecurityUtils;
import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.alibaba.fastjson.JSON;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 免登工具类
 *
 * @author huangguosheng@duiba.com.cn
 * @date 2022/3/22 3:42 下午
 **/
@Slf4j
public class AutoLoginUtil {
    private static String CACHE_COOKIE_DECRYPT_KEY = "cache_auto_login_cookie_decrypt_{timeStamp}";

    /**
     * 获取当前用户信息
     * 注意点：
     * 1、用户信息获取失败会返回null，接入方更具情况对外抛出异常
     * 2、有可能会出现窜数据情况（相同浏览器，appA留下的数据，appB来），接入方需要对appId进行判断
     *
     * @return
     * @throws BizException
     */
    public static ConsumerCookieDTO getConsumerInfo() {
        if (getIsNotLogin()) {
            log.info("用户信息获取失败，登录标识为未登录");
            return null;
        }

        String timestampStr = "";
        String encodeConsumerData = "";
        // 获取cookie中的数据
        Cookie[] cookies = CookieUtil.getCookies();
        if (cookies == null || cookies.length == 0) {
            log.info("[用户信息] 用户信息获取失败，无cookie");
            return null;
        }
        for (Cookie cookie : cookies) {
            if (AutoLoginConstant.CookieName.LOGIN_TIME_COOKIE.equals(cookie.getName())) {
                timestampStr = cookie.getValue();
            } else if (AutoLoginConstant.CookieName.LOGIN_CONSUMER_COOKIE.equals(cookie.getName())) {
                encodeConsumerData = cookie.getValue();
            }
        }

        if (StringUtils.isBlank(timestampStr) || StringUtils.isBlank(encodeConsumerData)) {
            log.info("[用户信息] 用户信息获取失败，cookie缺失");
            return null;
        }

        // 解密数据
        try {
            long timestamp = Long.parseLong(timestampStr);
            String remoteEncryptKey = getRemoteEncryptKey(timestamp);
            String dataJson = decrypt(encodeConsumerData, remoteEncryptKey);
            return JSON.parseObject(dataJson, ConsumerCookieDTO.class);
        } catch (Exception e) {
            log.warn("用户信息获取失败,可能登陆信息已过期或者假数据", e);
            return null;
        }
    }

    /**
     * 获取密钥
     *
     * @param timestamp
     * @return
     * @throws BizException
     */
    private static String getCookieEncryptKey(long timestamp) throws BizException {
        RedisCacheClient redisCacheClient = SpringUtil.getApplicationContext().getBean(RedisCacheClient.class);
        if (redisCacheClient == null) {
            return getRemoteEncryptKey(timestamp);
        }

        Date curDate = new Date(timestamp);
        DateTime truncate = DateUtil.truncate(curDate, DateField.HOUR_OF_DAY);
        String cacheKey = CACHE_COOKIE_DECRYPT_KEY.replace("{timeStamp}", "" + truncate.getTime());
        Object cacheData = redisCacheClient.get(cacheKey);
        if (cacheData != null) {
            return cacheData.toString();
        }

        String encryptKey = getRemoteEncryptKey(timestamp);
        if (StringUtils.isBlank(encryptKey)) {
            return null;
        }
        redisCacheClient.set(cacheKey, encryptKey, 1, TimeUnit.DAYS);
        return encryptKey;
    }

    private static String getRemoteEncryptKey(long timestamp) throws BizException {
        RemoteAutoLoginService remoteAutoLoginService = SpringUtil.getApplicationContext().getBean(RemoteAutoLoginService.class);
        String encryptKey = remoteAutoLoginService.getEncryptKey(timestamp);
        if (StringUtils.isBlank(encryptKey)) {
            log.info("cookie密钥密钥获取失败,返回为空 timestamp:{}", timestamp);
            throw new BizException("cookie密钥密钥获取失败");
        }
        return encryptKey;
    }

    /**
     * 获取Transfer信息
     *
     * @return
     */
    public static String getTransfer() {
        return CookieUtil.getCookieValue(AutoLoginConstant.CookieName.TRANSFER);
    }

    /**
     * 是否登录
     *
     * @return
     */
    public static boolean getIsNotLogin() {
        String cookieValue = CookieUtil.getCookieValue(AutoLoginConstant.CookieName.IS_NOT_LOGIN_USER);
        return cookieValue == null || Boolean.parseBoolean(cookieValue);
    }

    /**
     * 获取dcustom信息
     *
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String getDCustom() throws UnsupportedEncodingException {
        String dCustomValue = CookieUtil.getCookieValue(AutoLoginConstant.CookieName.DCUSTOM);
        if (StringUtils.isBlank(dCustomValue)) {
            return null;
        }
        return URLDecoder.decode(dCustomValue, "utf-8");
    }

    /**
     * 解密
     *
     * @param data
     * @param encryptKey
     * @return
     */
    public static String decrypt(String data, String encryptKey) {
        if (StringUtils.isBlank(encryptKey)) {
            throw new NullPointerException("encryptKey must not be null");
        }
        return new String(SecurityUtils.decodeByAes(SecurityUtils.decodeBase64(data), encryptKey), StandardCharsets.UTF_8);
    }

    /**
     * 加密
     *
     * @param data
     * @param encryptKey
     * @return
     */
    public static String encrypt(String data, String encryptKey) {
        if (StringUtils.isBlank(encryptKey)) {
            throw new NullPointerException("encryptKey must not be null");
        }
        return SecurityUtils.encode2StringByBase64(SecurityUtils.encodeByAes(data, encryptKey));
    }
}
