package cn.com.duiba.cloud.manage.service.sdk.util;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.cloud.manage.service.sdk.config.CookieConfig;
import cn.com.duiba.cloud.manage.service.sdk.model.dto.ParamHeaderDTO;
import cn.com.duiba.cloud.manage.service.sdk.model.dto.TokenCookieDTO;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import cn.hutool.core.codec.Base64;
import cn.hutool.core.net.URLDecoder;
import cn.hutool.core.net.URLEncodeUtil;
import cn.hutool.core.util.CharsetUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 *
 * @author zhangshun
 * @date 2019/1/31
 */
@Slf4j
public class RequestUtils {

    public static final String KEY_REQUEST_USERAGENT = "User-Agent";

    private RequestUtils() {
        //to do something
    }

    /**
     * 根据名称获取cookie
     *
     * @return cookie
     */
    public static String getCookie(HttpServletRequest request, String key) {
        return getCookies(request).get(key);
    }

    /**
     * 获取所有cookie
     *
     * @return Map<String, String>
     */
    public static Map<String, String> getCookies(HttpServletRequest request) {
        Cookie[] cookies = request.getCookies();
        if (Objects.isNull(cookies)) {
            return Collections.emptyMap();
        }
        Map<String, String> cookieMap = new HashMap<>(16);
        for (Cookie c : cookies) {
            cookieMap.put(c.getName(), c.getValue());
        }
        return cookieMap;
    }

    /**
     * 获取登录信息
     *
     * @return String
     */
    public static String getLoginCookieValue(HttpServletRequest request, String key) {
        String value = request.getHeader(key);
        if (StringUtils.isNotBlank(value)) {
            return value;
        }
        if (SpringEnvironmentUtils.isDevEnv() || SpringEnvironmentUtils.isTestEnv()) {
            value = request.getParameter(key);
            if (StringUtils.isNotBlank(value)) {
                return value;
            }
        }
        return getCookie(request, key);
    }

    /**
     * 解析用户登录数据
     *
     * @return TokenCookieDTO
     */
    public static TokenCookieDTO getTokenCookie(String tkv) {
        if (StringUtils.isBlank(tkv) || "undefined".equals(tkv) || StringUtils.equals(tkv, "null")) {
            return null;
        }
        try {
            String content = BlowfishUtils.decryptBlowfish(tkv, CookieConfig.KEY_ENCRYPT);
            return JSON.parseObject(content, TokenCookieDTO.class);
        } catch (Exception e) {
            log.error("解析Token错误 {}", tkv, e);
            return null;
        }
    }

    public static void main(String[] args) {
//        System.out.println(getTokenCookie("5GPNUyvLToZEbNDbPjtghnDufCFzUR8jZUv1XrTN2wgFXFsi5fGXYrBqAcQ7sTn9T5dEm2bxtRii8Nh6me4JWxvU621udimV3ZA9Ron2R9DAsYjUvAcS"));
//        System.out.println(getTokenCookie("5GPNUyvLToZEbNDbPjtghnDufCFzUR8jZUv1XrTN2wgFXFsi5fGXYrBqAcQ7sTn9T5dEm2bxtRii8Nh6me4JWxvU621udimV3ZA9Ron2R9DAsYjUvAcS"));

        System.out.println(JSONObject.toJSONString(Base64.decodeStr(URLDecoder.decode("eyJ0ZW5hbnRBcHBJZCI6NDEzfQ==",CharsetUtil.CHARSET_UTF_8))));
        System.out.println(JSONObject.toJSONString(Base64.decodeStr(URLDecoder.decode("eyJ0ZW5hbnRBcHBJZCI6NDAxfQ==",CharsetUtil.CHARSET_UTF_8))));
    }


    public static Map<String, Object> getParameters(HttpServletRequest request) {
        Map<String, String[]> paramMap = request.getParameterMap();
        if (MapUtils.isEmpty(paramMap)) {
            return Collections.emptyMap();
        }

        Map<String, Object> resultMap = new HashMap<>(paramMap.size());

        paramMap.forEach((k, v) -> resultMap.put(k, (v.length >= 1) ? v[0] : v));

        return resultMap;
    }

    public static Map<String, List<String>> getAllHeaders(HttpServletRequest request) {
        Enumeration<String> headerNames = request.getHeaderNames();
        if (headerNames == null || !headerNames.hasMoreElements()) {
            return Collections.emptyMap();
        }
        Map<String, List<String>> headers = new HashMap<>(16);
        while (headerNames.hasMoreElements()) {
            String name = headerNames.nextElement();
            List<String> valueList = new ArrayList<>();
            headers.put(name, valueList);
            Enumeration<String> values = request.getHeaders(name);
            if (values == null || !values.hasMoreElements()) {
                continue;
            }
            while (values.hasMoreElements()) {
                valueList.add(values.nextElement());
            }
        }
        return headers;
    }

    public static String getUserAgent(HttpServletRequest request) {
        String ua = request.getHeader(KEY_REQUEST_USERAGENT);
        if (StringUtils.isBlank(ua)) {
            return StringUtils.EMPTY;
        }
        return ua.toLowerCase();
    }

    public static String getInputSteam(HttpServletRequest request) {
        try (InputStream in = request.getInputStream()) {
            StringBuilder out = new StringBuilder();
            byte[] b = new byte[1024];
            for (int n; (n = in.read(b)) != -1; ) {
                out.append(new String(b, 0, n));
            }
            return out.toString();
        } catch (Exception e) {
            // ignore
            log.warn("获取RequestInputStream错误", e);
        }
        return StringUtils.EMPTY;
    }

    /**
     * 临时生成一个token
     * @return String
     */
    public static String generateToken(TokenCookieDTO tokenCookie) {
        String value = JSON.toJSONString(tokenCookie);
        return BlowfishUtils.encryptBlowfish(value, CookieConfig.KEY_ENCRYPT);
    }

    /**
     * 解析manage-local的参数
     * @param request
     * @return
     */
    public static ParamHeaderDTO getCookieParam(HttpServletRequest request) {
        String paramCookie = request.getHeader(CookieConfig.PARAM_HEADER);
        if (StringUtils.isBlank(paramCookie)) {
            return null;
        }
        String decodeStr = Base64.decodeStr(URLDecoder.decode(paramCookie, CharsetUtil.CHARSET_UTF_8));
        return JSON.parseObject(decodeStr, ParamHeaderDTO.class);
    }

    /**
     * 封装manage-local信息
     * @param tenantAppId
     * @return
     */
    public static String pkgCookieParam(Long tenantAppId) {
        ParamHeaderDTO paramCookie = new ParamHeaderDTO();
        paramCookie.setTenantAppId(tenantAppId);

        return URLEncodeUtil.encode(Base64.encode(JSON.toJSONString(paramCookie)));
    }
}
