package cn.com.duiba.cloud.zhongyan.goods.center.api.util;

import cn.com.duiba.cloud.zhongyan.goods.center.api.dto.GeneralBiz;
import cn.com.duiba.cloud.zhongyan.goods.center.api.dto.goods.AbstractGoodsDto;
import cn.com.duiba.cloud.zhongyan.goods.center.api.dto.goods.GoodsDetailDto;
import cn.com.duiba.cloud.zhongyan.goods.center.api.dto.goods.GoodsDto;
import cn.com.duiba.cloud.zhongyan.goods.center.api.dto.goods.SpuDto;
import cn.com.duiba.wolf.entity.Pair;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.Labels;
import com.google.common.collect.Lists;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/11/29 22:18
 */
public class GoodsSnapshotUtil {

    public static final String LABEL_SNAPSHOT = "snapshot";
    public static final String LABEL_TENANT = "tenant";
    public static final String LABEL_SUPPLIER = "supplier";
    private static final String SPLIT = "@";

    public static Pair<String, String> getSupplierSnapshotStr(GoodsDto goodsDto) {
        String s = JSON.toJSONString(goodsDto,
                                     Labels.includes(LABEL_SNAPSHOT, LABEL_SUPPLIER));
        return Pair.from(getSupplierSnapshotId(goodsDto), s);
    }

    /**
     * 供应商维度
     *
     * @param goodsDto
     *
     * @return spuId+版本号（时间）
     */
    public static String getSupplierSnapshotId(GoodsDto goodsDto) {
        return Optional.ofNullable(goodsDto).map(GoodsDto::getSpuDto).map(spuDto -> spuDto.getId() + "_" + getDateSecondStr(spuDto.getGmtModified())).orElse("");
    }

    /**
     * 获取商品快照Id
     * @param goodsDto
     * @return
     */
    public static String getSnapshotId(GoodsDto goodsDto) {
        return combineSnapshot(getSupplierSnapshotId(goodsDto), getTenantSnapshotId(goodsDto));
    }

    /**
     * 因为数据库存的是秒，所以需要转为秒
     *
     * @param date
     *
     * @return
     */
    private static String getDateSecondStr(Date date) {
        return String.valueOf(date.getTime() / 1000);
    }

    public static Pair<String, String> getTenantSnapshotStr(GoodsDto goodsDto) {
        String s = JSON.toJSONString(goodsDto,
                                     Labels.includes(LABEL_SNAPSHOT, LABEL_TENANT));
        return Pair.from(getTenantSnapshotId(goodsDto), s);
    }

    /**
     * 租户隔离
     *
     * @param goodsDto
     *
     * @return spuId+人+版本号（时间）
     */
    public static String getTenantSnapshotId(GoodsDto goodsDto) {
        return Optional.ofNullable(goodsDto)
                .map(GoodsDto::getSpuDto)
                .map(SpuDto::getDetail)
                .map(GoodsSnapshotUtil::assembleTenantSnapshotId)
                .orElse("");
    }

    private static String assembleTenantSnapshotId(GoodsDetailDto detail) {
        return String.format("%s_%s_%s_%s", detail.getGeneralId(), getDateSecondStr(detail.getGmtModified()), detail.getUserId(), detail.getUserType());
    }


    /**
     * @param goodsDto
     * @param supplierSnapshotStr
     * @param tenantSnapshotStr
     * @param skuId               是否指定sku
     */
    public static void assembleGoods(GoodsDto goodsDto,
                                     String supplierSnapshotStr,
                                     String tenantSnapshotStr,
                                     Long skuId) {
        GoodsDto supplierSnapshotDto = JSONObject.parseObject(supplierSnapshotStr, GoodsDto.class);
        GoodsDto tenantSnapshotDto = JSONObject.parseObject(tenantSnapshotStr, GoodsDto.class);

        BeanUtils.copy(supplierSnapshotDto, goodsDto);

        if (skuId != null) {
            // 过滤掉skuId

            goodsDto.setSkuList(goodsDto.getSkuList().stream()
                                        .filter(skuDto -> Objects.equals(skuDto.getId(), skuId))
                                        .collect(Collectors.toList()));
        }

        List<AbstractGoodsDto> dest = Lists.newArrayList(goodsDto.getSpuDto());
        dest.addAll(goodsDto.getSkuList());

        List<AbstractGoodsDto> source = Lists.newArrayList(tenantSnapshotDto.getSpuDto());
        source.addAll(tenantSnapshotDto.getSkuList());

        Map<String, AbstractGoodsDto> generalKeyMap = source.stream().collect(Collectors.toMap(GeneralBiz::generalKey, Function.identity()));

        for (AbstractGoodsDto dto : dest) {

            AbstractGoodsDto abstractGoodsDto = generalKeyMap.get(dto.generalKey());
            if (abstractGoodsDto != null) {
                dto.setDetail(abstractGoodsDto.getDetail());
                dto.setImageList(abstractGoodsDto.getImageList());
            }
        }
    }


    public static Pair<String, String> splitSnapshot(String snapshotId) {
        String[] split = snapshotId.split(SPLIT);
        return Pair.from(split[0], split[1]);
    }

    public static String combineSnapshot(String supplierSnapshotId, String tenantSnapshotId) {
        return supplierSnapshotId + SPLIT + tenantSnapshotId;
    }


}
