package cn.com.duiba.creditsclub.ecosphere.sdk.utils;

import cn.com.duiba.creditsclub.ecosphere.sdk.BizRuntimeException;

import java.util.List;

/**
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: QueryTable.java , v 0.1 2019-11-15 17:40 ZhouFeng Exp $
 */
public interface QueryTable<T extends Field> {

    /**
     * 创建一个元数据对象
     *
     * @return
     */
    QueryTableItem<T> newItem();

    /**
     * 添加一条数据
     *
     * @param item
     * @return
     */
    void add(QueryTableItem<T> item);

    /**
     * 通过id更新一条数据的某个字段
     * @param id
     * @param type 要更新的字段
     * @param value 更新的内容
     * @return
     */
    boolean update(long id, Field type, Object value);

    /**
     * 通过id更新一条数据的某个字段
     * @param id
     * @param item 更新的对象，所有不为null的字段均会被更新
     * @return
     */
    boolean update(long id, QueryTableItem<T> item);

    /**
     * 通过id删除一条记录
     *
     * @param id
     * @return
     */
    boolean delete(long id);

    /**
     * 自增（delta负数为自减）score字段
     *
     * @param id
     * @param delta 自增值
     * @param max   最大限制，计算后的值需要小于该值才能更新成功
     * @return
     */
    boolean increase(long id, long delta, long max);

    /**
     * 通过某个字段进行精确查询(根据插入时间倒序排列)
     * @param queryType 查询条件
     * @param value 查询值
     * @return
     */
    List<QueryTableItem<T>> query(Field queryType, Object value);

    /**
     * 通过某两个字段进行精确查询(根据插入时间倒序排列)
     * 尽量把区分度较高的字段作为第一个查询条件，将提高查询效率
     * @param type1 查询条件1
     * @param value1 查询值1
     * @param type2 查询条件2
     * @param value2 查询值2
     * @return
     */
    List<QueryTableItem<T>> query(Field type1, Object value1, Field type2, Object value2);

    /**
     * 通过某个字段进行批量查询
     *
     * @param type   查询条件
     * @param values 查询值
     * @return 未经排序的查询结果
     * @throws BizRuntimeException values长度超过200(不含)时将会抛出异常
     */
    List<QueryTableItem<T>> query(Field type, List<Object> values);

    /**
     * 通过某个字段进行精确分页查询(根据插入时间倒序排列)
     *
     * @param queryType 查询条件
     * @param value     查询值
     * @param pageNum   页码（查询第一页传入0，依次类推）
     * @param pageSize  每页数量
     * @return
     */
    PageList<QueryTableItem<T>> pageQuery(Field queryType, Object value, long pageNum, long pageSize);

    /**
     * 通过某个字段进行模糊分页查询(根据插入时间倒序排列)
     *
     * @param queryType 查询条件 仅支持EXT1,EXT2,EXT3,EXT4 四个字段的模糊查询
     * @param value     查询值 需要自行拼接%，例如前缀匹配查询key，则传入 key%
     * @param pageNum   页码（查询第一页传入0，依次类推）
     * @param pageSize  每页数量
     * @return
     */
    PageList<QueryTableItem<T>> inexplicitPageQuery(Field queryType, String value, long pageNum, long pageSize);


    /**
     * 通过某个字段进行精确分页查询(可排序)
     *
     * @param queryType     查询条件
     * @param value         查询值
     * @param sortField     排序字段
     * @param sortType      排序类型
     * @param pageNum       页码（查询第一页传入0，依次类推）
     * @param pageSize      每页数量
     * @return
     */
    PageList<QueryTableItem<T>> pageQuery(Field queryType, Object value, Field sortField, SortType sortType,
                                          long pageNum, long pageSize);

    enum SortType {
        DESC, ASC
    }

}
