package cn.com.duiba.customer.link.sdk.http;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.customer.link.sdk.SpringBeanFactory;
import cn.com.duiba.customer.link.sdk.config.InnerPathConfiguration;
import cn.com.duiba.wolf.utils.UrlUtils;
import cn.com.duiba.wolf.utils.UrlUtils2;
import com.alibaba.fastjson.JSON;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * DefaultHttpHelper默认实现类
 *
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: DefaultHttpHelper.java , v 0.1 2020-04-15 8:30 下午 ZhouFeng Exp $
 */
public class DefaultHttpHelper implements HttpHelper {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultHttpHelper.class);

    private CloseableHttpClient httpClient;
    private InnerPathConfiguration innerPathConfiguration;
    private RestTemplate businessRestTemplate;

    private final ExecutorService threadPool = Executors.newFixedThreadPool(4);

    private static final String domainSuffix = "m.duiba.com.cn";

    public ExecutorService getThreadPool() {
        return threadPool;
    }

    public DefaultHttpHelper(CloseableHttpClient httpClient) {
        this.httpClient = httpClient;
        innerPathConfiguration = SpringBeanFactory.getBean(InnerPathConfiguration.class);
        businessRestTemplate = (RestTemplate) SpringBeanFactory.getBean("businessRestTemplate");
    }

    @Override
    public String doGet(String url, Map<String, String> uriVariables) {
        HttpGet httpGet = new HttpGet(url);
        String host = httpGet.getURI().getHost();
        String domainUrl = getDomainUrl();
        if (host.endsWith(domainUrl) && businessRestTemplate != null) {
            String insideCallUrl = getInsideCallUrl(url, uriVariables, domainUrl);
            if (StringUtils.isNotBlank(insideCallUrl)) {
                try {
                    return businessRestTemplate.getForObject(insideCallUrl, String.class);
                } catch (Exception e) {
                    LOGGER.warn("loadBalance调用失败,url={},insideCallUrl={}", url, insideCallUrl, e);
                    throw new RuntimeException("loadBalance调用失败", e);
                }
            }
        }

        String fullUrl = url;
        if (MapUtils.isNotEmpty(uriVariables)) {
            fullUrl = UrlUtils.appendParams(url, uriVariables);
        }
        return execute(new HttpGet(fullUrl));
    }

    @NotNull
    private String getDomainUrl() {
        String domainUrl = domainSuffix;
        if (!SpringEnvironmentUtils.isProdEnv()) {
            domainUrl = domainUrl.replace("m.duiba.", "m.duiba" + SpringEnvironmentUtils.getCurrentEnv() + ".");
        }
        return domainUrl;
    }

    private String getInsideCallUrl(String url, Map<String, String> uriVariables, String domainUrl) {
        String insideUrl = null;

        int index = url.indexOf(domainUrl);
        if (index > -1) {
            String substring = url.substring(index);
            String[] arr = StringUtils.split(substring, "/");
            //排除调服务的根目录mapping
            if (arr.length > 2 && StringUtils.isNotBlank(arr[1])) {
                String value = innerPathConfiguration.getUrlPathMappingMap().get(arr[1]);
                if (StringUtils.isNotBlank(value)) {
                    String replace = substring.replace(domainUrl, value);
                    insideUrl = "http://" + replace;
                    if (MapUtils.isNotEmpty(uriVariables)) {
                        insideUrl = UrlUtils.appendParams(insideUrl, uriVariables);
                    }
                }
            }
        }

        return insideUrl;
    }

    @Override
    public <T> T doGet(String url, Map<String, String> uriVariables, Class<T> responseType) {
        String response = doGet(url, uriVariables);
        if (response == null) {
            return null;
        }
        return JSON.parseObject(response, responseType);
    }

    @Override
    public String doGet(String url, Map<String, String> uriVariables, Header header) {
        String fullUrl = url;
        if (MapUtils.isNotEmpty(uriVariables)) {
            fullUrl = UrlUtils.appendParams(url, uriVariables);
        }
        HttpGet httpGet = new HttpGet(fullUrl);
        httpGet.setHeader(header);
        return execute(httpGet);
    }

    @Override
    public String doPost(String url, Map<String, String> uriVariables) {
        HttpPost httpPost = new HttpPost(url);
        String host = httpPost.getURI().getHost();
        String domainUrl = getDomainUrl();
        if (host.endsWith(domainUrl) && businessRestTemplate != null) {
            String insideCallUrl = getInsideCallUrl(url, null, domainUrl);
            if (StringUtils.isNotBlank(insideCallUrl)) {
                try {
                    //注意此处是http form提交
                    MultiValueMap map = new LinkedMultiValueMap();
                    if (MapUtils.isNotEmpty(uriVariables)) {
                        for (Map.Entry<String, String> entry : uriVariables.entrySet()) {
                            map.add(entry.getKey(), entry.getValue());
                        }
                    }
                    ResponseEntity<String> responseEntity = businessRestTemplate.postForEntity(insideCallUrl, map, String.class);
                    if (responseEntity.getStatusCode() != HttpStatus.OK) {
                        throw new RuntimeException("loadBalance调用失败, code:" + responseEntity.getStatusCode());
                    }
                    return responseEntity.getBody();
                } catch (Exception e) {
                    LOGGER.warn("loadBalance调用失败,url={},insideCallUrl={}", url, insideCallUrl, e);
                    throw new RuntimeException("loadBalance调用失败", e);
                }
            }
        }

        List<NameValuePair> list = new ArrayList<>();
        if (MapUtils.isNotEmpty(uriVariables)) {
            for (Map.Entry<String, String> entry : uriVariables.entrySet()) {
                list.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
            }
        }
        UrlEncodedFormEntity formEntity = new UrlEncodedFormEntity(list, Charset.defaultCharset());
        httpPost.setEntity(formEntity);
        return execute(httpPost);
    }


    public String doPostUseTemplate(String url, Map<String, String> uriVariables) {

        try {
            MultiValueMap<String, String> map = new LinkedMultiValueMap<>();
            if (MapUtils.isNotEmpty(uriVariables)) {
                for (Map.Entry<String, String> entry : uriVariables.entrySet()) {
                    map.add(entry.getKey(), entry.getValue());
                }
            }
            ResponseEntity<String> responseEntity = businessRestTemplate.postForEntity(url, map, String.class);
            if (responseEntity.getStatusCode() != HttpStatus.OK) {
                throw new RuntimeException("loadBalance调用失败, code:" + responseEntity.getStatusCode());
            }
            return responseEntity.getBody();
        } catch (Exception e) {
            LOGGER.warn("loadBalance调用失败,url={},insideCallUrl={}", url, url, e);
            throw new RuntimeException("loadBalance调用失败", e);
        }
    }

    @Override
    public <T> T doPost(String url, Map<String, String> uriVariables, Class<T> responseType) {
        String response = doPost(url, uriVariables);
        if (response == null) {
            return null;
        }
        return JSON.parseObject(response, responseType);
    }

    @Override
    public String doPost(String url, Header header, HttpEntity httpEntity) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setHeader(header);
        httpPost.setEntity(httpEntity);
        return execute(httpPost);
    }

    @Override
    public String execute(HttpUriRequest httpUriRequest) {
        return CompletableFuture.supplyAsync(() -> {
            try (CloseableHttpResponse response = httpClient.execute(httpUriRequest)) {
                if (response.getStatusLine().getStatusCode() == 200) {
                    return EntityUtils.toString(response.getEntity(), "UTF-8");
                }
                LOGGER.warn("请求接口失败:url={}, code={}",
                        httpUriRequest.getURI().getHost() + httpUriRequest.getURI().getPath(), response.getStatusLine().getStatusCode());
                throw new RuntimeException("请求接口失败:" + response.getStatusLine().getStatusCode());
            } catch (Exception e) {
                LOGGER.warn("请求接口失败：url={}", httpUriRequest.getURI().getHost() + httpUriRequest.getURI().getPath(), e);
                throw new RuntimeException("请求接口失败", e);
            }
        }, threadPool).join();
    }

    @Override
    public String executeGet(String url, Map<String, String> uriVariables, Header[] headers) {
        String fullUrl = url;
        if (MapUtils.isNotEmpty(uriVariables)) {
            fullUrl = UrlUtils2.appendParams(url, uriVariables);
        }
        HttpGet httpGet = new HttpGet(fullUrl);
        httpGet.setHeaders(headers);
        return execute(httpGet);
    }

    @Override
    public String executePost(String url, Header[] headers, HttpEntity httpEntity) {
        HttpPost httpPost = new HttpPost(url);
        httpPost.setHeaders(headers);
        httpPost.setEntity(httpEntity);
        return execute(httpPost);
    }
}