package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.dto.saas.SaasGrantDto;
import cn.com.duiba.developer.center.api.domain.enums.saas.SaasFuncTypeEnum;
import cn.com.duiba.developer.center.api.domain.enums.saas.SaasMallTypeEnum;
import cn.com.duiba.developer.center.api.domain.paramquery.SaasGrantQueryParam;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasDevFreeService;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasGrantService;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.DateUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * Created by xiaoxuda on 2018/4/13.
 */
public class SaasGrantUtil {
    private static Logger logger = LoggerFactory.getLogger(SaasGrantUtil.class);

    private static RemoteSaasGrantService remoteSaasGrantService;
    private static RemoteSaasDevFreeService remoteSaasDevFreeService;
    private static AdvancedCacheClient advancedCacheClient;

    /**
     * 免费结束时间（该时间点之前不做流量限制）
     */
    public static final Date freeDeadline = DateUtils.getSecondDate("2018-08-01 00:00:00");
    /**
     * 开始收费时间
     */
    public static final Date chargeStartDate = DateUtils.getSecondDate("2018-04-25 00:00:00");

    /**
     * 试用模式下每日限流100次
     */
    public static final Long PROBATION_LIMIT_IN_DAY = 100L;

    /**
     * 免费类开发者名单
     */
    private static Cache<Long, Boolean> freeDevCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(1000).build();

    /**
     * 付费权限缓存
     */
    private static Cache<String, Boolean> grantCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(10000).build();

    /**
     * 是否是免费开发者
     *
     * @param developerId
     * @param degradedRunning
     * @return
     */
    public static boolean isFreeDev(Long developerId, boolean degradedRunning) {
        try {
            if(developerId == null){
                return false;
            }
            return freeDevCache.get(developerId, () -> Boolean.TRUE.equals(remoteSaasDevFreeService.isExist(developerId)));
        } catch (ExecutionException e) {
            logger.warn("开发者白名单查询发生异常, 是否降级:", degradedRunning, e);
            return degradedRunning;
        }
    }

    private static String generateKey(Long appId, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId) {
        StringBuilder builder = new StringBuilder();
        builder.append(appId).append("_").append(funcType).append("_")
                .append(funcSubType == null ? SaasGrantDto.DEFAULT_FUNC_SUB_TYPE : funcSubType).append("_")
                .append(funcRelId == null ? SaasGrantDto.DEFAULT_FUNC_REL_ID : funcRelId);
        return builder.toString();
    }

    /**
     * 权限鉴定：
     * 0.收费开始之前所有功能免费
     * 1.白名单拥有全平台功能免费权限；
     *
     * @param app
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @return
     * @throws ExecutionException
     */
    private static boolean hasGrant(AppSimpleDto app, SaasFuncTypeEnum funcType, Integer funcSubType,
                                    Long funcRelId) throws ExecutionException {
        if (app == null || funcType == null) {
            return false;
        }
        if (System.currentTimeMillis() < chargeStartDate.getTime()) {
            return true;
        }
        if (isFreeDev(app.getDeveloperId(), true)) {
            return true;
        }
        String key = generateKey(app.getId(), funcType, funcSubType, funcRelId);
        return grantCache.get(key, () -> Boolean.TRUE.equals(remoteSaasGrantService
                .hasGrant(new SaasGrantQueryParam(app.getId(), funcType, funcSubType, funcRelId))));
    }

    /**
     * 积分商城权限校验
     *
     * @param app
     * @param degradedRunning
     * @return
     */
    public static boolean hasCreditsMallGrant(AppSimpleDto app, boolean degradedRunning) {
        try {
            for(SaasMallTypeEnum typeEnum : SaasMallTypeEnum.values()) {
                if(hasGrant(app, SaasFuncTypeEnum.CREDITS_MALL, typeEnum.getValue(), null)){
                    return true;
                }
            }
            return false;
        } catch (ExecutionException e) {
            logger.warn("商城权限校验发生异常, 是否降级:", degradedRunning, e);
            return degradedRunning;
        }
    }

    /**
     * 签到权限校验
     * 1.开始收费之前入库的签到活动永久免费；
     *
     * @param app
     * @param signActId
     * @param degradedRunning
     * @param actOptTime      入库时间
     * @return
     */
    public static boolean hasSignGrant(AppSimpleDto app, Long signActId, Date actOptTime, boolean degradedRunning) {
        try {
            if(app == null || signActId == null){
                return false;
            }
            if (actOptTime != null && actOptTime.getTime() < chargeStartDate.getTime()) {
                return true;
            }
            //积分商城权限覆盖签到
            boolean signGrant = hasCreditsMallGrant(app, true);
            if (!signGrant) {
                signGrant = hasGrant(app, SaasFuncTypeEnum.SIGN, null, signActId);
            }
            return signGrant;
        } catch (ExecutionException e) {
            logger.warn("签到权限校验发生异常, 是否降级:", degradedRunning, e);
            return degradedRunning;
        }
    }

    /**
     * 活动权限校验
     * 1.开始收费之前入库的签到活动永久免费；
     * 2.拥有积分商城权限的app则拥有所有活动的使用权限
     *
     * @param app
     * @param actType
     * @param actId
     * @param degradedRunning
     * @param actOptTime      入库时间
     * @return
     */
    public static boolean hasActGrant(AppSimpleDto app, Integer actType, Long actId, Date actOptTime,
                                      boolean degradedRunning) {
        try {
            if(app == null || actType == null || actId == null){
                return false;
            }
            if (actOptTime != null && actOptTime.getTime() < chargeStartDate.getTime()) {
                return true;
            }
            boolean actGrant = hasGrant(app, SaasFuncTypeEnum.ACTIVITY, actType, actId);
            if (!actGrant) {
                actGrant = hasCreditsMallGrant(app, true);
            }
            return actGrant;
        } catch (ExecutionException e) {
            logger.warn("活动权限校验发生异常, 是否降级:", degradedRunning, e);
            return degradedRunning;
        }
    }

    /**
     * 用户访问，手机端校验app权限，没有权限则计入试用限流
     *
     * @param app
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @param actOptTime
     * @param degradedRunning
     * @return
     */
    public static boolean checkGrant(AppSimpleDto app, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId,
                                     Date actOptTime, boolean degradedRunning) {
        try {
            boolean hasGrant;
            if (SaasFuncTypeEnum.CREDITS_MALL.equals(funcType)) {
                hasGrant = hasCreditsMallGrant(app, degradedRunning);
            } else if (SaasFuncTypeEnum.SIGN.equals(funcType)) {
                hasGrant = hasSignGrant(app, funcRelId, actOptTime, degradedRunning);
            } else if (SaasFuncTypeEnum.ACTIVITY.equals(funcType)) {
                hasGrant = hasActGrant(app, funcSubType, funcRelId, actOptTime, degradedRunning);
            } else {
                hasGrant = true;
            }
            if (hasGrant) {
                return true;
            }
            if (System.currentTimeMillis() < freeDeadline.getTime()) {
                return true;
            }
            Long visitCount= advancedCacheClient.get(app.getId().toString());
            visitCount = visitCount == null ? 1 : visitCount;
            if(hasGrant = visitCount <= PROBATION_LIMIT_IN_DAY) {
                advancedCacheClient.set(app.getId().toString(), ++visitCount, DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
            }
            return hasGrant;
        }catch (Exception e){
            logger.warn("权限校验发生异常，是否降级：", degradedRunning, e);
            return degradedRunning;
        }
    }

    public static void setRemoteSaasGrantService(RemoteSaasGrantService remoteSaasGrantService) {
        SaasGrantUtil.remoteSaasGrantService = remoteSaasGrantService;
    }

    public static void setRemoteSaasDevFreeService(RemoteSaasDevFreeService remoteSaasDevFreeService) {
        SaasGrantUtil.remoteSaasDevFreeService = remoteSaasDevFreeService;
    }

    public static void setAdvancedCacheClient(AdvancedCacheClient advancedCacheClient) {
        SaasGrantUtil.advancedCacheClient = advancedCacheClient;
    }

}
