package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.dto.saas.SaasGrantDto;
import cn.com.duiba.developer.center.api.domain.dto.saas.SaasItemDto;
import cn.com.duiba.developer.center.api.domain.enums.saas.SaasFuncTypeEnum;
import cn.com.duiba.developer.center.api.domain.paramquery.SaasGrantQueryParam;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasDevFreeService;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasGrantService;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasItemSevice;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.DateUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * Created by xiaoxuda on 2018/4/13.
 */
public class SaasGrantUtil {
    private static Logger logger = LoggerFactory.getLogger(SaasGrantUtil.class);

    private static RemoteSaasGrantService remoteSaasGrantService;
    private static RemoteSaasDevFreeService remoteSaasDevFreeService;
    private static AdvancedCacheClient advancedCacheClient;
    private static RemoteSaasItemSevice remoteSaasItemSevice;

    /**
     * 试用模式下每日限流100次
     */
    public static final Long PROBATION_LIMIT_IN_DAY = 100L;

    /**
     * 免费类开发者名单
     */
    private static Cache<Long, Boolean> freeDevCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(1000).build();
    /**
     * 付费权限缓存
     */
    private static Cache<String, Boolean> grantCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(10000).build();
    /**
     * saas商品缓存
     */
    private static Cache<String, Boolean> saasItemCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(10000).build();

    /**
     * 是否是免费开发者
     *
     * @param developerId
     * @param degradedRunning
     * @return
     */
    public static boolean isFreeDev(Long developerId, boolean degradedRunning) {
        try {
            if (developerId == null) {
                return false;
            }
            return freeDevCache.get(developerId, () -> Boolean.TRUE.equals(remoteSaasDevFreeService.isExist(developerId)));
        } catch (ExecutionException e) {
            logger.warn("开发者白名单查询发生异常, 是否降级:", degradedRunning, e);
            return degradedRunning;
        }
    }

    private static String generateKey(Long appId, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId) {
        StringBuilder builder = new StringBuilder();
        builder.append(appId).append("_").append(funcType).append("_")
                .append(funcSubType == null ? SaasGrantDto.DEFAULT_FUNC_SUB_TYPE : funcSubType).append("_")
                .append(funcRelId == null ? SaasGrantDto.DEFAULT_FUNC_REL_ID : funcRelId);
        return builder.toString();
    }

    /**
     * 权限查询
     *
     * @param app
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @return
     * @throws ExecutionException
     */
    public static boolean hasGrant(AppSimpleDto app, SaasFuncTypeEnum funcType, Integer funcSubType,
                                   Long funcRelId) throws ExecutionException {
        if (app == null || funcType == null) {
            return false;
        }
        String key = generateKey(app.getId(), funcType, funcSubType, funcRelId);
        return grantCache.get(key, () -> Boolean.TRUE.equals(remoteSaasGrantService
                .hasGrant(new SaasGrantQueryParam(app.getId(), funcType, funcSubType, funcRelId))));
    }

    /**
     * 查询
     *
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @return
     */
    public static boolean isSaasItem(SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId) {
        String key = funcType.getValue() + "_" + funcSubType + "_" + funcRelId;
        try {
            return saasItemCache.get(key, () -> {
                SaasItemDto saasItem = remoteSaasItemSevice.findByUK(funcType, funcSubType, funcRelId);
                return saasItem != null && Boolean.TRUE.equals(saasItem.getOpenStatus());
            });
        } catch (ExecutionException e) {
            logger.warn("Saas商品查询发生异常, key:", key, e);
            return false;
        }
    }

    @Deprecated
    public static boolean checkGrant(AppSimpleDto app, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId,
                                     Date actOptTime, boolean degradedRunning) {
        return checkGrant(app, funcType, funcSubType, funcRelId, degradedRunning);
    }

    /**
     * 用户访问，手机端校验app权限，没有权限则计入试用限流
     * 0.KA开发者拥有全平台免费特权
     * 1.app必须具有商城权限
     * 2.活动、签到若已配置为saas收费商品且商品在有效期则需要权限鉴定
     * 3.验证活动、签到是否已购买且权限有效
     *
     * @param app
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @param degradedRunning
     * @return
     */
    public static boolean checkGrant(AppSimpleDto app, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId,
                                     boolean degradedRunning) {
        if (isFreeDev(app.getDeveloperId(), true)) {
            return true;
        }
        if (!isSaasItem(funcType, funcSubType, funcRelId)) {
            return true;
        }
        try {
            boolean hasGrant = hasGrant(app, funcType, funcSubType, funcRelId);
            if (hasGrant) {
                return true;
            }
            Long visitCount = advancedCacheClient.get(RedisKeyFactory.K005 + app.getId().toString());
            visitCount = visitCount == null ? 1 : visitCount;
            if (hasGrant = (visitCount <= PROBATION_LIMIT_IN_DAY)) {
                advancedCacheClient.set(RedisKeyFactory.K005 + app.getId().toString(), ++visitCount, DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
            }
            return hasGrant;
        } catch (Exception e) {
            logger.warn("权限校验发生异常，是否降级：", degradedRunning, e);
            return degradedRunning;
        }
    }

    public static void setRemoteSaasGrantService(RemoteSaasGrantService remoteSaasGrantService) {
        SaasGrantUtil.remoteSaasGrantService = remoteSaasGrantService;
    }

    public static void setRemoteSaasDevFreeService(RemoteSaasDevFreeService remoteSaasDevFreeService) {
        SaasGrantUtil.remoteSaasDevFreeService = remoteSaasDevFreeService;
    }

    public static void setAdvancedCacheClient(AdvancedCacheClient advancedCacheClient) {
        SaasGrantUtil.advancedCacheClient = advancedCacheClient;
    }

    public static void setRemoteSaasItemSevice(RemoteSaasItemSevice remoteSaasItemSevice) {
        SaasGrantUtil.remoteSaasItemSevice = remoteSaasItemSevice;
    }
}
