package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.developer.center.api.domain.enums.VipLimitTypeEnum;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;

/**
 * Created by xiaoxuda on 2016/11/15.
 * 提供按照用户vip等级做功能过滤的能力,用户调用 RemoteVipLimitService 接口获取
 */
public class VipLimitFilter implements Serializable {
    private static final long serialVersionUID = 1L;

    /**
     * app维度vip等级限制开关
     **/
    private boolean isFilterOpen;

    /**
     * 默认无参构造函数
     **/
    public VipLimitFilter() {
        isFilterOpen = false;
    }

    /**
     * @param isFilterOpen app的vip限制功能打开标记
     */
    public VipLimitFilter(boolean isFilterOpen) {
        this.isFilterOpen = isFilterOpen;
    }

    /**
     * 等级限制过滤，使用传入规则的方式，降低用户接入的代码耦合
     * @param collection
     * @param getter 信息读取规则
     * @param vipLevel 需要验证的vip等级
     * @param <T>
     */
    public <T> void vipLimitFilter(Collection<T> collection, VipLimitInfoGetter<T> getter, Integer vipLevel){
        //未开启vip过滤功能，不做处理
        if (!this.isFilterOpen) {
            return;
        }

        Iterator<T> iterator = collection.iterator();
        while(iterator.hasNext()) {
            if (!canPassVipLimit(iterator.next(),getter, vipLevel)) {
                iterator.remove();
            }
        }
    }

    /**
     * 等级限制判断，使用传入规则的方式，降低用户接入的代码耦合
     * @param t
     * @param getter
     * @param <T>
     * @param vipLevel 需要验证的vip等级
     * @return
     */
    public <T> boolean canPassVipLimit(T t, VipLimitInfoGetter<T> getter, Integer vipLevel){
        //未开启商品维度vip限制开关，直接返回true
        if (VipLimitTypeEnum.OFF_LEVEL.getCode().equals(getter.getVipLimitType(t))) {
            return true;
        }

        //取不到限制等级直接返回true
        if (getter.getVipLimits(t) == null) {
            return true;
        }

        //根据等级限制类型做判断
        long vipLevelLong = vipLevel.longValue();
        if (VipLimitTypeEnum.ABOVE_LEVEL.getCode().equals(getter.getVipLimitType(t))) {
            return vipLevelLong >= getter.getVipLimits(t);
        }
        if (VipLimitTypeEnum.ACCURATE_LEVEL.getCode().equals(getter.getVipLimitType(t))) {
            return isCanDo(getter.getVipLimits(t), vipLevelLong);
        }
        // 其他为异常情况
        return false;
    }

    //当为精确设置某个等级可兑时，判断某个等级是否可以兑换
    private boolean isCanDo(long vipLimits, long vipLevel) {
        long v = 1L << vipLevel;
        long ret = vipLimits & v;
        return ret != 0;
    }

    public boolean isFilterOpen() {
        return isFilterOpen;
    }

}
