package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.developer.center.api.domain.dto.AppNewExtraDto;
import cn.com.duiba.developer.center.api.domain.dto.saas.SaasGrantDto;
import cn.com.duiba.developer.center.api.domain.dto.saas.SaasItemDto;
import cn.com.duiba.developer.center.api.domain.enums.AccessAccountStatusEnum;
import cn.com.duiba.developer.center.api.domain.enums.saas.SaasFuncTypeEnum;
import cn.com.duiba.developer.center.api.domain.paramquery.SaasGrantQueryParam;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppNewExtraService;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasDevFreeService;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasGrantService;
import cn.com.duiba.developer.center.api.remoteservice.saas.RemoteSaasItemSevice;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import cn.com.duiba.wolf.utils.DateUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import com.google.common.base.Optional;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.commons.lang.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * Created by xiaoxuda on 2018/4/13.
 */
public class SaasGrantUtil {
    private static Logger logger = LoggerFactory.getLogger(SaasGrantUtil.class);

    private static RemoteSaasGrantService remoteSaasGrantService;
    private static RemoteSaasDevFreeService remoteSaasDevFreeService;
    private static AdvancedCacheClient advancedCacheClient;
    private static RemoteSaasItemSevice remoteSaasItemSevice;
    private static RemoteAppNewExtraService remoteAppNewExtraService;

    /**
     * 试用模式下每日限流100次
     */
    public static final Long PROBATION_LIMIT_IN_DAY = 100L;

    /**
     * 免费类开发者名单
     */
    private final static Cache<Long, Boolean> canFreeUseCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(1000).build();

    /**
     * 接入账号体系名单
     */
    private final static Cache<Long, Boolean> accessAccountCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(1000).build();
    /**
     * 付费权限缓存
     */
    private final static Cache<String, Optional<SaasGrantDto>> grantCache = CacheBuilder.newBuilder().expireAfterWrite(5,TimeUnit.MINUTES)
            .maximumSize(1000).build();

    /**
     * saas商品缓存
     */
    private final static Cache<String, Boolean> saasItemCache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(1000).build();

    /**
     * 是否可免费使用
     *
     * @param appId 必传
     * @param developerId
     * @param degradedRunning true：调用异常时支持降级通过
     * @return
     */
    public static boolean canFreeUse(Long developerId, Long appId, boolean degradedRunning) {
        try {
            if(appId == null){
                return false;
            }
            return canFreeUseCache.get(appId,
                    () -> Boolean.TRUE.equals(remoteSaasDevFreeService.canFreeUse(developerId, appId)));
        } catch (Exception e) {
            logger.warn("免费名单查询发生异常, 是否降级:{},developerId={}, appId={}, errmsg={}", degradedRunning, developerId, appId, e.getMessage());
            return degradedRunning;
        }
    }

    /**
     *
     * 判断app是否接入账号体系
     * @param appId 必传
     * @param degradedRunning true：调用异常时支持降级通过
     * @return
     */
    public static boolean canAccessAccount(Long appId, boolean degradedRunning) {
        try {
            if(appId == null){
                return false;
            }
            return accessAccountCache.get(appId,
                    () -> {
                        AppNewExtraDto appNewExtraDto = remoteAppNewExtraService.findByAppIdNew(appId);
                        return appNewExtraDto!=null && StringUtils.equals(appNewExtraDto.getAccountSystem(), AccessAccountStatusEnum.STATUS_OPEN.getCode());
                    });
        } catch (Exception e) {
            logger.warn("接入账号体系名单查询发生异常, 是否降级:{}, appId={},errmsg={}", degradedRunning, appId,e.getMessage());
            return degradedRunning;
        }
    }

    private static String generateKey(Long appId, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId) {
        StringBuilder builder = new StringBuilder();
        builder.append(appId).append("_").append(funcType).append("_")
                .append(funcSubType == null ? SaasGrantDto.DEFAULT_FUNC_SUB_TYPE : funcSubType).append("_")
                .append(funcRelId == null ? SaasGrantDto.DEFAULT_FUNC_REL_ID : funcRelId);
        return builder.toString();
    }

    /**
     * 权限查询
     *
     * @param appId
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @return
     * @throws ExecutionException
     */
    public static SaasGrantDto getGrant(Long appId, SaasFuncTypeEnum funcType, Integer funcSubType,
                                   Long funcRelId) throws ExecutionException {
        if (appId == null || funcType == null) {
            return null;
        }
        String key = generateKey(appId, funcType, funcSubType, funcRelId);
        return grantCache.get(key, () -> Optional.fromNullable(remoteSaasGrantService
                .findGrant(new SaasGrantQueryParam(appId, funcType, funcSubType, funcRelId)))).orNull();
    }

    /**
     * 查询
     *
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @return
     */
    public static boolean isSaasItem(SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId) {
        String key = funcType.getValue() + "_" + funcSubType + "_" + funcRelId;
        try {
            return saasItemCache.get(key, () -> {
                SaasItemDto saasItem = remoteSaasItemSevice.findByUK(funcType, funcSubType, funcRelId);
                return saasItem != null && Boolean.TRUE.equals(saasItem.getOpenStatus());
            });
        } catch (ExecutionException e) {
            logger.warn("Saas商品查询发生异常, key:{},errmsg={}", key, e.getMessage());
            return false;
        }
    }

    /**
     * 用户访问，手机端校验app权限，没有权限则计入试用限流
     * 0.KA开发者拥有全平台免费特权
     * 1.app必须具有商城权限
     * 2.活动、签到若已配置为saas收费商品且商品在有效期则需要权限鉴定
     * 3.验证活动、签到是否已购买且权限有效
     *
     * @param developerId
     * @param appId
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @param degradedRunning
     * @return
     */
    public static boolean checkGrant(Long developerId, Long appId, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId,
                                     boolean degradedRunning) {
        if(appId == null){
            return false;
        }
        if (canFreeUse(developerId, appId, true)) {
            return true;
        }
        if(funcType != null && SaasFuncTypeEnum.CREDITS_MALL.getValue() == funcType.getValue() && remoteSaasDevFreeService.accessNewVersionAuthority(appId)){
            return true;
        }
        if (!isSaasItem(funcType, funcSubType, funcRelId)) {
            return true;
        }
        try {
            SaasGrantDto saasGrant = getGrant(appId, funcType, funcSubType, funcRelId);
            if (saasGrant != null && (saasGrant.getDeadline() == null
                    || DateUtils.daysBetween(new Date(), saasGrant.getDeadline()) >= 0)) {
                return true;
            }
            Long visitCount = advancedCacheClient.get(RedisKeyFactory.K005 + appId.toString());
            visitCount = visitCount == null ? 1 : visitCount;
            if (visitCount <= PROBATION_LIMIT_IN_DAY) {
                advancedCacheClient.set(RedisKeyFactory.K005 + appId.toString(), ++visitCount, DateUtils.getToTomorrowSeconds(), TimeUnit.SECONDS);
                return true;
            }
            return false;
        } catch (Exception e) {
            logger.warn("权限校验发生异常，是否降级：{}, appId={},errmsg={}", degradedRunning, appId, e.getMessage());
            return degradedRunning;
        }
    }

    /**
     * 使用场景：需要用户支付（积分 or 钱）时的权限校验（不允许试用）
     * 用户参与，手机端校验app权限，不允许使用
     * 0.KA开发者拥有全平台免费特权
     * 1.app必须具有商城权限
     * 2.活动、签到若已配置为saas收费商品且商品在有效期则需要权限鉴定
     * 3.验证活动、签到是否已购买且权限有效
     *
     * @param developerId
     * @param appId
     * @param funcType
     * @param funcSubType
     * @param funcRelId
     * @param degradedRunning
     * @return
     */
    public static boolean checkGrantNoTry(Long developerId, Long appId, SaasFuncTypeEnum funcType, Integer funcSubType, Long funcRelId,
                                           boolean degradedRunning) {
        if (appId == null) {
            return false;
        }
        if (canFreeUse(developerId, appId, true)) {
            return true;
        }
        if (!isSaasItem(funcType, funcSubType, funcRelId)) {
            return true;
        }
        try {
            SaasGrantDto saasGrant = getGrant(appId, funcType, funcSubType, funcRelId);
            if (saasGrant != null && (saasGrant.getDeadline() == null
                    || DateUtils.daysBetween(new Date(), saasGrant.getDeadline()) >= 0)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            logger.warn("权限校验发生异常，是否降级：{}, appId={},errmsg={}", degradedRunning, appId, e.getMessage());
            return degradedRunning;
        }
    }

    public static void setRemoteSaasGrantService(RemoteSaasGrantService remoteSaasGrantService) {
        SaasGrantUtil.remoteSaasGrantService = remoteSaasGrantService;
    }

    public static void setRemoteSaasDevFreeService(RemoteSaasDevFreeService remoteSaasDevFreeService) {
        SaasGrantUtil.remoteSaasDevFreeService = remoteSaasDevFreeService;
    }

    public static void setAdvancedCacheClient(AdvancedCacheClient advancedCacheClient) {
        SaasGrantUtil.advancedCacheClient = advancedCacheClient;
    }

    public static void setRemoteSaasItemSevice(RemoteSaasItemSevice remoteSaasItemSevice) {
        SaasGrantUtil.remoteSaasItemSevice = remoteSaasItemSevice;
    }

    public static RemoteAppNewExtraService getRemoteAppNewExtraService() {
        return remoteAppNewExtraService;
    }

    public static void setRemoteAppNewExtraService(RemoteAppNewExtraService remoteAppNewExtraService) {
        SaasGrantUtil.remoteAppNewExtraService = remoteAppNewExtraService;
    }
}
