package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.developer.center.api.domain.dto.authority.BusinessWhiteListContentDto;
import cn.com.duiba.developer.center.api.domain.dto.authority.BusinessWhiteListDto;
import cn.com.duiba.developer.center.api.domain.enums.authority.AccessStatusType;
import cn.com.duiba.developer.center.api.domain.enums.authority.BusinessWhiteListType;
import cn.com.duiba.developer.center.api.remoteservice.authority.RemoteBusinessWhiteListContentService;
import cn.com.duiba.developer.center.api.remoteservice.authority.RemoteBusinessWhiteListService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by liukai on 2020/12/27.
 * 该工具主要用户封装常用app 应用白名单的操作
 */
public class WhiteAccessUtil {

    private static Logger log = LoggerFactory.getLogger(WhiteAccessUtil.class);

    private static Logger logger = LoggerFactory.getLogger(WhiteAccessUtil.class);

    private static RemoteBusinessWhiteListContentService remoteBusinessWhiteListContentService;

    private static RemoteBusinessWhiteListService remoteBusinessWhiteListService;

    private static StringRedisTemplate stringRedisTemplate;

    private static Double EXITST = 1.0;

    private static Double NOT_EXITST = 0.0;

    private static String PREFIX = "WHITE_LIST_";

    private static String PREFIX_SINGLE = "WHITE_LIST_SINGLE";

    private static String PREFIX_JSON = "WHITE_LIST_JSON";


    private static final Cache<QueryParam, Boolean> CAN_ACCESS_WHITE_LIST = Caffeine.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).maximumSize(1000)
            .build();

    private static final Cache<QueryParam, String> WHITE_LIST_JSON_VALUE = Caffeine.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).maximumSize(1000)
            .build();

    private static final Cache<QueryParam, List<String>> WHITE_LIST_VALUES = Caffeine.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).maximumSize(1000)
            .build();


    /**
     * 在某个业务里面 获取对应配置的json信息
     * @param uniqueCode
     * @return
     */
    public static String whiteListJson(String uniqueCode) {
        return WHITE_LIST_JSON_VALUE.get(new QueryParam(uniqueCode),queryParam->{
            String uniqueCodeKey = PREFIX_JSON + queryParam.getUniqueCode();
            try {
                String jsonValue = stringRedisTemplate.opsForValue().get(uniqueCodeKey);
                if (jsonValue == null) {
                    BusinessWhiteListDto whiteListConfig = remoteBusinessWhiteListService.getByCode(queryParam.getUniqueCode());
                    if(!BusinessWhiteListType.JSON_VALUE.getCode().equals(whiteListConfig.getBizType())){
                        log.warn("WHITE_LIST_JSON_VALUE 白名单json类型错误，请核对 queryParam = {}",JSON.toJSONString(queryParam));
                    }
                    BusinessWhiteListContentDto dto = remoteBusinessWhiteListContentService.getByRelId(whiteListConfig.getId());
                    if(dto!=null){
                        stringRedisTemplate.opsForValue().set(uniqueCodeKey, dto.getBizContent());
                        stringRedisTemplate.expire(uniqueCode,12,TimeUnit.HOURS);
                    }
                    return null;
                }
                return jsonValue;
            } catch (Exception e) {
                logger.warn("白名单查询失败。走默认降级策略 false equryParam = {}", JSON.toJSONString(queryParam), e);
                return null;
            }
        });
    }


    /**
     *  判断这个appid 是否在这个业务名单里面
     * @param appId
     * @param uniqueCode
     * @return json
     */
    public static Boolean canAccessWhiteList(Long appId, String uniqueCode) {
        return CAN_ACCESS_WHITE_LIST.get(new QueryParam(appId,uniqueCode),queryParam->{
            String uniqueCodeKey = PREFIX_SINGLE + queryParam.getUniqueCode();
            //1 先查redis 判断是否存在，无论是否在白名单内，都会有值 除非过期了。需要到mysql 查询
            try {
                Double score = stringRedisTemplate.opsForZSet().score(uniqueCodeKey, queryParam.getAppId().toString());
                if (score == null) {
                    BusinessWhiteListDto whiteListConfig = remoteBusinessWhiteListService.getByCode(queryParam.getUniqueCode());
                    if (whiteListConfig == null || AccessStatusType.CLOSE.getCode().equals(whiteListConfig.getOpenStatus())) {
                        if(BusinessWhiteListType.JSON_VALUE.getCode().equals(whiteListConfig.getBizType())){
                            log.warn("CAN_ACCESS_WHITE_LIST 白名单类型查询错误，请核对 queryParam = {}",JSON.toJSONString(queryParam));
                        }
                        stringRedisTemplate.opsForHash().put(uniqueCodeKey, queryParam.getAppId(), NOT_EXITST);
                        return false;
                    }
                    Boolean wasExist = remoteBusinessWhiteListContentService.getByRelIdAndValue(whiteListConfig.getId(), appId) != null;
                    stringRedisTemplate.opsForHash().put(uniqueCodeKey, queryParam.getAppId().toString(), (wasExist ? EXITST : NOT_EXITST));
                    stringRedisTemplate.expire(uniqueCodeKey,12,TimeUnit.HOURS);
                    return wasExist;
                }
                return EXITST.equals(score);
            } catch (Exception e) {
                logger.warn("白名单查询失败。走默认降级策略 false equryParam = {}", JSON.toJSONString(queryParam), e);
                return false;
            }
        });
    }


    /**
     * 在某个业务里面  所有名单内容
     * @param uniqueCode
     * @return json
     */
    public static List<String> fetchWhiteListValues(String uniqueCode) {
        return WHITE_LIST_VALUES.get(new QueryParam(uniqueCode),queryParam->{
            String uniqueCodeKey = PREFIX + queryParam.getUniqueCode();
            //1 先查redis 判断是否存在，无论是否在白名单内，都会有值 除非过期了。需要到mysql 查询
            try {
                String jsonValue = stringRedisTemplate.opsForValue().get(uniqueCodeKey);
                if (jsonValue == null) {
                    List<String> values = Lists.newArrayList();
                    BusinessWhiteListDto whiteListConfig = remoteBusinessWhiteListService.getByCode(queryParam.getUniqueCode());
                    if(BusinessWhiteListType.JSON_VALUE.getCode().equals(whiteListConfig.getBizType())){
                        log.warn("WHITE_LIST_JSON_VALUE 白名单json类型错误，请核对 queryParam = {}",JSON.toJSONString(queryParam));
                    }
                    List<BusinessWhiteListContentDto> dtos = remoteBusinessWhiteListContentService.getByRid(whiteListConfig.getId());
                    if(dtos!=null){
                        values.addAll(dtos.stream().map(a->a.getRelValue()).collect(Collectors.toList()));
                        stringRedisTemplate.opsForValue().set(uniqueCodeKey, JSON.toJSONString(values));
                        stringRedisTemplate.expire(uniqueCode,12,TimeUnit.HOURS);
                    }
                    return values;
                }
                return JSONArray.parseArray(jsonValue,String.class);
            } catch (Exception e) {
                logger.warn("白名单查询失败。走默认降级策略 false equryParam = {}", JSON.toJSONString(queryParam), e);
                return Lists.newArrayList();
            }
        });
    }


    private WhiteAccessUtil() {
        // empty
    }


    public static StringRedisTemplate getStringRedisTemplate() {
        return stringRedisTemplate;
    }

    public static void setStringRedisTemplate(StringRedisTemplate stringRedisTemplate) {
        WhiteAccessUtil.stringRedisTemplate = stringRedisTemplate;
    }

    public static RemoteBusinessWhiteListContentService getRemoteBusinessWhiteListContentService() {
        return remoteBusinessWhiteListContentService;
    }

    public static void setRemoteBusinessWhiteListContentService(RemoteBusinessWhiteListContentService remoteBusinessWhiteListContentService) {
        WhiteAccessUtil.remoteBusinessWhiteListContentService = remoteBusinessWhiteListContentService;
    }

    public static RemoteBusinessWhiteListService getRemoteBusinessWhiteListService() {
        return remoteBusinessWhiteListService;
    }

    public static void setRemoteBusinessWhiteListService(RemoteBusinessWhiteListService remoteBusinessWhiteListService) {
        WhiteAccessUtil.remoteBusinessWhiteListService = remoteBusinessWhiteListService;
    }

    public static class QueryParam {

        /**
         * 需要判断的appid
         */
        private Long appId;

        /**
         * 判断appId 是否在uniqueCode 所在的白名单内
         */
        private String uniqueCode;

        public Long getAppId() {
            return appId;
        }

        public void setAppId(Long appId) {
            this.appId = appId;
        }

        public String getUniqueCode() {
            return uniqueCode;
        }

        public void setUniqueCode(String uniqueCode) {
            this.uniqueCode = uniqueCode;
        }

        public QueryParam(Long appId, String uniqueCode) {
            this.appId = appId;
            this.uniqueCode = uniqueCode;
        }

        public QueryParam( String uniqueCode) {
            this.uniqueCode = uniqueCode;
        }
    }


}
