package cn.com.duiba.developer.center.biz.bo.impl;


import java.util.*;

import javax.annotation.PostConstruct;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;

import cn.com.duiba.developer.center.api.domain.dto.*;
import cn.com.duiba.developer.center.api.domain.paramquery.PageQueryEntity;
import cn.com.duiba.developer.center.api.domain.vo.PaginationVO;
import cn.com.duiba.developer.center.biz.bo.AppLayoutBo;
import cn.com.duiba.developer.center.biz.dao.statistics.OdpsAppLayoutStatDao;
import cn.com.duiba.developer.center.biz.dataobject.credits.AppDO;
import cn.com.duiba.developer.center.biz.dataobject.credits.AppLayoutBrickDO;
import cn.com.duiba.developer.center.biz.dataobject.statistics.AppLayoutCountDo;
import cn.com.duiba.developer.center.biz.entity.AppLayoutEntity;
import cn.com.duiba.developer.center.biz.event.AppCreateEvent;
import cn.com.duiba.developer.center.biz.event.AppUpdateEvent;
import cn.com.duiba.developer.center.biz.event.FloorSkinUpdateEvent;
import cn.com.duiba.developer.center.biz.service.credits.AppService;
import cn.com.duiba.developer.center.biz.service.credits.floor.AppLayoutBrickService;
import cn.com.duiba.developer.center.biz.service.credits.floor.AppLayoutService;
import cn.com.duiba.developer.center.biz.service.credits.floor.CreditsFloorSkinService;
import cn.com.duiba.developer.center.biz.service.credits.floor.CreditsFloorSkinSpecifyService;
import cn.com.duiba.developer.center.biz.service.credits.floor.jsonconfig.BaseJson;
import cn.com.duiba.developer.center.biz.service.credits.floor.jsonconfig.FloorSortJson;
import cn.com.duiba.developer.center.common.constants.CacheConstants;
import cn.com.duiba.developer.center.common.constants.DsConstants;
import cn.com.duiba.developer.center.common.support.BizEventBus;
import cn.com.duiba.developer.center.common.tools.DateUtil;
import cn.com.duiba.developer.center.common.tools.ValidatorTool;
import cn.com.duiba.service.exception.BusinessException;
import cn.com.duiba.wolf.cache.CacheClient;
import cn.com.duiba.wolf.utils.BeanUtils;


/**
 * Created by liuyao on 16/7/28.
 */
@Service
public class AppLayoutBoImpl implements AppLayoutBo {
    private static final Logger log= LoggerFactory.getLogger(AppLayoutBoImpl.class);
    @Autowired
    private AppLayoutBrickService appLayoutBrickService;
    @Autowired
    private AppLayoutService appLayoutService;
    @Autowired
    private OdpsAppLayoutStatDao odpsAppLayoutStatDao;
    @Autowired
    private BizEventBus eventBus;
    @Autowired
    private AppService appService;
    @Autowired
    private CreditsFloorSkinService creditsFloorSkinService;
    @Autowired
    private CreditsFloorSkinSpecifyService creditsFloorSkinSpecifyService;
    @Autowired
    private CacheClient memcachedClient;

    @Value("${dcm.theme.defaultId}")
    private String defaultThemeId;

    private Ordering<AppLayoutBrickDO> ordering = Ordering.natural().onResultOf(new Function<AppLayoutBrickDO,Integer>(){
        @Override
        public Integer apply(AppLayoutBrickDO arg0) {
            return arg0.getPayload();
        }
    });

    @PostConstruct
    public void init(){
        eventBus.register(this);
    }

    @Override
    public PaginationVO<AppLayoutBrickDto> getSystemThemeLayoutPage(PageQueryEntity params) {
        PaginationVO<AppLayoutBrickDto> page = new PaginationVO<AppLayoutBrickDto>();

        List<AppLayoutBrickDO> list = appLayoutBrickService.getSystemThemeLayoutPageList(params);
        list = ordering.sortedCopy(list);

        AppLayoutTransform transformTool = new AppLayoutTransform(Optional.of(list));
        transformTool.setNeedSetAppCount(true);

        Integer count = appLayoutBrickService.getSystemThemeLayoutPageCount(params);
        page.setRows(transformTool.transform());

        page.setTotalCount(count.longValue());

        return page;
    }

    @Override
    public PaginationVO<AppLayoutBrickDto> getCustomThemeLayoutPage(PageQueryEntity params) {
        PaginationVO<AppLayoutBrickDto> page = new PaginationVO<AppLayoutBrickDto>();
        List<AppLayoutBrickDO> list = appLayoutBrickService.getCustomThemeLayoutList(params);
        list = ordering.sortedCopy(list);

        AppLayoutTransform transformTool = new AppLayoutTransform(Optional.of(list));

        Integer count = appLayoutBrickService.getCustomThemeLayoutCount(params);
        page.setRows(transformTool.transform());
        page.setTotalCount(count.longValue());
        return page;
    }

    /**
     *当app的创建时,同时创建AppLayoutDO
     */
    @Subscribe
    public void createAppListener(AppCreateEvent event) {
        AppDO app = event.getApp();

        AppLayoutEntity appLayoutDO = new AppLayoutEntity(true);
        appLayoutDO.setAppId(app.getId());
        JSONObject json = new JSONObject();
        json.put("0",new Long(defaultThemeId));
        appLayoutDO.setBrickIds(json.toJSONString());
        appLayoutDO.setThemeColor("#11c3bc");
        appLayoutService.insert(appLayoutDO);
    }


    /**
     *当楼层皮肤发生变更时,要清除所有的使用该皮肤的appLayout缓存
     */
    @Subscribe
    public void floorSkinUpdateListener(FloorSkinUpdateEvent event) {
        Collection<Long> appIds = appLayoutService.useSkinAppIds(event.getSkinId());
        for(Long appId : appIds){
            removeCacheByAppId(appId);
        }
    }

    @Override
    @Transactional(DsConstants.DATABASE_CREDITS)
    public void updateThemeColor(Long appId, String color) throws BusinessException {
        AppSimpleDto app = appService.getObject(appId);
        if(Objects.equal(null,app)){
            throw new BusinessException("App不存在");
        }
        try{
            appLayoutService.updateThemeColor(appId,color);
            //兼容老逻辑,同时更新app中的color
            appService.updateColor(appId,color);

            removeCacheByAppId(appId);
            AppUpdateEvent event = new AppUpdateEvent(appId);
            eventBus.post(event);
        }catch(Exception e){
            log.error("修改皮肤主题色失败",e);
            throw new BusinessException("修改皮肤主题色失败");
        }
    }

    @Override
    public void devSetSkin(Long appId, Long skinId) throws BusinessException {

        AppLayoutDto applayout = appLayoutService.buildAppLayoutByAppId(appId);
        if(Objects.equal(null,applayout)) throw new BusinessException("此AppId不合法");
        CreditsFloorSkinDto skin = creditsFloorSkinService.selectFloorSkinById(skinId);
        if(Objects.equal(null,skin)) throw new BusinessException("设置的皮肤不存在不合法");

        //判断是否定向
        if(skin.getSpecify()){
            List<Long> specifyappIds = creditsFloorSkinSpecifyService.selectSpecifyAppIdsBySkinId(skinId);
            if(!specifyappIds.contains(appId)) throw new BusinessException("此皮肤无法使用");
        }

        //生成楼层排序序列
        JSONArray array = JSONArray.parseArray(skin.getSkinList());
        List<FloorSortJson> floorList = Lists.newArrayList();
        for(int i= 0;i<array.size();i++){
            JSONObject json = array.getJSONObject(i);
            FloorSortJson it = new FloorSortJson();
            it.setId(json.getLong("id"));
            it.setFloorType(json.getInteger("type"));
            it.setEnable(true);//初始化楼层全部为显示状态
            floorList.add(it);
        }
        appLayoutService.updateSkin(appId,skinId,floorList);
        removeCacheByAppId(appId);
    }

    @Override
    public void devReSetFloorList(Long appId, JSONArray floorSortList) throws BusinessException {

        AppLayoutDto applayout = getObject(appId);
        if(applayout==null) throw new BusinessException("此AppId不合法");
        if(applayout.getSkinType()!=AppLayoutDto.Skin_Type_Floor_Skin) throw new BusinessException("老皮肤状态无法执行此操作");
        JSONArray array = applayout.getDevelopSortJson();
        if(array.size()!=floorSortList.size()){
            throw new BusinessException("楼层配置个数与皮肤不一致");
        }

        Map<Long,Integer> map = Maps.newHashMap();
        for(int i=0;i<array.size();i++){
            //[{id:楼层Id,floorType:楼层类型,enable:是否展示}]
            JSONObject json = array.getJSONObject(i);
            map.put(json.getLong("id"),json.getInteger("floorType"));
        }
        //与皮肤json必须一致
        for(int i=0;i<floorSortList.size();i++){
            JSONObject json = floorSortList.getJSONObject(i);
            if(!map.containsKey(json.getLong("id"))) throw new BusinessException("提交的楼层配置与皮肤不一致");
            int type = map.get(json.getLong("id"));
            if(!Objects.equal(type, json.getInteger("floorType"))) throw new BusinessException("提交的楼层配置与皮肤不一致");
        }
        appLayoutService.updateDevSortJsonByAppId(appId,floorSortList);
        removeCacheByAppId(appId);
    }

    @Override
    public Integer updateDevFloorConfigByAppId(Long appId, Long floorId, Map<String, Object> floorConfig) throws BusinessException {
        try{
            AppLayoutDto appLayoutDto = appLayoutService.buildAppLayoutByAppId(appId);
            if (appLayoutDto == null) {
                throw new BusinessException("对应App不存在");
            }
            if (Objects.equal(AppLayoutDto.Skin_Type_Old_Theme, appLayoutDto.getSkinType())) {
                throw new BusinessException("App处于老皮肤状态,无法更新楼层配置");
            }
            CreditsFloorSkinDto skin = creditsFloorSkinService.selectFloorSkinById(appLayoutDto.getSkinId());
            Map<Long,Integer> typeMap = skin.getTypeMap();
            if (!typeMap.containsKey(floorId)) throw new BusinessException("App没有该楼层配置, appId=" + appId + ", floorId=" + floorId);

            Integer floorType = typeMap.get(floorId);

            Class<? extends BaseJson> clazz = appLayoutService.getBaseJson(floorType);
            BaseJson jsonConfig = clazz.newInstance();
            org.apache.commons.beanutils.BeanUtils.populate(jsonConfig,floorConfig);//BaseJson中的属性过滤json中多余的键值对,缺失的以默认值代替
            ValidatorTool.valid(jsonConfig);
            String json = JSONObject.toJSONString(jsonConfig);

            Map<Long,JSONObject> jsonMap = transformJsonConfig(appLayoutDto.getDevelopDataShowJson(), typeMap);
            jsonMap.put(floorId,JSONObject.parseObject(json));

            int ret = appLayoutService.updateDevFloorConfigByAppId(appId,jsonMap);
            removeCacheByAppId(appId);
            return ret;
        }catch (BusinessException e) {
            throw e;
        }catch (Exception e){
            log.error("更新楼层配置失败",e);
            return 0;
        }
    }

    @Override
    public AppLayoutDto getObject(Long appId){//这里需要缓存
        AppLayoutDto appLayoutDto = memcachedClient.get(getLayoutKey(appId));
        if(!Objects.equal(null,appLayoutDto)) return appLayoutDto;

        appLayoutDto = appLayoutService.buildAppLayoutByAppId(appId);
        if(Objects.equal(null,appLayoutDto)) return null;
        switch (appLayoutDto.getSkinType()){
            case AppLayoutDto.Skin_Type_Old_Theme:
                AppLayoutBrickDO brick = appLayoutBrickService.findCacheAppLayoutBrick(appLayoutDto.getBrickId());
                appLayoutDto.setBrickId(brick.getId());
                appLayoutDto.setBrickMd5(brick.getMd5());
                appLayoutDto.setImageStyle(brick.getImageStyle());
                appLayoutDto.setDevelopDataShowJson(Collections.<Long, JSONObject>emptyMap());//防止预览时报空指针异常
                break;
            case AppLayoutDto.Skin_Type_Floor_Skin:
                CreditsFloorSkinDto skin = creditsFloorSkinService.selectFloorSkinById(appLayoutDto.getSkinId());
                JSONArray skinArray = JSONArray.parseArray(skin.getSkinList());
                Map<Long,JSONObject> skinMap = Maps.newLinkedHashMap();
                for(int i = 0;i<skinArray.size();i++){
                    JSONObject json = skinArray.getJSONObject(i);
                    skinMap.put(json.getLong("id"),json);
                }
                appLayoutDto.setDevelopDataShowJson(transformJsonConfig(appLayoutDto.getDevelopDataShowJson(),skin.getTypeMap()));

                //去除皮肤中没有的项
                JSONArray jsonArray = new JSONArray();
                Set<Long> devFloorIds = Sets.newHashSet();
                JSONArray developSortJson = appLayoutDto.getDevelopSortJson();

                Map<Long,JSONObject> jsonMap = appLayoutDto.getDevelopDataShowJson();
                for(int i = 0;i<developSortJson.size();i++){
                    JSONObject json = developSortJson.getJSONObject(i);
                    Long floorId = json.getLong("id");
                    devFloorIds.add(floorId);
                    if(!skinMap.containsKey(floorId)) continue;//如果该楼层被管理员删除,则去除
                    if(json.getInteger("floorType")== CreditsFloorCodeDto.CREDITS_FLOOR_TYPE_PAGE){
                        json.put("pageId",skinMap.get(floorId).getLong("pageId"));
                        JSONObject config = jsonMap.get(floorId);
                        if(!config.containsKey("floorTitle")|| StringUtils.isBlank(config.getString("floorTitle"))){
                            config.put("floorTitle",skinMap.get(floorId).getString("name"));
                        }
                    }
                    jsonArray.add(json);
                }
                //把皮肤中新加的项加到最后,并且enable为不展示状态
                Set<Long> addIds = Sets.difference(skinMap.keySet(),devFloorIds);
                for(Long floorId:addIds){
                    JSONObject json = skinMap.get(floorId);
                    FloorSortJson it = new FloorSortJson();
                    it.setId(floorId);
                    it.setFloorType(json.getInteger("type"));
                    it.setEnable(false);
                    jsonArray.add(JSONObject.toJSON(it));
                }
                appLayoutDto.setDevelopSortJson(jsonArray);

        }
        memcachedClient.set(getLayoutKey(appId),appLayoutDto,60);
        return appLayoutDto;
    }


    class AppLayoutTransform{
        private boolean needSetAppCount = false;
        private List<AppLayoutBrickDO> list;

        private Map<Long,Long> idAndCount = new HashMap<Long,Long>();

        public AppLayoutTransform(Optional<List<AppLayoutBrickDO>> list){
            this.list = list.get();
        }

        private List<AppLayoutBrickDto> transform(){
            if(list.isEmpty()){
                Collections.emptyList();
            }

            List<AppLayoutBrickDto> newlist = Lists.transform(list, new Function<AppLayoutBrickDO, AppLayoutBrickDto>() {

                private Long isNewDate = new Date().getTime()-3*24*60*60*1000L;

                @Override
                public AppLayoutBrickDto apply(AppLayoutBrickDO input) {
                    AppLayoutBrickDto dto = new AppLayoutBrickDto();

                    BeanUtils.copy(input,dto);
                    if(needSetAppCount){
                        dto.setAppCount(idAndCount.get(input.getId()));
                    }
                    dto.setNew(input.getGmtCreate().getTime()>isNewDate);

                    return dto;
                }
            });
            return newlist;
        }

        public void setNeedSetAppCount(boolean needSetAppCount) {
            this.needSetAppCount = needSetAppCount;
            if(!needSetAppCount)return;

            Set<Long> idSet = Sets.newHashSet();
            for(AppLayoutBrickDO brickDO : this.list){
                idSet.add(brickDO.getId());
            }
            List<Long> ids = Lists.newArrayList(idSet);
            Collection<AppLayoutCountDo> countList = odpsAppLayoutStatDao.getLayoutCount(ids, DateUtil.getDayStr(DateUtil.daysAddOrSub(new Date(), -1)));
            for(AppLayoutCountDo brick:countList){
                this.idAndCount.put(brick.getId(), brick.getAppCount());
            }

        }
    }

    /**
     * 转化app楼层配置
     * @param typeMap
     * @return
     */
    private Map<Long,JSONObject> transformJsonConfig(Map<Long,JSONObject> appFloorConfig, Map<Long, Integer> typeMap){

        Map<Long,JSONObject> jsonMap = Maps.newHashMap(appFloorConfig);

        Set<Long> floorIds = Sets.newHashSet(jsonMap.keySet());

        //如果config里面没有的配置,即为添加的新楼层,这里为其补充配置
        for(Long floorId:Sets.difference(typeMap.keySet(),floorIds)){
            jsonMap.put(floorId,new JSONObject());
        }
        //如果app配置里面有,皮肤里面没有,则为无效楼层
        for(Long floorId:Sets.difference(floorIds,typeMap.keySet())){
            jsonMap.remove(floorId);
        }

        for(Long floorId:jsonMap.keySet()){
            JSONObject config = jsonMap.get(floorId);
            if(typeMap.containsKey(floorId)){
                try{
                    Integer type = typeMap.get(floorId);
                    BaseJson bean = appLayoutService.getBaseJson(type).newInstance();
                    org.apache.commons.beanutils.BeanUtils.populate(bean,config);
                    config = JSONObject.parseObject(JSONObject.toJSONString(bean));
                }catch(Exception e){
                    log.error("json转换失败",e);
                }
            }
            jsonMap.put(floorId,config);
        }
        return jsonMap;
    }


    private String getLayoutKey(Long appId){
        StringBuilder sb = new StringBuilder();
        sb.append(CacheConstants.KEY_APP_LAYOUT_BY_APPID).append(appId);
        return sb.toString();
    }

    private void removeCacheByAppId(Long appId){
        memcachedClient.remove(getLayoutKey(appId));
        memcachedClient.remove("chaos.keyAppLayoutByAppId_"+appId);

    }

}
