package cn.com.duiba.developer.center.biz.service.credits.aerosol;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import cn.com.duiba.developer.center.api.domain.vo.PaginationVO;
import cn.com.duiba.developer.center.common.constants.CacheConstants;
import cn.com.duiba.wolf.cache.CacheClient;
import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.developer.center.api.domain.paramquery.AppAerosolParams;
import cn.com.duiba.developer.center.biz.dao.center_config.AerosolDao;
import cn.com.duiba.developer.center.biz.entity.AerosolEntity;
import cn.com.duiba.developer.center.biz.entity.AerosolGroupEntity;
import cn.com.duiba.developer.center.biz.service.credits.aerosol.pojo.AerosolCount;

/**
 * Created by liuyao on 16/9/8.
 */
@Service
public class AerosolService {
    @Autowired
    private AerosolDao aerosolDao;
    @Autowired
    private CacheClient memcachedClient;

    private String getkeyByAppId(Long appId){
        StringBuilder sb = new StringBuilder();
        sb.append(CacheConstants.KEY_APP_AEROSOL_BY_APP_ID);
        sb.append(appId);
        return sb.toString();
    }

    private void removeCache(Long appId){
        String key = getkeyByAppId(appId);
        memcachedClient.remove(key);
    }

    private void removeCache(List<Long> appIds){
        for(Long appId:appIds){
            removeCache(appId);
        }
    }

    public AerosolEntity findByAppId(Long appId) {
        String key = getkeyByAppId(appId);
        AerosolEntity entity = memcachedClient.get(key);
        if(entity==null){
            entity = aerosolDao.findByAppId(appId);
            if(entity==null){
                return null;
            }else{
                memcachedClient.set(key,entity,60);
            }
        }
        return entity;
    }

    public AerosolEntity findNoCacheByAppId(Long appId) {
        return aerosolDao.findByAppId(appId);
    }

    public List<AerosolEntity> findByAppIds(List<Long> appIds){
        if(appIds.isEmpty()){
            return Collections.emptyList();
        }
        return aerosolDao.findByAppIds(appIds);
    }

    public List<AerosolEntity> findByIds(List<Long> ids){
        if(ids.isEmpty()){
            return Collections.emptyList();
        }
        return  aerosolDao.findByIds(ids);
    }

    public int batchInsert(List<AerosolEntity> list){
        if(list.isEmpty()){
            return 0;
        }
        return aerosolDao.batchInsert(list);
    }

    public int insertAerosol(AerosolEntity entity){
        List<AerosolEntity> list = Lists.newArrayList(entity);
        return batchInsert(list);
    }

    /**
     * 将游离的浮标加到组中
     * @param appId
     * @param groupId
     * @param isShow
     * @return
     */
    public int comeToAerosol(Long appId,Long groupId,Boolean isShow){
        List<Long> appIds = Lists.newArrayList(appId);
        return batchComeToAerosol(appIds,groupId,isShow);
    }

    /**
     * 批量将游离的浮标加到组中
     * @param appIds
     * @param groupId
     * @param isShow
     * @return
     */
    public int batchComeToAerosol(List<Long> appIds,Long groupId,Boolean isShow){
        if(appIds.isEmpty()){
            return 0;
        }
        int ret = aerosolDao.batchComeToAerosol(appIds,groupId,isShow);
        removeCache(appIds);
        return ret;
    }

    /**
     * 加载浮标组下所有的App浮标
     * @param groupId
     * @return
     */
    public List<AerosolEntity> findAerosolByGroupId(Long groupId){
        return aerosolDao.findAerosolByGroupId(groupId);
    }

    public int batchUpdateAerosol(List<Long> appIds,AerosolEntity aerosol){
        if(StringUtils.isBlank(aerosol.getMd5()) || StringUtils.isBlank(aerosol.getConfigUrl())){
            throw new RuntimeException("浮标的MD5和配置链接不能为空内容不能为空");
        }
        if(appIds.isEmpty()){
            return 0;
        }
        int ret = aerosolDao.batchUpdateAerosol(appIds,aerosol);
        removeCache(appIds);
        return ret;
    }

    /**
     * 更新单个浮标的内容配置
     * @param aerosol 浮标配置
     * @return
     */
    public int updateAerosol(AerosolEntity aerosol){
        if(Objects.equals(null,aerosol.getAppId())){
            throw new RuntimeException("更新浮标配置缺失AppId");
        }
        List<Long> appIds = Lists.newArrayList(aerosol.getAppId());
        return batchUpdateAerosol(appIds,aerosol);
    }

    public int deleteAerosolByAppIds(List<Long> appIds){
        if(appIds.isEmpty()){
            return 0;
        }
        int ret = aerosolDao.deleteAerosolByAppIds(appIds);
        removeCache(appIds);
        return ret;
    }

    public int deleteAerosol(Long appId){
        List<Long> appIds = Lists.newArrayList(appId);
        return deleteAerosolByAppIds(appIds);
    }

    public List<AerosolCount> findGroupCounts(List<Long> groupIds){
        if(groupIds.isEmpty()){
            return Collections.emptyList();
        }
        return aerosolDao.findGroupCounts(groupIds);
    }

    public PaginationVO<AerosolEntity> findAerosolPage(AppAerosolParams params){
        List<AerosolEntity> list = aerosolDao.findAerosolPageList(params);
        Long count = aerosolDao.findAerosolPageCount(params);
        PaginationVO<AerosolEntity> page = new PaginationVO<AerosolEntity>();
        page.setRows(list);
        page.setTotalCount(count);
        return page;
    }

    /**
     * 修改浮标组下所有浮标的显示状态
     * @param groupId
     * @param isShow
     * @return
     */
    public int updateAerosolShowByGroupStatus(Long groupId,Boolean isShow){
        List<AerosolEntity> list = findAerosolByGroupId(groupId);
        if(list.isEmpty()){
            return 0;
        }
        List<Long> appIds = Lists.transform(list, new Function<AerosolEntity, Long>() {
            @Override
            public Long apply(AerosolEntity aerosolEntity) {
                return aerosolEntity.getAppId();
            }
        });
        int ret = aerosolDao.updateAerosolShowByAppIds(appIds,isShow);
        removeCache(appIds);
        return ret;
    }

    /**
     * 修改单个浮标的显示状态
     * @param appId
     * @param isShow
     * @return
     */
    public int updateAerosolShowByAppId(Long appId,Boolean isShow){
        List<Long> appIds = Lists.newArrayList(appId);
        int ret = aerosolDao.updateAerosolShowByAppIds(appIds,isShow);
        removeCache(appIds);
        return ret;
    }

    /**
     * 当浮标组被删除时,把浮标组下的所有浮标变成游离态
     * @param appIds
     * @return
     */
    public int batchUpdateGroupIdToNullByAppIds(List<Long> appIds){
        if(appIds.isEmpty()){
            return 0;
        }
        int ret = aerosolDao.batchUpdateGroupIdToNullByAppIds(appIds);
        removeCache(appIds);
        return ret;
    }

    public int updateGroupIdToNullByAppId(Long appId){
        List<Long> appIds = Lists.newArrayList(appId);
        return batchUpdateGroupIdToNullByAppIds(appIds);
    }
}
