package cn.com.duiba.developer.center.biz.bo;

import cn.com.duiba.developer.center.api.domain.dto.AerosolDto;
import cn.com.duiba.developer.center.api.domain.dto.AerosolGroupDto;
import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.paramquery.AppAerosolParams;
import cn.com.duiba.developer.center.api.domain.vo.PaginationVO;
import cn.com.duiba.developer.center.biz.entity.AerosolEntity;
import cn.com.duiba.developer.center.biz.entity.AerosolGroupEntity;
import cn.com.duiba.developer.center.biz.service.credits.aerosol.AerosolGroupService;
import cn.com.duiba.developer.center.biz.service.credits.AppService;

import cn.com.duiba.developer.center.biz.service.credits.aerosol.AerosolService;

import cn.com.duiba.developer.center.biz.service.credits.aerosol.pojo.AerosolCount;
import cn.com.duiba.developer.center.common.constants.DsConstants;
import cn.com.duiba.service.exception.BusinessException;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;


/**
 * Created by liuyao on 16/9/8.
 */
@Service
public class AerosolBo{
    private static final Logger logger= LoggerFactory.getLogger(AerosolBo.class);
    @Autowired
    private AerosolService aerosolService;
    @Autowired
    private AerosolGroupService aerosolGroupService;
    @Autowired
    private AppService appService;

    public AerosolDto getAerosolDto(Long appId) {
        AerosolDto dto = new AerosolDto();
        AerosolEntity entity=aerosolService.findByAppId(appId);
        if(entity==null){
            return null;
        }
        dto.setId(entity.getId());
        dto.setShow(entity.getIsShow());
        dto.setAppId(entity.getAppId());
        dto.setMd5(entity.getMd5());
        dto.setConfigUrl(entity.getConfigUrl());
        dto.setGroupId(entity.getGroupId());
        dto.setAppName(entity.getAppName());
        return dto;
    }

    public AerosolDto loadAerosolNoCache(Long appId) {
        AerosolDto dto = new AerosolDto();
        AerosolEntity entity=aerosolService.findNoCacheByAppId(appId);
        if(entity==null){
            return null;
        }
        dto.setId(entity.getId());
        dto.setShow(entity.getIsShow());
        dto.setAppId(entity.getAppId());
        dto.setMd5(entity.getMd5());
        dto.setConfigUrl(entity.getConfigUrl());
        dto.setGroupId(entity.getGroupId());
        dto.setAppName(entity.getAppName());
        return dto;
    }

    public List<AerosolGroupDto> loadAerosolGroup(){
        List<AerosolGroupDto> aerosolGroupDtos=new ArrayList<>();
        List<AerosolGroupEntity> list=aerosolGroupService.findAerosolGroupList();

        List<Long> groupIds = Lists.newArrayList();
        for(AerosolGroupEntity entity:list){
            groupIds.add(entity.getId());
            AerosolGroupDto aerosolGroupDto=new AerosolGroupDto();
            aerosolGroupDto.setId(entity.getId());
            aerosolGroupDto.setGroupName(entity.getGroupName());
            aerosolGroupDto.setGmtModified(entity.getGmtModified());
            aerosolGroupDtos.add(aerosolGroupDto);
        }
        List<AerosolCount> counts = aerosolService.findGroupCounts(groupIds);
        Map<Long,Integer> countMap = Maps.newHashMap();
        for(AerosolCount it:counts){
            countMap.put(it.getGroupId(),it.getTotle());
        }
        for(AerosolGroupDto dto:aerosolGroupDtos){
            if(countMap.containsKey(dto.getId())){
                dto.setCount(countMap.get(dto.getId()));
            }else{
                dto.setCount(0);
            }
        }
        return aerosolGroupDtos;
    }

    @Transactional(DsConstants.DATABASE_CREDITS)
    public Boolean updateAerosolGroupShowStatus(Long groupId) throws BusinessException{
        AerosolGroupEntity group = aerosolGroupService.findGroupById(groupId);
        if(group==null){
            throw new BusinessException("不存在此groupId");
        }
        boolean isShow = !group.getIsShow();
        aerosolGroupService.updateAerosolGroupShowStatus(groupId,isShow);
        aerosolService.updateAerosolShowByGroupStatus(groupId, isShow);
        return isShow;

    }

    public Boolean updateAerosolShowStatus(Long appId) throws BusinessException{
        AerosolEntity entity = aerosolService.findNoCacheByAppId(appId);
        if(entity==null){
            throw new BusinessException("该App当前没有配置浮标");
        }
        if(entity.getGroupId()!=null){
            throw new BusinessException("浮标非游离态");
        }
        Boolean isShow = !entity.getIsShow();
        aerosolService.updateAerosolShowByAppId(appId,isShow);
        return isShow;
    }

    public Long createAerosolGroup(AerosolGroupDto groupDto)throws BusinessException{
        try{
            return aerosolGroupService.createAerosolGroup(groupDto);
        }catch (Exception e){
            logger.error("创建浮标分组失败",e);
            throw new BusinessException("创建浮标分组失败");
        }
    }

    @Transactional(DsConstants.DATABASE_CREDITS)
    public void deleteAerosolGroup(Long id) throws BusinessException{
        try{
            aerosolGroupService.deleteAerosolGroup(id);
            List<AerosolEntity> aerosols = aerosolService.findAerosolByGroupId(id);
            Set<Long> appIds = Sets.newHashSet();
            for(AerosolEntity it:aerosols){
                appIds.add(it.getAppId());
            }
            aerosolService.batchUpdateGroupIdToNullByAppIds(Lists.newArrayList(appIds));
        }catch (Exception e){
            logger.error("删除浮标分组失败",e);
            throw new BusinessException("删除浮标分组失败");
        }
    }

    public void addToGroup(Long groupId, Long appId)throws BusinessException{
        AerosolGroupEntity group = aerosolGroupService.findGroupById(groupId);
        if(group==null){
            throw new BusinessException("浮标组不存在");
        }
        AerosolEntity aerosol = aerosolService.findNoCacheByAppId(appId);
        if(aerosol==null){
            throw new BusinessException("浮标不存在");
        }
        if(aerosol.getGroupId()!=null){
            throw new BusinessException("浮标已经归属在组中");
        }
        aerosolService.comeToAerosol(appId,groupId,group.getIsShow());
    }

    public void batchAddToGroup(Long groupId, Set<Long> appIds) throws BusinessException{
        AerosolGroupEntity group = aerosolGroupService.findGroupById(groupId);
        if(group==null){
            throw new BusinessException("浮标组不存在");
        }
        List<AerosolEntity> aerosols = aerosolService.findByAppIds(Lists.newArrayList(appIds));
        Set<Long> hasAerosols = Sets.newHashSet();
        for(AerosolEntity it:aerosols){
            hasAerosols.add(it.getAppId());
        }
        //已经存在的部分更新其归属
        aerosolService.batchComeToAerosol(Lists.newArrayList(hasAerosols),groupId,group.getIsShow());

        Set<Long> notHas = Sets.difference(appIds,hasAerosols);
        Map<Long,String> appNameMap = Maps.newHashMap();
        List<AppSimpleDto> apps = appService.findByAppIds(Lists.newArrayList(notHas));
        for(AppSimpleDto app:apps){
            appNameMap.put(app.getId(),app.getName());
        }

        List<AerosolEntity> inserts = Lists.newArrayList();
        for(Long appId:notHas){
            AerosolEntity insert = new AerosolEntity();
            insert.setAppId(appId);
            insert.setAppName(appNameMap.get(appId));
            insert.setMd5(group.getMd5());
            insert.setConfigUrl(group.getConfigUrl());
            insert.setGroupId(group.getId());
            insert.setIsShow(group.getIsShow());
            inserts.add(insert);
        }
        //不存在的部分新增浮标
        aerosolService.batchInsert(inserts);
    }

    public void delForGroup(Long appId)throws BusinessException{
        int ret = aerosolService.updateGroupIdToNullByAppId(appId);
        if(ret!=1){
            throw new BusinessException("退组失败");
        }
    }

    public void synAerosolGroup(Long groupId,Set<Long> igenoreIds)throws BusinessException{
        AerosolGroupEntity group = aerosolGroupService.findGroupById(groupId);

        List<AerosolEntity> aerosols = aerosolService.findAerosolByGroupId(groupId);

        List<Long> appIds = Lists.newArrayList();

        for(AerosolEntity it:aerosols){
            if(!igenoreIds.contains(it.getAppId())){
                appIds.add(it.getAppId());
            }
        }
        AerosolEntity update = new AerosolEntity();
        update.setMd5(group.getMd5());
        update.setConfigUrl(group.getConfigUrl());
        aerosolService.batchUpdateAerosol(appIds,update);
    }

    public void delAerosolByGroupId(Long groupId,Set<Long> igenoreIds)throws BusinessException{

        List<AerosolEntity> aerosols = aerosolService.findAerosolByGroupId(groupId);

        List<Long> appIds = Lists.newArrayList();

        for(AerosolEntity it:aerosols){
            if(!igenoreIds.contains(it.getAppId())){
                appIds.add(it.getAppId());
            }
        }
        aerosolService.deleteAerosolByAppIds(appIds);
    }

    public void delAerosolByAppId(Long appId)throws BusinessException{
        aerosolService.deleteAerosol(appId);
    }

    public void createAerosol(AerosolDto config) throws BusinessException{
        AppSimpleDto app=appService.getObject(config.getAppId());
        if(app==null){
            throw new BusinessException("app不存在");
        }
        AerosolEntity entity=aerosolService.findNoCacheByAppId(config.getAppId());
        if(entity!=null){
            throw new BusinessException("app已经存在浮标配置");
        }
        AerosolEntity insert=new AerosolEntity();
        insert.setAppId(app.getId());
        insert.setAppName(app.getName());
        insert.setGroupId(null);
        insert.setMd5(config.getMd5());
        insert.setConfigUrl(config.getConfigUrl());
        insert.setIsShow(false);
        aerosolService.insertAerosol(insert);
    }

    public void updateAerosol(AerosolDto config)throws BusinessException{
        AppSimpleDto app=appService.getObject(config.getAppId());
        if(app==null){
            throw new BusinessException("app不存在");
        }
        AerosolEntity entity=aerosolService.findNoCacheByAppId(config.getAppId());
        if(entity==null){
            throw new BusinessException("该app的浮标不存在");
        }
        AerosolEntity update=new AerosolEntity();
        update.setAppId(app.getId());
        update.setMd5(config.getMd5());
        update.setConfigUrl(config.getConfigUrl());
        aerosolService.updateAerosol(update);
    }

    public AerosolGroupDto getAerosolGroupById(Long groupId){
        AerosolGroupEntity group = aerosolGroupService.findGroupById(groupId);
        if(group==null){
            return null;
        }
        AerosolGroupDto dto = new AerosolGroupDto();
        dto.setId(group.getId());
        dto.setGroupName(group.getGroupName());
        dto.setMd5(group.getMd5());
        dto.setConfigUrl(group.getConfigUrl());
        dto.setShow(group.getIsShow());
        return dto;
    }
   
    public Boolean updateGroupConfig(AerosolGroupDto aerosolGroupDto) throws BusinessException{
        try{
            aerosolGroupService.updateAerosolGroup(aerosolGroupDto);
            return true;
        }catch (Exception e){
            logger.error("更新浮标组配置失败",e);
            throw new BusinessException("更新浮标组配置失败");
        }
    }

    public List<AerosolDto> loadGroupAppList(Long groupId){
        List<AerosolEntity> entityList= aerosolService.findAerosolByGroupId(groupId);
        return transform(entityList);
    }

    private List<AerosolDto> transform(List<AerosolEntity> entityList){
        if(entityList.isEmpty()){
            return Collections.emptyList();
        }
        Map<Long,String> groupName = Maps.newHashMap();
        List<AerosolGroupEntity> groupEntityList = aerosolGroupService.findAerosolGroupList();
        for(AerosolGroupEntity it:groupEntityList){
            groupName.put(it.getId(),it.getGroupName());
        }

        List<AerosolDto> resultList=new ArrayList<>();

        for(AerosolEntity it:entityList){
            AerosolDto aerosolDto=new AerosolDto();
            try{
                BeanUtils.copyProperties(aerosolDto,it);
                if(Objects.equal(it.getGroupId(),null)){
                    aerosolDto.setGroupName("游离态");
                    aerosolDto.setGroupId(null);
                }else{
                    aerosolDto.setGroupName(groupName.get(it.getGroupId()));
                }
            }catch (Exception e){
                logger.error("浮标对象转化异常",e);
                continue;
            }
            resultList.add(aerosolDto);
        }
        return resultList;
    }

    public PaginationVO<AerosolDto> loadAerosolPage(AppAerosolParams params){
        PaginationVO<AerosolDto> page=new PaginationVO<AerosolDto>();

        if(params.getAppId()!=null){
            AerosolEntity aerosolEntity = aerosolService.findNoCacheByAppId(params.getAppId());
            if(aerosolEntity==null){
                return page;
            }
            List<AerosolEntity> list = Lists.newArrayList();
            list.add(aerosolEntity);
            page.setRows(transform(list));
            page.setTotalCount(1L);
            return page;
        }

        if(StringUtils.isNotBlank(params.getGroupName())) {
            List<Long> groupIds = aerosolGroupService.findGroupIdByGroupNameLike(params.getGroupName());
            params.setGroupIds(groupIds);
        }

        PaginationVO<AerosolEntity> entityPaginationVO = aerosolService.findAerosolPage(params);

        page.setRows(transform(entityPaginationVO.getRows()));
        page.setTotalCount(entityPaginationVO.getTotalCount());
        return page;
    }

    public List<AerosolDto> loadAerosolByAppIds(Set<Long> appIds){
        List<AerosolEntity> list = aerosolService.findByAppIds(Lists.newArrayList(appIds));
        List<AerosolDto> resultList = Lists.newArrayList();
        resultList.addAll(transform(list));
        if(appIds.size()==list.size()) {
            return resultList;
        }
        Set<Long> hasSet = Sets.newHashSet();
        for(AerosolDto it:resultList){
            hasSet.add(it.getAppId());
        }
        List<AppSimpleDto> appList= appService.findByAppIds(Lists.newArrayList(Sets.difference(appIds,hasSet)));

        for(AppSimpleDto app:appList){
            AerosolDto aerosolDto=new AerosolDto();
            aerosolDto.setAppId(app.getId());
            aerosolDto.setAppName(app.getName());
            aerosolDto.setGroupName("未创建");
            resultList.add(aerosolDto);
        }
        return resultList;
    }


}
