package cn.com.duiba.developer.center.biz.bo;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.google.common.base.Objects;

import cn.com.duiba.developer.center.api.domain.dto.DeveloperDto;
import cn.com.duiba.developer.center.api.domain.paramquery.CreateDeveloperParams;
import cn.com.duiba.developer.center.biz.entity.DeveloperEntity;
import cn.com.duiba.developer.center.biz.event.DeveloperCreateEvent;
import cn.com.duiba.developer.center.biz.service.credits.DeveloperService;
import cn.com.duiba.developer.center.common.constants.DsConstants;
import cn.com.duiba.developer.center.common.support.BizEventBus;
import cn.com.duiba.developer.center.common.tools.MessageDigestUtils;
import cn.com.duiba.developer.center.common.tools.SwitchTool;
import cn.com.duiba.service.exception.BusinessException;
import cn.com.duiba.wolf.utils.BlowfishUtils;

/**
 * Created by liuyao on 2016/10/8.
 */
@Service
public class DeveloperBo {
    private static final Logger logger= LoggerFactory.getLogger(AppBulletinBo.class);
    @Autowired
    private DeveloperService developerService;

    private static final BeanCopier DEVELOPER_COPY = BeanCopier.create(DeveloperEntity.class,DeveloperDto.class,false);

    @Value("${dcm.login.encrypt.key}")
    private String loginEncryptKey;
    @Autowired
    private BizEventBus eventBus;

    /**
     * 创建开发者
     * @param params
     * @param needVerify
     * @return
     * @throws BusinessException
     */
    @Transactional(value = DsConstants.DATABASE_CREDITS)
    public Long createDeveloper(CreateDeveloperParams params, Boolean needVerify) throws BusinessException {
        DeveloperEntity developer = developerService.findByEmail(params.getEmail());
        if(!Objects.equal(developer,null)){
            throw new BusinessException("此邮箱已经注册过了");
        }

        DeveloperEntity insert = new DeveloperEntity(true);
        if(needVerify!=null && !needVerify){//不需要校验
            insert.setVerify(true);
        }
        insert.setEmail(params.getEmail());
        insert.setName(params.getName());
        insert.setPhone(params.getPhone());
        insert.setCompany(params.getCompany());

        String password = BlowfishUtils.encryptBlowfish(MessageDigestUtils.SHA(params.getPassword()), loginEncryptKey);
        insert.setPassword(password);

        developerService.insertDeveloper(insert);

        developer = developerService.findByEmail(params.getEmail());

        SwitchTool tool = SwitchTool.getSwitchTool(developer.getId(),developer.getDeveloperSwitch());
        tool.setSwitchByBoundType(DeveloperDto.SWITCH_POWERED_BY,true);
        developerService.updateDeveloperSwitch(tool);

        DeveloperCreateEvent event = new DeveloperCreateEvent();
        event.setAppName(params.getAppName());
        event.setDeveloper(developer);
        eventBus.post(event);

        return developer.getId();
    }

    /**
     * 根据邮箱账号的获取开发者
     * @param email
     * @return
     */
    public DeveloperDto findDeveloperByEmail(String email){
        DeveloperEntity developer = developerService.findByEmail(email);
        return transform(developer);
    }

    private DeveloperDto transform(DeveloperEntity developer){
        DeveloperDto developerDto = new DeveloperDto();
        try{
            DEVELOPER_COPY.copy(developer,developerDto,null);
        }catch(Exception e){
            logger.error("DeveloperDto转换失败",e);
            return null;
        }
        return developerDto;
    }
}
