package cn.com.duiba.developer.center.biz.service.credits.impl;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.biz.dataobject.credits.AppDO;
import cn.com.duiba.developer.center.api.domain.vo.AppDeveloperVO;
import cn.com.duiba.developer.center.api.domain.vo.AppManageVO;
import cn.com.duiba.developer.center.biz.dao.app.AppDao;
import cn.com.duiba.developer.center.biz.event.AppUpdateEvent;
import cn.com.duiba.developer.center.biz.service.credits.AppService;
import cn.com.duiba.developer.center.common.constants.CacheConstants;
import cn.com.duiba.developer.center.common.support.BizEventBus;
import cn.com.duiba.developer.center.common.tools.SecureTool;
import cn.com.duiba.developer.center.common.tools.SwitchTool;
import cn.com.duiba.wolf.cache.CacheClient;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.eventbus.Subscribe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * Created by liuyao on 16/7/18.
 */
@Service
public class AppServiceImpl implements AppService {
    private static final Logger log= LoggerFactory.getLogger(AppServiceImpl.class);
    @Resource
    private AppDao appDao;
    @Autowired
    private CacheClient memcachedClient;
    @Autowired
    private BizEventBus eventBus;

    @PostConstruct
    public void init(){
        eventBus.register(this);
    }


    //TODO 因为是部分迁移,在清除App缓存时同时清除duiba-service服务的App缓存。等完全迁移后删掉这个代码,以及AppUpdateEvent
    @Subscribe
    public void AppUpdateLister(AppUpdateEvent event){
        Long id = event.getAppId();

        AppSimpleDto app = this.getObject(id);
        memcachedClient.remove("chaos.key_appId_"+id);
        memcachedClient.remove("chaos.key_appId_"+app.getAppKey());
    }

    @Override
    public List<AppSimpleDto> findByIds(List<Long> ids) {
        if(Objects.equals(null,ids) || ids.isEmpty()){
            return Collections.emptyList();
        }
        return appDao.findByIds(ids);

    }
    @Override
    public List<AppSimpleDto> findAll() {
        return appDao.findAll();
    }

    @Override
    public AppSimpleDto getObject(Long appId){
        AppSimpleDto a = memcachedClient.get(getAppIdKey(appId));
        if(Objects.equals(null,a)) {
            a = appDao.find(appId);
            if(a!=null) {
                a.setAppSecret(getAppSecret(appId));
                memcachedClient.set(getAppIdKey(appId), a, 300);
            }
        }
        return a;
    }

    private String getAppIdKey(Long id){
        return CacheConstants.KEY_APP + id;
    }

    private void removeCacheById(Long id){
        try{
            memcachedClient.remove(getAppIdKey(id));
            AppUpdateEvent event = new AppUpdateEvent();
            event.setAppId(id);
            eventBus.post(event);
        }catch(Exception e){
            log.error("App缓存清除失败",e);
        }

    }

    public void insert(AppDO app) {
        appDao.insert(app);
    }

    @Override
    public void update(AppSimpleDto app) {
        appDao.update(app);
        removeCacheById(app.getId());
    }

    @Override
    public void updateUrls(AppSimpleDto app) {
        appDao.updateUrls(app);
        removeCacheById(app.getId());
    }

    public List<Long> findAllEnable() {
        return appDao.findAllEnable();
    }

    public int updateSwitch(SwitchTool tool) {
        int ret = appDao.updateSwitch(tool.getId(), tool.getSwicthValue());
        removeCacheById(tool.getId());
        return ret;
    }

    public List<AppSimpleDto> findAllBySpecifyAndBlack(List<Long> specifyAppIds, List<Long> blackAppIds) {
        return appDao.findAllBySpecifyAndBlack(specifyAppIds, blackAppIds);
    }
    @Override
    public AppSimpleDto findByAppKey(String appKey) {
        return appDao.findByAppKey(appKey);
    }
    @Override
    public List<AppSimpleDto> findAllByDeveloper(Long developerId) {
        return appDao.findAllByDeveloper(developerId);
    }
    @Override
    public List<AppSimpleDto> findAppNameByNameLike(String appName) {
        return appDao.findAppNameByNameLike(appName);
    }
    @Override
    public Integer countItemAutoRecommandApp(Long itemId) {
        return appDao.countItemAutoRecommandApp(itemId);
    }
    @Override
    public List<Long> getAutoRecommandAppIdListByItemId(Long itemId) {
        return appDao.getAutoRecommandAppIdListByItemId(itemId);
    }
    @Override
    public List<AppSimpleDto> findAllByDeveloperIds(List<Long> developerIds) {
        if(developerIds == null) throw new RuntimeException("查询App时,传入developerIds为空");
        if(developerIds.isEmpty()) return Collections.emptyList();
        return appDao.findAllByDeveloperIds(developerIds);
    }
    @Override
    public List<AppManageVO> findAppListByCondition(Map<String, Object> queryMap) {
        return appDao.findAppListByCondition(queryMap);
    }
    @Override
    public Integer findAppListCount(Map<String, Object> queryMap) {
        return appDao.findAppListCount(queryMap);
    }
    @Override
    public AppSimpleDto findFristAppByDeveloperId(Long developerId){
        String key = CacheConstants.FIRST_APP_OF_DEV_ID + developerId;
        AppSimpleDto firstApp = memcachedClient.get(key);
        if(firstApp == null){
            firstApp = appDao.findFristAppByDeveloperId(developerId);
            memcachedClient.set(key,firstApp,1, TimeUnit.HOURS);
        }
        return firstApp;
    }
    @Override
    public List<AppSimpleDto> findByAppIds(List<Long> appIds) {
        if(appIds.isEmpty()) return Collections.emptyList();
        return appDao.findByAppIds(appIds);
    }
    @Override
    public List<AppDeveloperVO> selectAppDeveloper(List<Long> appIdList) {
        return appDao.selectAppDeveloper(appIdList);
    }
    @Override
    public List<AppDeveloperVO> selectAllAppDeveloper(Integer start, Integer pageSize) {
        return appDao.selectAllAppDeveloper(start, pageSize);
    }
    @Override
    public Map<String, Object> selectOneAppDeveloper(Long appId) {
        return appDao.selectOneAppDeveloper(appId);
    }
    @Override
    public List<Long> selectAppIdByName(String appName) {
        return appDao.selectAppIdByName(appName);
    }
    @Override
    public List<Long> selectAppIdByEmail(String email) {
        return appDao.selectAppIdByEmail(email);
    }
    @Override
    public List<Long> selectAppIdByEmailAndName(String email, String appName) {
        return appDao.selectAppIdByEmailAndName(email, appName);
    }
    @Override
    public Integer countApp() {
        return appDao.countApp();
    }
    @Override
    public List<AppDeveloperVO> selectAppDeveloperByAppId(List<Long> appIdList) {
        return appDao.selectAppDeveloperByAppId(appIdList);
    }

    @Override
    public Integer findAppCountByDevId(Long developerId) {
        return appDao.findAppCountByDevId(developerId);
    }

    private Cache<Long,String> appSecretCache = CacheBuilder.newBuilder().expireAfterWrite(1, TimeUnit.DAYS).build();
    @Override
    /**
     * secretCode 是永恒不变的,不需要失效处理
     */
    public String getAppSecret(final Long id){
        try {
            return appSecretCache.get(id, new Callable<String>() {
                @Override
                public String call() throws Exception {
                    String secretCode = appDao.findAppSecretCode(id);
                    return SecureTool.decryptAppSecretCode(secretCode);
                }
            });
        } catch (ExecutionException e) {
            log.error("加载AppSecret失败",e);
            return "";
        } catch(com.google.common.cache.CacheLoader.InvalidCacheLoadException e){
            log.error("加载AppSecret失败",e);
            return "";
        }
    }

    @Override
    public void updateColor(Long id, String color) {
        appDao.updateAppColor(id,color);
    }


}
