package cn.com.duiba.developer.center.biz.service.schedule.impl;

import static com.google.common.collect.Lists.newLinkedList;

import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.developer.center.api.domain.dto.AppPushTaskDto;
import cn.com.duiba.developer.center.api.domain.enums.AppPushStatusEnum;
import cn.com.duiba.developer.center.api.domain.enums.DuibaTaskStatusEnum;
import cn.com.duiba.developer.center.biz.dao.schedule.AppPushTaskDao;
import cn.com.duiba.developer.center.biz.dao.schedule.DuibaTaskDao;
import cn.com.duiba.developer.center.biz.entity.AppPushTaskEntity;
import cn.com.duiba.developer.center.biz.entity.DuibaTaskEntity;
import cn.com.duiba.developer.center.biz.service.schedule.AppPushTaskService;
import cn.com.duiba.developer.center.common.constants.DsConstants;
import cn.com.duiba.developer.center.common.constants.ScheduleTaskConstants;
import cn.com.duiba.wolf.utils.ObjectUtil;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * @Author【yaocheng@duiba.com.cn】
 * @Date 2016/11/8 10:29
 */
@Service
public class AppPushServiceImpl implements AppPushTaskService {

    @Autowired
    private AppPushTaskDao   appPushTaskDao;

    @Autowired
    private DuibaTaskDao     duibaTaskDao;

    // 每次批量捞取计划任务数量
    private static final int QUERY_TASK_LIMIT = 200;

    @Override
    public List<AppPushTaskDto> queryAppPushTaskForEdit(Long duibaTaskId) {
        List<AppPushTaskEntity> entities = appPushTaskDao.queryAppPushTaskByDuibaId(duibaTaskId);
        List<AppPushTaskDto> appPushTaskDtos = convertPushEntity2TaskDto(entities);
        return appPushTaskDtos;
    }

    @Override
    @Transactional(DsConstants.DATABASE_CREDITS)
    public void create(List<AppPushTaskDto> appPushTasks) {
        DuibaTaskEntity duibaTaskEntity = convertTaskDto2DuibaEntity(appPushTasks.get(0));
        Long id = duibaTaskDao.insert(duibaTaskEntity);
        for (AppPushTaskDto appPushTaskDto : appPushTasks) {
            appPushTaskDto.setDuibaTaskId(id);
        }
        List<AppPushTaskEntity> pushTaskEntitys = convertTaskDto2PushEntity(appPushTasks);
        for (AppPushTaskEntity pushTaskEntity : pushTaskEntitys) {
            appPushTaskDao.insert(pushTaskEntity);
        }
    }

    @Override
    @Transactional(DsConstants.DATABASE_CREDITS)
    public void update(List<AppPushTaskDto> appPushTasks) {
        DuibaTaskEntity duibaTaskEntity = convertTaskDto2DuibaEntity(appPushTasks.get(0));
        duibaTaskDao.update(duibaTaskEntity);
        appPushTaskDao.deleteByDuibaTaskId(appPushTasks.get(0).getDuibaTaskId());
        List<AppPushTaskEntity> pushTaskEntitys = convertTaskDto2PushEntity(appPushTasks);
        for (AppPushTaskEntity pushTaskEntity : pushTaskEntitys) {
            appPushTaskDao.insert(pushTaskEntity);
        }
    }

    @Override
    public void executeRightNow(Long duibaTaskId) {
        Date date = new Date();
        appPushTaskDao.updatePushTimeByDuibaTaskId(duibaTaskId, date);
        duibaTaskDao.updatePushTimeById(duibaTaskId, date);
    }

    @Override
    public void delete(Long duibaTaskId) {
        duibaTaskDao.delete(duibaTaskId, ScheduleTaskConstants.APP_PUSH_TASK_DELETED_TRUE);
        appPushTaskDao.delete(duibaTaskId, ScheduleTaskConstants.APP_PUSH_TASK_DELETED_TRUE);
    }

    private List<AppPushTaskEntity> convertTaskDto2PushEntity(List<AppPushTaskDto> appPushTasks) {
        List<AppPushTaskEntity> pushTaskEntitys = Lists.newLinkedList();
        for (AppPushTaskDto appPushTaskDto : appPushTasks) {
            AppPushTaskEntity entity = new AppPushTaskEntity();
            entity.setAppId(appPushTaskDto.getAppId());
            if (appPushTaskDto.getId() != null) {
                entity.setId(appPushTaskDto.getId());
            }
            entity.setDeleted(ScheduleTaskConstants.APP_PUSH_TASK_DELETED_FALSE);
            if (appPushTaskDto.getDuibaTaskId() != null) {
                entity.setDuibaTaskId(appPushTaskDto.getDuibaTaskId());
            }
            Map<String,String> extParam=appPushTaskDto.getExtParam();
            entity.setExtParam(extParam!=null?JSON.toJSONString(extParam):null);
            entity.setPushTime(appPushTaskDto.getPushTime());
            entity.setPushTypes(JSON.toJSONString(appPushTaskDto.getPushTypes()));
            entity.setSourceId(appPushTaskDto.getSourceId());
            entity.setSourceType(appPushTaskDto.getSourceType());
            entity.setStatus(AppPushStatusEnum.INIT.getCode());
            pushTaskEntitys.add(entity);
        }
        return pushTaskEntitys;
    }

    private DuibaTaskEntity convertTaskDto2DuibaEntity(AppPushTaskDto appPushTask) {
        DuibaTaskEntity entity = new DuibaTaskEntity();
        if(appPushTask == null){
        	return entity;
        }
        if(appPushTask.getExtParam() !=null && !appPushTask.getExtParam().isEmpty()){
        	entity.setExtParam(JSON.toJSONString(appPushTask.getExtParam()));
        }
        entity.setSourceType(appPushTask.getSourceType());
        entity.setStatus(DuibaTaskStatusEnum.OPEN.getCode());
        entity.setId(appPushTask.getDuibaTaskId());
        if(CollectionUtils.isNotEmpty(appPushTask.getPushTypes())){
        	entity.setPushTypes(JSON.toJSONString(appPushTask.getPushTypes()));
        }
        entity.setDeleted(ScheduleTaskConstants.APP_PUSH_TASK_DELETED_FALSE);
        entity.setPushTime(appPushTask.getPushTime());
        entity.setTitle(appPushTask.getTitle());
        entity.setSourceName(appPushTask.getSourceName());
        return entity;
    }

    private List<AppPushTaskDto> convertPushEntity2TaskDto(List<AppPushTaskEntity> pushTaskEntitys) {
        List<AppPushTaskDto> appPushTaskDtos = newLinkedList();
        for (AppPushTaskEntity entity : pushTaskEntitys) {
            appPushTaskDtos.add(convertToDto(entity));
        }
        return appPushTaskDtos;
    }

    @SuppressWarnings("unchecked")
    private AppPushTaskDto convertToDto(AppPushTaskEntity appPushTaskEntity) {
        if (appPushTaskEntity == null) {
            return null;
        }
        AppPushTaskDto appPushTaskDto = new AppPushTaskDto();
        appPushTaskDto.setId(appPushTaskEntity.getId());
        appPushTaskDto.setSourceId(appPushTaskEntity.getSourceId());
        appPushTaskDto.setSourceType(appPushTaskEntity.getSourceType());
        appPushTaskDto.setAppId(appPushTaskEntity.getAppId());
        if(StringUtils.isNotBlank(appPushTaskEntity.getPushTypes())){
        	appPushTaskDto.setPushTypes(JSONObject.parseArray(appPushTaskEntity.getPushTypes(), String.class));
        }
        appPushTaskDto.setStatus(appPushTaskEntity.getStatus());
        appPushTaskDto.setPushTime(appPushTaskEntity.getPushTime());
        if(StringUtils.isNotBlank(appPushTaskEntity.getExtParam())){
        	appPushTaskDto.setExtParam(JSON.parseObject(appPushTaskEntity.getExtParam(), Map.class));
        }
        return appPushTaskDto;
    }

    @Override
    public List<AppPushTaskDto> queryAppPushTaskToExec() {
        List<AppPushTaskEntity> list = appPushTaskDao.selectAppPushTaskToExec(QUERY_TASK_LIMIT);
        return convertPushEntity2TaskDto(list);
    }

    @Override
    public void updateAppPushTaskStatus(long appPushTaskId, String status) {
        appPushTaskDao.updateAppPushTaskStatus(appPushTaskId, status);
    }

    @Override
    public void updateAppPushTaskTime(long appPushTaskId, Date pushTime) {
        appPushTaskDao.updateAppPushTaskTime(appPushTaskId, pushTime);
    }

    @Override
    public void updateExtParamById(long appPushTaskId, String key, String value) {
        AppPushTaskDto dto = convertToDto(appPushTaskDao.selectById(appPushTaskId));
        if (dto != null) {
            Map<String, String> extParam = dto.getExtParam();
            if (extParam == null) {
                extParam = new HashMap<String, String>();
            }
            extParam.put(key, value);
            appPushTaskDao.updateExtParamById(appPushTaskId, JSON.toJSONString(extParam));
        }
    }

    @Override
    public List<AppPushTaskDto> findScheduledTaskList(int pageIndex, int pageSize, AppPushTaskDto appPushTaskDto) {
        DuibaTaskEntity param = convertTaskDto2DuibaEntity(appPushTaskDto);
        List<DuibaTaskEntity> list = duibaTaskDao.selectScheduledTaskList(pageIndex, pageSize, param);
        List<AppPushTaskDto> ret = Lists.newArrayList();
        for(DuibaTaskEntity dte: list){
        	AppPushTaskDto apt = new AppPushTaskDto();
        	apt.setId(dte.getId());
        	apt.setDuibaTaskId(dte.getId());
        	if(StringUtils.isNotBlank(dte.getExtParam())){
        		apt.setExtParam(JSON.parseObject(dte.getExtParam(), Map.class));
        	}
        	apt.setPushTime(dte.getPushTime());
        	if(StringUtils.isNotBlank(dte.getPushTypes())){
        		apt.setPushTypes(JSONObject.parseArray(dte.getPushTypes(), String.class));
        	}
        	apt.setSourceType(dte.getSourceType());
        	apt.setSourceName(dte.getSourceName());
        	apt.setStatus(dte.getStatus());
        	apt.setTitle(dte.getTitle());
        	ret.add(apt);
        }
        return ret;
    }

    @Override
    public Integer findScheduledTaskListCount(AppPushTaskDto appPushTaskDto) {
        return duibaTaskDao.selectScheduledTaskListCount(convertTaskDto2DuibaEntity(appPushTaskDto));
    }

    @Override
    public List<AppPushTaskDto> findScheduledTaskAppList(int pageIndex, int pageSize) {
        List<AppPushTaskEntity> list = appPushTaskDao.selectScheduledTaskAppList(pageIndex, pageSize);
        return convertPushEntity2TaskDto(list);
    }

    @Override
    public Integer findScheduledTaskAppCount() {
        return appPushTaskDao.selectScheduledTaskAppCount();
    }

    @Transactional(DsConstants.DATABASE_CREDITS)
    @Override
    public void openPushTask(Long duibaTaskId) {
        appPushTaskDao.updateFilterStatus(duibaTaskId, AppPushStatusEnum.TO_EXEC.getCode());
        duibaTaskDao.updateStatus(duibaTaskId, DuibaTaskStatusEnum.OPEN.getCode());
    }

    @Transactional(DsConstants.DATABASE_CREDITS)
    @Override
    public void closePushTask(Long duibaTaskId) {
        appPushTaskDao.updateFilterStatus(duibaTaskId, AppPushStatusEnum.INIT.getCode());
        duibaTaskDao.updateStatus(duibaTaskId, DuibaTaskStatusEnum.CLOSE.getCode());
    }

	@Override
	public Map<Long,Integer> findBatchScheduledTaskAppNums(List<Long> pushTaskIds) {
		Map<Long,Map<String,String>> ret = appPushTaskDao.selectBatchScheduledTaskAppNums(pushTaskIds);
		Map<Long,Integer> res = Maps.newHashMap();
		for(Long key: ret.keySet()){
			Map<String,String> map = ret.get(key);
			Object o = map.get("cont");
			Long l  =(long)o;
			res.put((long)key, l.intValue());
		}
		
		return  res;
	}
}
