package cn.com.duiba.tuia.dsp.engine.api.dsp;

import cn.com.duiba.tuia.dsp.engine.api.config.DspRateLimiter;
import cn.com.duiba.tuia.dsp.engine.api.config.RateLimiterProperties;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.AdxCommonBidRequest;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.AdxCommonBidResponse;
import cn.com.duiba.tuia.dsp.engine.api.enums.DspEnum;
import cn.com.duiba.tuia.dsp.engine.api.exception.DspException;
import com.dianping.cat.Cat;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

/**
 * 提供adx-web进行外部竞价的门面接口
 *
 * @author lizhiheng
 */
@SuppressWarnings({"rawtypes", "UnstableApiUsage"})
@Component
public class DspFacade extends DspFacadeInterface implements ApplicationContextAware {

    private ApplicationContext applicationContext;

    // FIXME: 2022/7/8 需要修改成实际使用的线程池
    @Resource
    ExecutorService executorService;
    @Resource
    RateLimiterProperties rateLimiterProperties;
    @Resource
    DspRateLimiter rateLimiter;

    private static final Logger LOGGER = LoggerFactory.getLogger(DspFacade.class);

    @Override
    public AdxCommonBidResponse doInvoke(AdxCommonBidRequest adxCommonBidRequest, List<Integer> dspIds) throws DspException {
        List<AdxCommonBidResponse> adxCommonBidResponses = this.generateFutures(adxCommonBidRequest, dspIds);
        return this.comparePrice(adxCommonBidResponses);
    }

    @Override
    public String priceEncryption(BigDecimal price, Integer dspId) {
        AbstractDspCaller caller = applicationContext.getBean(DspEnum.getByDspId(dspId).getClazz());
        return caller.priceEncryption(price);
    }

    public List<AdxCommonBidResponse> generateFutures(AdxCommonBidRequest adxCommonBidRequest, List<Integer> dspIds) {
        if (rateLimiter.getRateLimiter().tryAcquire(rateLimiterProperties.getRequireTimeout(), TimeUnit.MICROSECONDS)) {
            List<CompletableFuture<AdxCommonBidResponse>> futures = new ArrayList<>();
            dspIds.forEach(idea -> {
                AbstractDspCaller dspCaller = applicationContext.getBean(DspEnum.getByDspId(idea).getClazz());
                CompletableFuture<AdxCommonBidResponse> adxCommonBidResponseCompletableFuture =
                        CompletableFuture.supplyAsync(() -> dspCaller.doBid(adxCommonBidRequest), executorService);
                futures.add(adxCommonBidResponseCompletableFuture);
            });
            try {
                CompletableFuture.allOf(futures.toArray(new CompletableFuture[]{})).get(rateLimiterProperties.getFutureTimeOut(), TimeUnit.MILLISECONDS);
            } catch (TimeoutException e) {
                Cat.logMetricForCount("DSP并行请求总线执行超时");
            } catch (InterruptedException e) {
                Cat.logMetricForCount("DSP并行请求总线执行中断");
            } catch (Exception e) {
                Cat.logMetricForCount("DSP并行请求总线执行失败");
                LOGGER.warn("DSP并行请求总线执行失败", e);
            }

            return futures.stream().map(future -> {
                try {
                    return future.get(rateLimiterProperties.getFutureTimeOut(), TimeUnit.MILLISECONDS);
                } catch (TimeoutException e) {
                    Cat.logMetricForCount("DSP请求单个任务超时");
                } catch (InterruptedException e) {
                    Cat.logMetricForCount("DSP请求单个任务中断");
                } catch (Exception e) {
                    Cat.logMetricForCount("DSP请求单个DSP失败");
                    LOGGER.warn("DSP请求单个DSP失败", e);
                }
                return null;
            }).collect(Collectors.toList());
        } else {
            Cat.logMetricForCount("DSP限流");
            // 采样日志
            int i = (int) (Math.random() * 10000);
            if (i <= 1) {
                LOGGER.info("DSP请求限流，超过阈值，{}个/s", rateLimiterProperties.getTokenPerSec());
            }
            return null;
        }

    }

    @Override
    public void setApplicationContext(@NotNull ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}