package cn.com.duiba.tuia.dsp.engine.api.dsp;

import cn.com.duiba.spring.boot.starter.dsp.sampler.SamplerLog;
import cn.com.duiba.tuia.adx.center.api.dto.IdeaPriceStrategyDTO;
import cn.com.duiba.tuia.adx.center.api.dto.ResourceIdeaDto;
import cn.com.duiba.tuia.adx.center.api.enums.DspTargetType;
import cn.com.duiba.tuia.adx.center.api.remoteservice.RemoteResourceIdeaService;
import cn.com.duiba.tuia.dsp.engine.api.util.StringRedisHandler;
import com.alibaba.fastjson.JSONObject;
import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * 出价相关服务
 * 工程层面出价
 * adx曝光 x_e
 * adx-cpm消耗 x_m_f
 * adx点击 x_c
 * adx-cpc消耗 x_c_f
 * 活动访问 a_r
 * 活动参与 a_j
 * 广告券计费点击 ef_c
 * 广告消耗 f
 * 落地页转化 l_c
 * 后端转化 subType值，1-N
 */
@Slf4j
@Service
public class BidStatService {
    private static final String OMD = "OMD_";
    private static final String DAY = "1";
    private static final String HOUR = "2";
    private static final String MINUTE = "3";

    private static final String PCTR = "pctr_";
    private static final String PCVR = "pcvr_";
    private static final String FACTOR = "factor_";
    private static final String LAST_FACTOR = "last_factor_";
    private static final String ROI_FACTOR = "roi_factor_";
    private static final String LASE_ROI_FACTOR = "last_roi_factor_";

    public static final String DEFAULT_PCTR = "0.15";
    private static final String DEFAULT_PCVR = "0.02";
    private static final String DEFAULT_FACTOR = "1.0";

    @Resource
    private ExecutorService executorService = Executors.newFixedThreadPool(10);
    @Resource
    private StringRedisHandler stringRedisHandler;
    @Resource
    private RemoteResourceIdeaService remoteResourceIdeaService;

    //获取大数据数据的缓存
    private AsyncLoadingCache<String, Optional<Map<String, JSONObject>>> omdDataCache = Caffeine.newBuilder()
            .refreshAfterWrite(1, TimeUnit.MINUTES)
            .executor(executorService)
            .buildAsync(key -> Optional.of(getData(key)));

    /**
     * 创意缓存
     */
    private AsyncLoadingCache<Long, Optional<ResourceIdeaDto>> resourceIdeaCache = Caffeine.newBuilder()
            .refreshAfterWrite(2, TimeUnit.MINUTES)
            .buildAsync(key -> {
                try {
                    return Optional.ofNullable(remoteResourceIdeaService.getIdeaById(key));
                } catch (Exception e) {
                    // error级别降到warn级别
                    log.warn(String.format("load resourceIdeaCache error: %s", key), e);
                }
                return Optional.empty();
            });

    //PCTR
    //● 资源位_计划_联盟媒体_联盟广告位>
    //● 资源位_联盟媒体_联盟广告位>
    private AsyncLoadingCache<String, Optional<String>> pCTRCache = Caffeine.newBuilder()
            .refreshAfterWrite(1, TimeUnit.MINUTES)
            .executor(executorService)
            .buildAsync(key -> {
                try {
                    if (StringUtils.isBlank(key)) {
                        return Optional.of(DEFAULT_PCTR);
                    }
                    String[] s = key.split("_");
                    if (s.length == 4) {
                        String resourceId = s[0];
                        String ideaId = s[1];
                        String appId = s[2];
                        String slotId = s[3];
                        String redisKey = PCTR + key;
                        Optional<String> optional = Optional.ofNullable(stringRedisHandler.get(redisKey));
                        if (optional.isPresent()) {
                            return optional;
                        }
                        //计算pctr
                        Double pCtr = calculatePCTR(resourceId, ideaId, appId, slotId);
                        if (pCtr != null) {
                            return Optional.of(String.valueOf(pCtr));
                        }
                        return Optional.empty();
                    }
                    if (s.length == 3) {
                        String redisKey = PCTR + key;
                        String resourceId = s[0];
                        String appId = s[1];
                        String slotId = s[2];
                        Optional<String> optional = Optional.ofNullable(stringRedisHandler.get(redisKey));
                        if (optional.isPresent()) {
                            return optional;
                        }
                        //计算pctr
                        Double pCtr = calculatePCTR(resourceId, null, appId, slotId);
                        if (pCtr != null) {
                            return Optional.of(String.valueOf(pCtr));
                        }
                    }

                } catch (Exception e) {
                    log.error(String.format("load pCTRCache error: %s", key), e);
                }
                return Optional.of(DEFAULT_PCTR);
            });

    //pcvr
    private AsyncLoadingCache<String, Optional<String>> pCVRCache = Caffeine.newBuilder()
            .refreshAfterWrite(1, TimeUnit.SECONDS)
            .executor(executorService)
            .buildAsync(key -> {
                try {
                    if (StringUtils.isBlank(key)) {
                        return Optional.of(DEFAULT_PCVR);
                    }
                    String[] s = key.split("_");
                    if (s.length == 4) {
                        String resourceId = s[0];
                        String ideaId = s[1];
                        String appId = s[2];
                        String slotId = s[3];
//                        String redisKey = PCVR + key;
//                        Optional<String> optional = Optional.ofNullable(stringRedisHandler.get(redisKey));
//                        if (optional.isPresent()) {
//                            return optional;
//                        }
                        //计算pcvr
                        Double pCtr = calculatePCVR(resourceId, ideaId, appId, slotId);
                        if (pCtr != null) {
                            return Optional.of(String.valueOf(pCtr));
                        }
                    }
                } catch (Exception e) {
                    log.error(String.format("load pCVRCache error: %s", key), e);
                }
                return Optional.of(DEFAULT_PCVR);
            });

    //factor 维稳因子
    private AsyncLoadingCache<String, Optional<String>> factorCache = Caffeine.newBuilder()
            .refreshAfterWrite(1, TimeUnit.MINUTES)
            .executor(executorService)
            .buildAsync(key -> {
                try {
                    if (StringUtils.isBlank(key)) {
                        return Optional.of(DEFAULT_FACTOR);
                    }
                    String[] s = key.split("_");
                    if (s.length != 4) {
                        return Optional.empty();
                    }
                    String resourceId = s[0];
                    String ideaId = s[1];
                    String appId = s[2];
                    String slotId = s[3];
                    String redisKey = FACTOR + key;
                    Optional<String> optional = Optional.ofNullable(stringRedisHandler.get(redisKey));
                    if (optional.isPresent()) {
                        return optional;
                    }
                    //计算factor
                    Double factor = calculateFactor(resourceId, ideaId, appId, slotId);
                    if (factor != null && !factor.isNaN()) {
//                        stringRedisHandler.set(redisKey, String.valueOf(factor), 1, TimeUnit.MINUTES);
                        return Optional.of(String.valueOf(factor));
                    }
                } catch (Exception e) {
                    log.error(String.format("load factorCache error: %s", key), e);
                }
                return Optional.of(DEFAULT_FACTOR);
            });

    //roi 维稳因子
    private AsyncLoadingCache<String, Optional<String>> roiFactorCache = Caffeine.newBuilder()
            .refreshAfterWrite(1, TimeUnit.MINUTES)
            .executor(executorService)
            .buildAsync(key -> {
                try {
                    String[] s = key.split("_");
                    if (s.length != 4) {
                        return Optional.empty();
                    }
                    String resourceId = s[0];
                    String ideaId = s[1];
                    String appId = s[2];
                    String slotId = s[3];
                    String redisKey = ROI_FACTOR + key;
                    Optional<String> optional = Optional.ofNullable(stringRedisHandler.get(redisKey));
                    if (optional.isPresent()) {
                        return optional;
                    }
                    //计算roi factor
                    Double roiFactor = calculateRoiFactor(resourceId, ideaId, appId, slotId);
                    if (roiFactor != null) {
                        return Optional.of(String.valueOf(roiFactor));
                    }
                } catch (Exception e) {
                    log.error(String.format("load roiFactorCache error: %s", key), e);
                }
                return Optional.of(DEFAULT_FACTOR);
            });

    //资源位id 计划id 联盟媒体id 联盟广告位id
    private Double calculatePCTR(String resourceId, String ideaId, String appId, String slotId) {
        String baseKey = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        if (StringUtils.isEmpty(ideaId)) {
            baseKey = resourceId + "_" + appId + "_" + slotId;
        }
        Optional<Map<String, JSONObject>> ideaResultMapOptional
                = omdDataCache.get(baseKey).getNow(Optional.empty());
        if (ideaResultMapOptional.isPresent()) {
            String key_min = OMD + baseKey + "_" + MINUTE;
            String key_hour = OMD + baseKey + "_" + HOUR;
            String key_day = OMD + baseKey + "_" + DAY;
            Map<String, JSONObject> resultMap = ideaResultMapOptional.get();
            if (resultMap.get(key_min) == null && resultMap.get(key_hour) == null && resultMap.get(key_day) == null) {
                SamplerLog.info("ocpm ideaId is {} appId is {} slotId is {} pctr 获取无数据", ideaId, appId, slotId);
                return null;
            }

            JSONObject value_min = resultMap.get(key_min);
            JSONObject value_hour = resultMap.get(key_hour);
            JSONObject value_day = resultMap.get(key_day);
            if (value_min == null && value_hour == null && value_day == null) {
                return null;
            }
            if (value_min == null) {
                value_min = new JSONObject();
            }
            if (value_hour == null) {
                value_hour = new JSONObject();
            }
            if (value_day == null) {
                value_day = new JSONObject();
            }
            return getFusionPCtr(value_min, value_hour, value_day);
        }
        return null;
    }

    //获取融合ctr
    private Double getFusionPCtr(JSONObject value_min, JSONObject value_hour, JSONObject value_day) {
        int click_min = value_min != null && value_min.get("x_c") != null ? value_min.getInteger("x_c") : 0;
        int click_hour = value_hour != null && value_hour.get("x_c") != null ? value_hour.getInteger("x_c") : 0;
        int click_day = value_day != null && value_day.get("x_c") != null ? value_day.getInteger("x_c") : 0;
        int exposure_min = value_min != null && value_min.get("x_e") != null ? value_min.getInteger("x_e") : 0;
        int exposure_hour = value_hour != null && value_hour.get("x_e") != null ? value_hour.getInteger("x_e") : 0;
        int exposure_day = value_day != null && value_day.get("x_e") != null ? value_day.getInteger("x_e") : 0;

        if (exposure_day == 0 && exposure_hour == 0 && exposure_min == 0) {
            return null;
        }
        if (click_day < 20) {
            return null;
        }
        if (click_hour < 20) {
            //近1小时点击小于20 使用最近一天数据
            return 1.0 * click_day / exposure_day;
        }
        if (click_min < 20) {
            //近10分钟点击小于20 使用0.3最近一天+0.7最近一小时
            //DataUtil.division()
            return 0.3 * click_day / exposure_day + 0.7 * click_hour / exposure_hour;
        }
        //0.1*最近一天 + 0.3*最近一小时 + 0.6*最近10分钟
        return 0.1 * click_day / exposure_day + 0.3 * click_hour / exposure_hour + 0.6 * click_min / exposure_min;
    }

    private Double calculatePCVR(String resourceId, String ideaId, String appId, String slotId) {
        String baseKey = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        Optional<Map<String, JSONObject>> ideaResultMapOptional
                = omdDataCache.get(baseKey).getNow(Optional.empty());
        if (ideaResultMapOptional.isPresent() && StringUtils.isNotEmpty(ideaId)) {
            String key_min = OMD + baseKey + "_" + MINUTE;
            String key_hour = OMD + baseKey + "_" + HOUR;
            String key_day = OMD + baseKey + "_" + DAY;
            Map<String, JSONObject> resultMap = ideaResultMapOptional.get();
            Optional<ResourceIdeaDto> ideaOptional = resourceIdeaCache.get(Long.valueOf(ideaId)).getNow(Optional.empty());
            if (!ideaOptional.isPresent()) {
                SamplerLog.info("getFusionPcvr 没有找到创意返回默认值 ideaId is {}", ideaId);
                return null;
            }
            ResourceIdeaDto resourceIdeaDto = ideaOptional.get();
            IdeaPriceStrategyDTO ideaPriceStrategyDTO = resourceIdeaDto.getIdeaPriceStrategyDTO();
            if (ideaPriceStrategyDTO == null) {
                SamplerLog.info("getFusionPcvr 没有找到创意的转化类型 ideaId is {}", ideaId);
                return null;
            }
            Integer targetType = ideaPriceStrategyDTO.getTargetType();
            if (Objects.equals(targetType, DspTargetType.MATERIAL_CLICK.getValue())) {
                //转化目标 入口素材点击
                return 1.0;
            }

            if (resultMap.get(key_min) == null && resultMap.get(key_hour) == null && resultMap.get(key_day) == null) {
                SamplerLog.info("ocpm pcvr 获取无数据");
                return null;
            }

            JSONObject value_min = resultMap.get(key_min);
            JSONObject value_hour = resultMap.get(key_hour);
            JSONObject value_day = resultMap.get(key_day);
            if (value_min == null && value_hour == null && value_day == null) {
                return null;
            }
            if (value_min == null) {
                value_min = new JSONObject();
            }
            if (value_hour == null) {
                value_hour = new JSONObject();
            }
            if (value_day == null) {
                value_day = new JSONObject();
            }
            Double fusionPCvr = getFusionPCvr(value_min, value_hour, value_day, ideaId);
            log.info("ocpm fusionPCvr is {} ideaId is {}", fusionPCvr, ideaId);
        }
        return null;
    }

    //获取融合cvr
    private Double getFusionPCvr(JSONObject value_min, JSONObject value_hour, JSONObject value_day, String ideaId) {
        Optional<ResourceIdeaDto> ideaOptional = resourceIdeaCache.get(Long.valueOf(ideaId)).getNow(Optional.empty());
        if (!ideaOptional.isPresent()) {
            SamplerLog.info("getFusionPcvr 没有找到创意返回默认值 ideaId is {}", ideaId);
            return null;
        }
        ResourceIdeaDto resourceIdeaDto = ideaOptional.get();
        IdeaPriceStrategyDTO ideaPriceStrategyDTO = resourceIdeaDto.getIdeaPriceStrategyDTO();
        if (ideaPriceStrategyDTO == null) {
            SamplerLog.info("getFusionPcvr 没有找到创意的转化类型 ideaId is {}", ideaId);
            return null;
        }
        Integer targetType = ideaPriceStrategyDTO.getTargetType();

        String str_click_min = value_min.getString("x_c");
        String str_click_hour = value_hour.getString("x_c");
        String str_click_day = value_day.getString("x_c");

        int convert_min = 0;
        int convert_hour = 0;
        int convert_day = 0;
        int click_min = 0;
        int click_hour = 0;
        int click_day = 0;
        if (StringUtils.isNotEmpty(str_click_min)) {
            click_min = Integer.parseInt(str_click_min);
        }
        if (StringUtils.isNotEmpty(str_click_hour)) {
            click_hour = Integer.parseInt(str_click_hour);
        }
        if (StringUtils.isNotEmpty(str_click_day)) {
            click_day = Integer.parseInt(str_click_day);
        }

        if (Objects.equals(targetType, DspTargetType.ACTIVITY_JOIN.getValue())) {
            //活动参与
            String str_a_join_min = value_min.getString("a_j");
            String str_a_join_hour = value_hour.getString("a_j");
            String str_a_join_day = value_day.getString("a_j");
            if (StringUtils.isNotEmpty(str_a_join_min)) {
                convert_min = Integer.parseInt(str_a_join_min);
            }
            if (StringUtils.isNotEmpty(str_a_join_hour)) {
                convert_hour = Integer.parseInt(str_a_join_hour);
            }
            if (StringUtils.isNotEmpty(str_a_join_day)) {
                convert_day = Integer.parseInt(str_a_join_day);
            }
        } else if (Objects.equals(targetType, DspTargetType.ADVERT_CLICK.getValue())) {
            //券计费点击
            String str_t_click_min = value_min.getString("ef_c");
            String str_t_click_hour = value_hour.getString("ef_c");
            String str_t_click_day = value_day.getString("ef_c");
            if (StringUtils.isNotEmpty(str_t_click_min)) {
                convert_min = Integer.parseInt(str_t_click_min);
            }
            if (StringUtils.isNotEmpty(str_t_click_hour)) {
                convert_hour = Integer.parseInt(str_t_click_hour);
            }
            if (StringUtils.isNotEmpty(str_t_click_day)) {
                convert_day = Integer.parseInt(str_t_click_day);
            }
        } else if (Objects.equals(targetType, DspTargetType.LANDING_PAGE_CONVERSIONS.getValue())) {
            //落地页转化
            String str_l_convert_min = value_min.getString("l_c");
            String str_l_convert_hour = value_hour.getString("l_c");
            String str_l_convert_day = value_day.getString("l_c");
            if (StringUtils.isNotEmpty(str_l_convert_min)) {
                convert_min = Integer.parseInt(str_l_convert_min);
            }
            if (StringUtils.isNotEmpty(str_l_convert_hour)) {
                convert_hour = Integer.parseInt(str_l_convert_hour);
            }
            if (StringUtils.isNotEmpty(str_l_convert_day)) {
                convert_day = Integer.parseInt(str_l_convert_day);
            }
        } else {
            //其他subType处理的
            String str_convert_min = value_min.getString(String.valueOf(targetType));
            String str_convert_hour = value_hour.getString(String.valueOf(targetType));
            String str_convert_day = value_day.getString(String.valueOf(targetType));
            if (StringUtils.isNotEmpty(str_convert_min)) {
                convert_min = Integer.parseInt(str_convert_min);
            }
            if (StringUtils.isNotEmpty(str_convert_hour)) {
                convert_hour = Integer.parseInt(str_convert_hour);
            }
            if (StringUtils.isNotEmpty(str_convert_day)) {
                convert_day = Integer.parseInt(str_convert_day);
            }
        }

        log.info("计算cvr click_day is {} click_hour is {} click_min is {} convert_day is {} convert_hour is {} convert_min is {}",
                click_day, click_hour, click_min, convert_day, convert_hour, convert_min);
        if (click_day == 0 && click_hour == 0 && click_min == 0) {
            return null;
        }
        if (convert_day < 10) {
            return null;
        }
        if (click_hour < 100 && click_day != 0) {
            return 1.0 * convert_day / click_day;
        }
        if (click_min < 50 && click_day != 0) {
            return 0.3 * convert_day / click_day + 0.7 * convert_hour / click_hour;
        }
        return 0.1 * convert_day / click_day + 0.3 * convert_hour / click_hour + 0.6 * convert_min / click_min;
    }

    /**
     * 计算维稳因子
     *
     * @param resourceId
     * @param ideaId
     * @param appId
     * @param slotId
     * @return 维稳因子
     */
    private Double calculateFactor(String resourceId, String ideaId, String appId, String slotId) {
        Optional<ResourceIdeaDto> ideaOptional = resourceIdeaCache.get(Long.valueOf(ideaId)).getNow(Optional.empty());
        if (!ideaOptional.isPresent()) {
            SamplerLog.info("calculateFactor 没有找到创意返回默认值 ideaId is {}", ideaId);
            return null;
        }
        ResourceIdeaDto resourceIdeaDto = ideaOptional.get();
        if (resourceIdeaDto.getIdeaPriceStrategyDTO() == null) {
            SamplerLog.info("calculateFactor 没有找到创意的转化类型 ideaId is {}", ideaId);
            return null;
        }
        IdeaPriceStrategyDTO ideaPriceStrategyDTO = resourceIdeaDto.getIdeaPriceStrategyDTO();
        Integer targetType = ideaPriceStrategyDTO.getTargetType();
        Double targetPrice = ideaPriceStrategyDTO.getTargetPrice();
        if (targetType == null || targetPrice == null) {
            return null;
        }

        int convert_min = 0;
        int convert_hour = 0;
        int convert_day = 0;
        String baseKey = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        String key_min = OMD + baseKey + "_" + MINUTE;
        String key_hour = OMD + baseKey + "_" + HOUR;
        String key_day = OMD + baseKey + "_" + DAY;
        Optional<Map<String, JSONObject>> ideaResultMapOptional
                = omdDataCache.get(baseKey).getNow(Optional.empty());
        if (!ideaResultMapOptional.isPresent()) {
            return null;
        }
        Map<String, JSONObject> resultMap = ideaResultMapOptional.get();
        JSONObject value_min = resultMap.getOrDefault(key_min, new JSONObject());
        JSONObject value_hour = resultMap.getOrDefault(key_hour, new JSONObject());
        JSONObject value_day = resultMap.getOrDefault(key_day, new JSONObject());
        //获取消耗
        String cpm_fee_min = value_min.getString("x_m_f");
        String cpm_fee_hour = value_hour.getString("x_m_f");
        String cpm_fee_day = value_day.getString("x_m_f");
        String cpc_fee_min = value_min.getString("x_c_f");
        String cpc_fee_hour = value_hour.getString("x_c_f");
        String cpc_fee_day = value_day.getString("x_c_f");
        //单位 分
        long fee_min = 0L;
        long fee_hour = 0L;
        long fee_day = 0L;
        if (StringUtils.isNotEmpty(cpm_fee_min)) {
            long cpm_fee = Long.parseLong(cpm_fee_min);
            fee_min = cpm_fee / 10000000;
            if (StringUtils.isNotEmpty(cpc_fee_min)) {
                long cpc_fee = Long.parseLong(cpc_fee_min);
                fee_min = fee_min + cpc_fee / 10000000;
            }
        }
        if (StringUtils.isNotEmpty(cpm_fee_hour)) {
            long cpm_fee = Long.parseLong(cpm_fee_hour);
            fee_hour = cpm_fee / 10000000;
            if (StringUtils.isNotEmpty(cpc_fee_hour)) {
                long cpc_fee = Long.parseLong(cpc_fee_hour);
                fee_hour = fee_hour + cpc_fee / 10000000;
            }
        }
        if (StringUtils.isNotEmpty(cpm_fee_day)) {
            long cpm_fee = Long.parseLong(cpm_fee_day);
            fee_day = cpm_fee / 10000000;
            if (StringUtils.isNotEmpty(cpc_fee_day)) {
                long cpc_fee = Long.parseLong(cpc_fee_day);
                fee_day = fee_day + cpc_fee / 10000000;
            }
        }

        if (Objects.equals(targetType, DspTargetType.MATERIAL_CLICK.getValue())) {
            //转化目标 入口素材点击
            if (resultMap.get(key_min) == null && resultMap.get(key_hour) == null && resultMap.get(key_day) == null) {
                SamplerLog.info("ocpm 获取数据结果为空 key_min {} key_hour {} key_day {}", key_min, key_hour, key_day);
                return null;
            }
            SamplerLog.info("ocpm key is {} key_min {} key_hour {} key_day {}", baseKey, key_min, key_hour, key_day);
            Integer click_min = value_min.getInteger("x_c");
            Integer click_hour = value_hour.getInteger("x_c");
            Integer click_day = value_day.getInteger("x_c");
            if (click_min == null) {
                click_min = 0;
            }
            if (click_hour == null) {
                click_hour = 0;
            }
            if (click_day == null) {
                click_day = 0;
            }
            convert_min = click_min;
            convert_hour = click_hour;
            convert_day = click_day;
        } else if (Objects.equals(targetType, DspTargetType.ACTIVITY_JOIN.getValue())) {
            //转化目标 活动参与
            if (resultMap.get(key_min) == null && resultMap.get(key_hour) == null && resultMap.get(key_day) == null) {
                SamplerLog.info("ocpm 获取数据结果为空 key_min {} key_hour {} key_day {}", key_min, key_hour, key_day);
                return null;
            }
            Integer join_min = value_min.getInteger("a_j");
            Integer join_hour = value_hour.getInteger("a_j");
            Integer join_day = value_day.getInteger("a_j");
            if (join_min == null) {
                join_min = 0;
            }
            if (join_hour == null) {
                join_hour = 0;
            }
            if (join_day == null) {
                join_day = 0;
            }
            convert_min = join_min;
            convert_hour = join_hour;
            convert_day = join_day;
        } else if (Objects.equals(targetType, DspTargetType.ADVERT_CLICK.getValue())) {
            //转化目标 券点击
            if (resultMap.get(key_min) == null && resultMap.get(key_hour) == null && resultMap.get(key_day) == null) {
                SamplerLog.info("ocpm 获取数据结果为空 key_min {} key_hour {} key_day {}", key_min, key_hour, key_day);
                return null;
            }
            Integer e_click_min = value_min.getInteger("ef_c");
            Integer e_click_hour = value_hour.getInteger("ef_c");
            Integer e_click_day = value_day.getInteger("ef_c");
            if (e_click_min == null) {
                e_click_min = 0;
            }
            if (e_click_hour == null) {
                e_click_hour = 0;
            }
            if (e_click_day == null) {
                e_click_day = 0;
            }
            convert_min = e_click_min;
            convert_hour = e_click_hour;
            convert_day = e_click_day;
        } else if (Objects.equals(targetType, DspTargetType.LANDING_PAGE_CONVERSIONS.getValue())) {
            //转化目标 落地页转化
            if (resultMap.get(key_min) == null && resultMap.get(key_hour) == null && resultMap.get(key_day) == null) {
                SamplerLog.info("ocpm 获取数据结果为空 key_min {} key_hour {} key_day {}", key_min, key_hour, key_day);
                return null;
            }
            Integer l_convert_min = value_min.getInteger("l_c");
            Integer l_convert_hour = value_hour.getInteger("l_c");
            Integer l_convert_day = value_day.getInteger("l_c");
            if (l_convert_min == null) {
                l_convert_min = 0;
            }
            if (l_convert_hour == null) {
                l_convert_hour = 0;
            }
            if (l_convert_day == null) {
                l_convert_day = 0;
            }
            convert_min = l_convert_min;
            convert_hour = l_convert_hour;
            convert_day = l_convert_day;
        } else {
            String str_convert_min = value_min.getString(String.valueOf(targetType));
            String str_convert_hour = value_hour.getString(String.valueOf(targetType));
            String str_convert_day = value_day.getString(String.valueOf(targetType));
            if (StringUtils.isNotEmpty(str_convert_min)) {
                convert_min = Integer.parseInt(str_convert_min);
            }
            if (StringUtils.isNotEmpty(str_convert_hour)) {
                convert_hour = Integer.parseInt(str_convert_hour);
            }
            if (StringUtils.isNotEmpty(str_convert_day)) {
                convert_day = Integer.parseInt(str_convert_day);
            }
        }

        String lastFactorKey_min = LAST_FACTOR + resourceId + "_" + ideaId + "_" + appId + "_" + slotId + "_" + MINUTE;
        String lastFactorKey_hour = LAST_FACTOR + resourceId + "_" + ideaId + "_" + appId + "_" + slotId + "_" + HOUR;
        String lastFactorKey_day = LAST_FACTOR + resourceId + "_" + ideaId + "_" + appId + "_" + slotId + "_" + DAY;
        List<String> factorKeys = new ArrayList<>();
        factorKeys.add(lastFactorKey_min);
        factorKeys.add(lastFactorKey_hour);
        factorKeys.add(lastFactorKey_day);
        Map<String, String> factorMap = stringRedisHandler.pipelineValue(factorKeys);
        double factor_min = factorMap.get(lastFactorKey_min) == null ? 1.0 : Double.parseDouble(factorMap.get(lastFactorKey_min));
        factor_min = checkFactor(factor_min);
        double factor_hour = factorMap.get(lastFactorKey_hour) == null ? 1.0 : Double.parseDouble(factorMap.get(lastFactorKey_hour));
        factor_hour = checkFactor(factor_hour);
        double factor_day = factorMap.get(lastFactorKey_day) == null ? 1.0 : Double.parseDouble(factorMap.get(lastFactorKey_day));
        factor_day = checkFactor(factor_day);
        SamplerLog.info("factor key is {} factor_min is {} factor_hour is {} factor_day is {} targetPrice is {} convert_day is {} " +
                        "fee_day is {} convert_hour is {} fee_hour is {} convert_min is {} fee_min is {} factorMap is {}"
                , baseKey, factor_min, factor_hour, factor_day, targetPrice, convert_day, fee_day, convert_hour, fee_hour, convert_min
                , fee_min, factorMap
        );

        //计算各个时间维度的factor
        double factor_new_day = factor_day * ((targetPrice * 100 * convert_day - fee_day) / fee_day / 2 + 1);
        factor_new_day = checkFactor(factor_new_day);
        if (fee_day < 2000) {
            factor_new_day = 1.0;
        }
        stringRedisHandler.set(lastFactorKey_day, String.valueOf(factor_new_day), 20, TimeUnit.MINUTES);
        double factor_hour_new = factor_hour * (targetPrice * 100 * convert_hour - fee_hour) / fee_hour / 2 + 1;
        factor_hour_new = checkFactor(factor_hour_new);
        if (fee_hour < 2000) {
            factor_hour_new = 1.0;
        }
        stringRedisHandler.set(lastFactorKey_hour, String.valueOf(factor_hour_new), 20, TimeUnit.MINUTES);
        double factor_new_min = factor_min * (targetPrice * 100 * convert_min - fee_min) / fee_min / 2 + 1;
        factor_new_min = checkFactor(factor_new_min);
        if (fee_min < 1000) {
            factor_new_min = 1.0;
        }
        stringRedisHandler.set(lastFactorKey_min, String.valueOf(factor_new_min), 20, TimeUnit.MINUTES);
        SamplerLog.info("factor key is {} factor_new_day is {} factor_hour_new is {} factor_new_min is {}", baseKey, factor_new_day, factor_hour_new, factor_new_min);
        double factor;
        if (fee_hour < 2000) {
            factor = factor_new_day;
            SamplerLog.info("key is {} factor is {}", baseKey, factor);
            return checkFactor(factor);
        }
        if (fee_min < 1000) {
            factor = 0.3 * factor_new_day +
                    0.7 * factor_hour_new;
            SamplerLog.info("key is {} factor is {}", baseKey, factor);
            return checkFactor(factor);
        }
        factor = 0.1 * factor_new_day +
                0.3 * factor_hour_new +
                0.6 * factor_new_min;
        SamplerLog.info("key is {} factor is {}", baseKey, factor);
        return checkFactor(factor);
    }

    private Map<String, JSONObject> getData(String key) {
        List<String> keys = new ArrayList<>();
        //资源位_计划_联盟媒体_联盟广告位
        String key_min = OMD + key + "_" + MINUTE;
        String key_hour = OMD + key + "_" + HOUR;
        String key_day = OMD + key + "_" + DAY;
        keys.add(key_min);
        keys.add(key_hour);
        keys.add(key_day);
        Map<String, String> resultMap = stringRedisHandler.pipelineValue(keys);
        Map<String, JSONObject> result = new HashMap<>(resultMap.size());
        for (String resultKey : resultMap.keySet()) {
            JSONObject data = JSONObject.parseObject(resultMap.get(resultKey));
            result.put(resultKey, data);
        }
        return result;
    }

    private Double calculateRoiFactor(String resourceId, String ideaId, String appId, String slotId) {
        String baseKey = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        String lastRoiFactorKey_min = LASE_ROI_FACTOR + baseKey + "_" + MINUTE;
        String lastRoiFactorKey_hour = LASE_ROI_FACTOR + baseKey + "_" + HOUR;
        String lastRoiFactorKey_day = LASE_ROI_FACTOR + baseKey + "_" + DAY;
        Optional<ResourceIdeaDto> resourceIdeaOptional = resourceIdeaCache.get(Long.valueOf(ideaId)).getNow(Optional.empty());
        if (!resourceIdeaOptional.isPresent()) {
            return null;
        }
        ResourceIdeaDto resourceIdeaDto = resourceIdeaOptional.get();
        if (resourceIdeaDto.getRoiState() == 0 || resourceIdeaDto.getMinRoi() == null) {
            return null;
        }
        double targetRoi = resourceIdeaDto.getMinRoi() / 100;
        Optional<Map<String, JSONObject>> resultMapOptional = omdDataCache.get(baseKey).getNow(Optional.empty());
        if (!resultMapOptional.isPresent()) {
            return null;
        }
        Map<String, JSONObject> resultMap = resultMapOptional.get();
        String key_min = OMD + baseKey + "_" + MINUTE;
        String key_hour = OMD + baseKey + "_" + HOUR;
        String key_day = OMD + baseKey + "_" + DAY;
        JSONObject value_min = resultMap.getOrDefault(key_min, new JSONObject());
        JSONObject value_hour = resultMap.getOrDefault(key_hour, new JSONObject());
        JSONObject value_day = resultMap.getOrDefault(key_day, new JSONObject());
        //获取消耗
        String cpm_fee_min = value_min.getString("x_m_f");
        String cpm_fee_hour = value_hour.getString("x_m_f");
        String cpm_fee_day = value_day.getString("x_m_f");
        String cpc_fee_min = value_min.getString("x_c_f");
        String cpc_fee_hour = value_hour.getString("x_c_f");
        String cpc_fee_day = value_day.getString("x_c_f");
        //广告消耗
        long ad_fee_min = value_min.getString("f") == null ? 0 : Long.parseLong(value_min.getString("f"));
        long ad_fee_hour = value_hour.getString("f") == null ? 0 : Long.parseLong(value_hour.getString("f"));
        long ad_fee_day = value_day.getString("f") == null ? 0 : Long.parseLong(value_day.getString("f"));
        //单位 分
        long fee_min = 0L;
        long fee_hour = 0L;
        long fee_day = 0L;
        if (StringUtils.isNotEmpty(cpm_fee_min)) {
            long cpm_fee = Long.parseLong(cpm_fee_min);
            fee_min = cpm_fee / 10000000;
            if (StringUtils.isNotEmpty(cpc_fee_min)) {
                long cpc_fee = Long.parseLong(cpc_fee_min);
                fee_min = fee_min + cpc_fee / 10000000;
            }
        }
        if (StringUtils.isNotEmpty(cpm_fee_hour)) {
            long cpm_fee = Long.parseLong(cpm_fee_hour);
            fee_hour = cpm_fee / 10000000;
            if (StringUtils.isNotEmpty(cpc_fee_hour)) {
                long cpc_fee = Long.parseLong(cpc_fee_hour);
                fee_hour = fee_hour + cpc_fee / 10000000;
            }
        }
        if (StringUtils.isNotEmpty(cpm_fee_day)) {
            long cpm_fee = Long.parseLong(cpm_fee_day);
            fee_day = cpm_fee / 10000000;
            if (StringUtils.isNotEmpty(cpc_fee_day)) {
                long cpc_fee = Long.parseLong(cpc_fee_day);
                fee_day = fee_day + cpc_fee / 10000000;
            }
        }
        List<String> factorKeys = new ArrayList<>();
        factorKeys.add(lastRoiFactorKey_min);
        factorKeys.add(lastRoiFactorKey_hour);
        factorKeys.add(lastRoiFactorKey_day);
        Map<String, String> factorMap = stringRedisHandler.pipelineValue(factorKeys);
        double factor_min = factorMap.get(lastRoiFactorKey_min) == null ? 1.0 : Double.parseDouble(lastRoiFactorKey_min);
        double factor_hour = factorMap.get(lastRoiFactorKey_hour) == null ? 1.0 : Double.parseDouble(lastRoiFactorKey_hour);
        double factor_day = factorMap.get(lastRoiFactorKey_hour) == null ? 1.0 : Double.parseDouble(lastRoiFactorKey_hour);
        factor_day = checkFactor(factor_day);
        factor_hour = checkFactor(factor_hour);
        factor_min = checkFactor(factor_min);
        if (fee_day == 0L || fee_hour == 0L || fee_min == 0L) {
            return null;
        }

        //计算各个时间维度的factor
        double factor_new_day = factor_day * ((double) ad_fee_day / fee_day - targetRoi) / targetRoi / 2 + 1;
        factor_new_day = checkFactor(factor_new_day);
        if (fee_day < 500) {
            factor_new_day = 1.0;
        }
        stringRedisHandler.set(lastRoiFactorKey_day, String.valueOf(factor_new_day), 20, TimeUnit.MINUTES);
        double factor_hour_new = factor_hour * ((double) ad_fee_hour / fee_hour - targetRoi) / targetRoi / 2 + 1;
        factor_hour_new = checkFactor(factor_hour_new);
        if (fee_hour < 500) {
            factor_hour_new = 1.0;
        }
        stringRedisHandler.set(lastRoiFactorKey_hour, String.valueOf(factor_hour_new), 20, TimeUnit.MINUTES);
        double factor_new_min = factor_min * ((double) ad_fee_min / fee_min - targetRoi) / targetRoi / 2 + 1;
        factor_new_min = checkFactor(factor_new_min);
        if (fee_min < 500) {
            factor_new_min = 1.0;
        }
        stringRedisHandler.set(lastRoiFactorKey_min, String.valueOf(factor_new_min), 20, TimeUnit.MINUTES);

        if (fee_day < 500) {
            return null;
        }
        double factor;
        if (fee_hour < 500) {
            factor = factor_day * factor_new_day;
            return checkFactor(factor);
        }
        if (fee_min < 500) {
            factor = 0.5 * factor_new_day + 0.5 * factor_hour_new;
            return checkFactor(factor);
        }
        factor = 0.4 * factor_new_day + 0.2 * factor_hour_new +
                0.4 * factor_new_min;
        return checkFactor(factor);
    }

    private double checkFactor(double factor) {
        if (factor < 0.8) {
            return 0.8;
        }
        if (factor > 2) {
            return 2.0;
        }
        if (Double.isNaN(factor)) {
            return 1.0;
        }
        return factor;
    }

    public Double getPCtr(String resourceId, String ideaId, String appId, String slotId) {
        String key = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        String redisKey = PCTR + key;
        Optional<String> now = pCTRCache.get(key).getNow(Optional.empty());
        Double pctr = null;
        if (now.isPresent()) {
            pctr = new BigDecimal(now.get()).setScale(7, RoundingMode.UP).doubleValue();
        }
        if (pctr != null && !Double.valueOf(DEFAULT_PCTR).equals(pctr)) {
            stringRedisHandler.set(redisKey, String.valueOf(pctr), 1, TimeUnit.MINUTES);
            return pctr;
        }
        String resourceKey = resourceId + "_" + appId + "_" + slotId;
        String redisResourceKey = PCTR + resourceKey;
        now = pCTRCache.get(resourceKey).getNow(Optional.empty());
        if (now.isPresent()) {
            pctr = new BigDecimal(now.get()).setScale(7, RoundingMode.UP).doubleValue();
            if (!Double.valueOf(DEFAULT_PCTR).equals(pctr)) {
                stringRedisHandler.set(redisResourceKey, String.valueOf(pctr), 1, TimeUnit.MINUTES);
            }
            return pctr;
        }
        return 0.0;
    }

    public Double getPCvr(String resourceId, String ideaId, String appId, String slotId) {
        String key = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        String redisKey = PCVR + key;
        Optional<String> now = pCVRCache.get(key).getNow(Optional.empty());
        Double pcvr = null;
        if (now.isPresent()) {
            pcvr = new BigDecimal(now.get()).setScale(7, RoundingMode.UP).doubleValue();
        }
        if (pcvr != null && pcvr != 0.02) {
            stringRedisHandler.set(redisKey, String.valueOf(pcvr), 1, TimeUnit.MINUTES);
            return pcvr;
        }
        return 0.0;
    }

    public double getFactor(String resourceId, String ideaId, String appId, String slotId) {
        String key = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        Optional<String> now = factorCache.get(key).getNow(Optional.empty());
        double factor = 1.0;
        if (now.isPresent() && StringUtils.isNotBlank(now.get())) {
            if (Double.valueOf(now.get()).isNaN()) {
                return factor;
            }
            factor = new BigDecimal(now.get()).setScale(7, RoundingMode.UP).doubleValue();
        }
        return factor;
    }

    public double getRoiFactor(String resourceId, String ideaId, String appId, String slotId) {
        String key = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        Optional<String> now = roiFactorCache.get(key).getNow(Optional.empty());
        double factor = 1.0;
        if (now.isPresent()) {
            factor = new BigDecimal(now.get()).setScale(7, RoundingMode.UP).doubleValue();
        }
        return factor;
    }

    public ResourceIdeaDto getResourceIdeaByCache(String ideaId) {
        Optional<ResourceIdeaDto> optional = resourceIdeaCache.get(Long.valueOf(ideaId)).getNow(Optional.empty());
        return optional.orElse(null);
    }

    public static void main(String[] args) throws InterruptedException {
        String resourceId = "1120";
        String ideaId = "38640";
        String appId = "testApp";
        String slotId = "10088";
        String baseKey = resourceId + "_" + ideaId + "_" + appId + "_" + slotId;
        BidStatService bidStatService = new BidStatService();
        String r_baseKey = resourceId + "_" + appId + "_" + slotId;
        bidStatService.omdDataCache.put(r_baseKey, CompletableFuture.supplyAsync(() -> {
            Map<String, JSONObject> resultMap = new HashMap<>();
            JSONObject resourceMin = new JSONObject();
            JSONObject resourceHour = new JSONObject();
            JSONObject resourceDay = new JSONObject();
            String r_key_min = OMD + r_baseKey + "_" + MINUTE;
            String r_key_hour = OMD + r_baseKey + "_" + HOUR;
            String r_key_day = OMD + r_baseKey + "_" + DAY;
            resourceMin.put("x_e", "400");
            resourceMin.put("x_m_f", "0");
            resourceMin.put("x_c", "200");
            resourceMin.put("x_c_f", "0");
            resourceMin.put("f", "0");
            resourceMin.put("a_j", "0");
            resourceMin.put("ef_c", "0");
            resultMap.put(r_key_min, resourceMin);
            resourceHour.put("x_e", "400");
            resourceHour.put("x_m_f", "0");
            resourceHour.put("x_c", "200");
            resourceHour.put("x_c_f", "0");
            resourceHour.put("f", "0");
            resourceMin.put("a_j", "0");
            resourceHour.put("ef_c", "0");
            resultMap.put(r_key_hour, resourceHour);
            resourceDay.put("x_e", "400");
            resourceDay.put("x_m_f", "0");
            resourceDay.put("x_c", "300");
            resourceDay.put("x_c_f", "0");
            resourceDay.put("f", "0");
            resourceDay.put("a_j", "0");
            resourceDay.put("ef_c", "0");
            resultMap.put(r_key_day, resourceDay);
            return Optional.of(resultMap);
        }));
        bidStatService.omdDataCache.put(baseKey, CompletableFuture.supplyAsync(() -> {
            Map<String, JSONObject> resultMap = new HashMap<>();
            JSONObject ideaMin = new JSONObject();
            JSONObject ideaHour = new JSONObject();
            JSONObject ideaDay = new JSONObject();
            String key_min = OMD + baseKey + "_" + MINUTE;
            String key_hour = OMD + baseKey + "_" + HOUR;
            String key_day = OMD + baseKey + "_" + DAY;
            ideaMin.put("x_e", "1000");//曝光
            ideaMin.put("x_m_f", "0");
            ideaMin.put("x_c", "100");
            ideaMin.put("x_c_f", "100000000000");
            ideaMin.put("f", "10000");
            ideaMin.put("a_j", "50");
            ideaMin.put("ef_c", "50");
            ideaMin.put("6", "50");
            resultMap.put(key_min, ideaMin);
            ideaHour.put("x_e", "5000");
            ideaHour.put("x_m_f", "0");
            ideaHour.put("x_c", "200");
            ideaHour.put("x_c_f", "100000000000");
            ideaHour.put("f", "10000");
            ideaHour.put("a_j", "100");
            ideaHour.put("ef_c", "100");
            ideaHour.put("6", "100");
            resultMap.put(key_hour, ideaHour);
            ideaDay.put("x_e", "6000");
            ideaDay.put("x_m_f", "0");
            ideaDay.put("x_c", "300");
            ideaDay.put("x_c_f", "100000000000");
            ideaDay.put("f", "10000");
            ideaDay.put("a_j", "100");
            ideaDay.put("ef_c", "200");
            ideaDay.put("6", "200");
            resultMap.put(key_day, ideaDay);
            return Optional.of(resultMap);
        }));
//        Double pctr = bidStatService.getPCtr(resourceId, ideaId, appId, slotId);
//        System.out.println(pctr);
//        Thread.sleep(5000);
//        pctr = bidStatService.getPCtr(resourceId, ideaId, appId, slotId);
//        System.out.println(pctr);

//        Double pCvr = bidStatService.getPCvr(resourceId, ideaId, appId, slotId);
//        System.out.println(pCvr);
//        Thread.sleep(5000);
//        pCvr = bidStatService.getPCvr(resourceId, ideaId, appId, slotId);
//        System.out.println(pCvr);

//        double factor = bidStatService.getFactor(resourceId, ideaId, appId, slotId);
//        System.out.println(factor);
//        Thread.sleep(5000);
//        factor = bidStatService.getFactor(resourceId, ideaId, appId, slotId);
//        System.out.println(factor);

        double roiFactor = bidStatService.getRoiFactor(resourceId, ideaId, appId, slotId);
        System.out.println(roiFactor);
        Thread.sleep(5000);
        roiFactor = bidStatService.getRoiFactor(resourceId, ideaId, appId, slotId);
        System.out.println(roiFactor);
    }


}
