package cn.com.duiba.tuia.dsp.engine.api.dsp.iqiyi;

import cn.com.duiba.spring.boot.starter.dsp.sampler.SamplerLog;
import cn.com.duiba.tuia.dsp.engine.api.dsp.AbstractDspCaller;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.AdxCommonBidRequest;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.DspInfo;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.AdxCommonBidResponse;
import cn.com.duiba.tuia.dsp.engine.api.dsp.iqiyi.bean.IqiyiBid;
import cn.com.duiba.tuia.dsp.engine.api.dsp.iqiyi.convert.IqiyiReqConvert;
import cn.com.duiba.tuia.dsp.engine.api.dsp.iqiyi.convert.IqiyiRespConvert;
import cn.com.duiba.tuia.dsp.engine.api.enums.DspEnum;
import cn.com.duiba.tuia.dsp.engine.api.exception.DspException;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.github.luben.zstd.Zstd;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

import static cn.com.duiba.tuia.dsp.engine.api.exception.DspErrorCode.DSP_HTTP_INVOKE_FAILED;
import static cn.com.duiba.tuia.dsp.engine.api.exception.DspErrorCode.DSP_INVOKE_FAILED;

@Component
@Slf4j
public class IqiyiDspInvoker extends AbstractDspCaller<IqiyiBid.BidResponse, IqiyiBid.BidRequest> {
    @Autowired
    private IqiyiReqConvert iqiyiReqConvert;
    @Autowired
    private IqiyiRespConvert iqiyiRespConvert;
    @Resource
    IqiyiProperties iqiyiProperties;
    @Resource(name = "dspRestTemplate")
    private RestTemplate restTemplate;

    @Override
    protected IqiyiBid.BidResponse invokeDsp(IqiyiBid.BidRequest iqiyiReq) {
        if (Objects.isNull(iqiyiReq)) {
            return null;
        }
        Cat.logMetricForCount("爱奇艺DSP调用");
        try {
            return CatUtils.executeInCatTransaction(() -> this.doHttpInvoke(iqiyiProperties.getUrl(), iqiyiReq), "invokeDSP", "iqiyi");
        } catch (ResourceAccessException e) {
            log.warn("爱奇艺 DSP,http请求异常,{}", e.getMessage());
            throw new DspException(DSP_HTTP_INVOKE_FAILED, this.getDsp(), e);
        } catch (Exception e) {
            log.error("爱奇艺 DSP请求失败,{}", e.getMessage());
            throw new DspException(DSP_INVOKE_FAILED, this.getDsp(), e);
        } catch (Throwable e) {
            log.error("爱奇艺其他调用异常", e);
            return null;
        }
    }

    private IqiyiBid.BidResponse doHttpInvoke(String url, IqiyiBid.BidRequest rtbRequest) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/octet-stream");
        headers.add("Content-Encoding", "zstd");
        headers.add("Accept-Encoding", "zstd");
        byte[] compressData = Zstd.compress(rtbRequest.toByteArray());
        HttpEntity<byte[]> entity = new HttpEntity<>(compressData, headers);
        ResponseEntity<byte[]> responseEntity = null;
        try {
            responseEntity = restTemplate.exchange(url, HttpMethod.POST, entity, byte[].class);
        } catch (Exception e) {
            if (!(e instanceof ResourceAccessException)) {
                SamplerLog.warn("爱奇艺调用异常", e);
            }
            return null;
        }
        HttpStatus statusCode = responseEntity.getStatusCode();
        if (HttpStatus.NO_CONTENT.equals(statusCode)) {
            return null;
        }

        byte[] body = responseEntity.getBody();
        if (body == null) {
            return null;
        }
        int frameContentSize = (int) Zstd.getFrameContentSize(body);
        if (frameContentSize < 0) {
            SamplerLog.info("爱奇艺 response {}", responseEntity);
            return null;
        }
        byte[] array = new byte[frameContentSize];
        Zstd.decompress(array, body);
        try {
            IqiyiBid.BidResponse bidResponse = IqiyiBid.BidResponse.parseFrom(array);
            if (CollectionUtils.isNotEmpty(bidResponse.getBidList())) {
                Cat.logMetricForCount("爱奇艺DSP返回");
                return bidResponse;
            }
        } catch (InvalidProtocolBufferException e) {
            SamplerLog.warn("爱奇艺返回数据解析失败", e);
        }
        return null;
    }

    @Override
    public String priceEncryption(BigDecimal price) {
        try {
            return PriceEncrypterUtils.encryptPrice(price.intValue(),
                                                    iqiyiProperties.getInitializeVector().getBytes(StandardCharsets.UTF_8),
                                                    iqiyiProperties.getEncryptionToken().getBytes(StandardCharsets.UTF_8),
                                                    iqiyiProperties.getIntegrityToken().getBytes(StandardCharsets.UTF_8));
        } catch (Exception e) {
            log.warn("爱奇艺价格加密失败 ", e);
        }
        return null;
    }

    @Override
    protected Integer getDspId() {
        return DspEnum.DSP_16.getDspId();
    }

    @Override
    protected DspEnum getDsp() {
        return DspEnum.DSP_16;
    }

    @Override
    protected String replaceUrl(String price, String url) {
        url = url.replace("${__AUCTION_PRICE__}", price);
        return url;
    }

    @Override
    public IqiyiBid.BidRequest convertReq(AdxCommonBidRequest adxCommonBidRequest, DspInfo dspInfo) throws DspException {
        return iqiyiReqConvert.convert(adxCommonBidRequest, dspInfo);
    }

    @Override
    public AdxCommonBidResponse convertResp(IqiyiBid.BidResponse bidResponse) throws DspException {
        return iqiyiRespConvert.respConvert(bidResponse);
    }
}
