package cn.com.duiba.tuia.dsp.engine.api.util;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.regex.Pattern;

@Slf4j
public class BigDataUaUtils {
    public static final String ANDROID = "android";
    public static final String IPHONE = "iphone";
    private static final String IPAD = "ipad";
    private static final String MAC = "mac";
    private static final String MAC_OS = "mac os";
    private static final String MACOS = "macos";
    private static final String HARMONYOS = "harmonyos";
    private static final String LIKE = "like";
    private static final String IOS = "ios";
    private static final String MACINTOSH = "macintosh";
    private static final Pattern pattern = Pattern.compile("^[a-z0-9_\\-\\.]*$");

    public static UserAgentDto parseUserAgent(String ua) {
        try {
            if (StringUtils.isBlank(ua)) {
                return null;
            }

            if (ua.contains("Mozilla%2F")) {
                ua = ua.substring(0, ua.lastIndexOf("%"));
                ua = URLDecoder.decode(ua, StandardCharsets.UTF_8.name());
            }

            ua = ua.toLowerCase();
            if (!ua.contains("(") || !ua.contains(")")) {
                return null;
            }

            String coreUa = "";
            String[] array = StringUtils.split(ua, "(");

            for (String tmp : array) {
                if (!tmp.contains("mozilla") && !tmp.contains("mozila") && !tmp.contains("zzcios") && (tmp.contains("android") || tmp.contains("iphone") || tmp.contains("ipad") || tmp.contains("mac os"))) {
                    coreUa = StringUtils.split(tmp, ")")[0];
                    break;
                }
            }

            if (StringUtils.isBlank(coreUa)) {
                return null;
            }

            if (coreUa.contains("iphone") && coreUa.contains("mac os")) {
                return getIphoneList(coreUa, "iphone");
            }

            if (coreUa.contains("ipad") && coreUa.contains("mac os")) {
                return getIphoneList(coreUa, "ipad");
            }

            if (coreUa.contains("macintosh") && coreUa.contains("mac os")) {
                return getMacList(coreUa, "mac");
            }

            if (coreUa.contains("harmonyos")) {
                String model = getHarmonyModel(coreUa);
                if (StringUtils.isNotBlank(model)) {
                    return getHarmonyOSList(coreUa, model);
                }
            } else if (coreUa.contains("android")) {
                String model = getAndroidModel(coreUa);
                if (StringUtils.isNotBlank(model)) {
                    coreUa = StringUtils.replace(coreUa, "zh-cn", "");
                    return getAndroidList(coreUa, model);
                }
            }
        } catch (Exception e) {
            log.error("UserAgentUtil parseUserAgent error, ua = {}", ua, e);
        }

        return null;
    }

    private static String check(String osVersion) {
        if (StringUtils.isBlank(osVersion)) {
            return null;
        } else {
            return pattern.matcher(osVersion).matches() ? osVersion.replaceAll("_", ".") : null;
        }
    }

    private static String checkModel(String model) {
        if (StringUtils.isBlank(model)) {
            return null;
        } else {
            if (StringUtils.split(model, "/").length > 1) {
                model = StringUtils.split(model, "/")[0];
            }

            if (model.contains(".")) {
                return null;
            } else {
                return model.contains("android") ? null : model;
            }
        }
    }

    private static String getHarmonyModel(String coreUa) {
        String[] data = StringUtils.split(coreUa, ";");

        for (int i = 0; i < data.length; ++i) {
            if ("harmonyos".equals(data[i].trim()) && i + 1 < data.length) {
                return data[i + 1].trim();
            }
        }

        return null;
    }

    private static String getAndroidModel(String tmpCoreUa) {
        String coreUa = StringUtils.replace(tmpCoreUa, "zh-cn;", "");
        if (coreUa.contains("build")) {
            coreUa = StringUtils.replace(coreUa, "; build", " build");
            String[] split = StringUtils.split(coreUa, ";");

            for (String data : split) {
                if (data.contains("build")) {
                    return checkModel(data.substring(0, data.indexOf("build")).trim());
                }
            }
        } else if (coreUa.contains("bulid")) {
            String[] split = StringUtils.split(coreUa, ";");

            for (String data : split) {
                if (data.contains("bulid")) {
                    return checkModel(data.substring(0, data.indexOf("bulid")).trim());
                }
            }
        } else if (coreUa.contains("miui/")) {
            String[] split = StringUtils.split(coreUa, ";");

            for (String data : split) {
                if (data.contains("miui/")) {
                    return checkModel(data.substring(0, data.indexOf("miui/")).trim());
                }
            }
        } else {
            String[] data = StringUtils.split(coreUa, ";");

            for (int i = 0; i < data.length; ++i) {
                if (data[i].contains("android") && i + 1 < data.length) {
                    return checkModel(data[i + 1].trim());
                }
            }
        }

        return null;
    }

    private static UserAgentDto getIphoneList(String coreUa, String model) {
        String osVersion = "";
        String[] split = StringUtils.split(coreUa, ";");

        for (String data : split) {
            if (data.contains("mac os") && data.contains("like")) {
                osVersion = data.substring(data.indexOf("os") + 2, data.indexOf("like")).trim();
                break;
            }
        }

        UserAgentDto dto = new UserAgentDto();
        dto.setPhoneModel(model);
        dto.setOs("ios");
        dto.setOsVersion(check(osVersion));
        return dto;
    }

    private static UserAgentDto getMacList(String coreUa, String model) {
        String osVersion = "";
        String[] split = StringUtils.split(coreUa, ";");

        for (String data : split) {
            if (data.contains("mac os")) {
                osVersion = data.substring(data.indexOf("mac os x") + 8).trim();
                break;
            }
        }

        if (osVersion.contains("rv:")) {
            osVersion = osVersion.substring(0, osVersion.indexOf("rv:")).trim();
        }

        if (osVersion.contains(" ")) {
            osVersion = StringUtils.replace(osVersion, " ", "_");
        }

        UserAgentDto dto = new UserAgentDto();
        dto.setPhoneModel(model);
        dto.setOs("macos");
        dto.setOsVersion(check(osVersion));
        return dto;
    }

    private static UserAgentDto getHarmonyOSList(String coreUa, String model) {
        String osVersion = "";
        String[] split = StringUtils.split(coreUa, ";");

        for (String data : split) {
            if (data.contains("harmonyos")) {
                osVersion = StringUtils.replace(data, "harmonyos", "").trim();
                break;
            }
        }

        UserAgentDto dto = new UserAgentDto();
        dto.setPhoneModel(model);
        dto.setOs("harmonyos");
        dto.setOsVersion(check(osVersion));
        return dto;
    }

    private static UserAgentDto getAndroidList(String coreUa, String model) {
        String osVersion = "";
        String[] split = StringUtils.split(coreUa, ";");

        for (String data : split) {
            if (data.contains("android")) {
                osVersion = StringUtils.replace(data, "android", "").trim();
                break;
            }
        }

        UserAgentDto dto = new UserAgentDto();
        dto.setPhoneModel(model);
        dto.setOs("android");
        dto.setOsVersion(check(osVersion));
        return dto;
    }

    @Getter
    @Setter
    public static class UserAgentDto{
        private String phoneModel;
        private String os;
        private String osVersion;
    }
}
