package cn.com.duiba.tuia.dsp.engine.api.dsp.alipay;

import cn.com.duiba.spring.boot.starter.dsp.sampler.SamplerLog;
import cn.com.duiba.tuia.dsp.engine.api.dsp.AbstractDspCaller;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.convert.AlipayDspReqConvert;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.convert.AlipayDspRespConvert;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.proto.AlipayDsp;
import cn.com.duiba.tuia.dsp.engine.api.dsp.alipay.proto.AlipayRelDspReq;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.AdxCommonBidRequest;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.req.DspInfo;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.AdxCommonBidResponse;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.CommonCreative;
import cn.com.duiba.tuia.dsp.engine.api.dsp.common.resp.CommonSeatBid;
import cn.com.duiba.tuia.dsp.engine.api.enums.DspEnum;
import cn.com.duiba.tuia.dsp.engine.api.exception.DspException;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.JsonFormat;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.net.SocketTimeoutException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

@Slf4j
@Component
public class AlipayDspInvoker extends AbstractDspCaller<AlipayDsp.BidResponse, AlipayRelDspReq> {
    @Autowired
    private AlipayDspReqConvert alipayDspReqConvert;

    @Autowired
    private AlipayDspRespConvert alipayDspRespConvert;

    @Autowired
    private AlipayProperties alipayProperties;

    @Resource(name = "alipayRestTemplate")
    private RestTemplate alipayRestTemplate;

    @Resource(name = "redis03StringRedisTemplate")
    private StringRedisTemplate stringRedisTemplate03;

    @Override
    protected AlipayDsp.BidResponse invokeDsp(AlipayRelDspReq bidRequest) {
        if (bidRequest == null) {
            return null;
        }
        Cat.logMetricForCount("支付宝DSP调用");
        try {
            return CatUtils.executeInCatTransaction(() -> this.doHttpInvoke(alipayProperties.getUrl(), bidRequest), "invokeDSP", "alipay");
        } catch (Throwable t) {
            Cat.logMetricForCount("支付宝_竞价失败");
            if (!(t instanceof RestClientException || t instanceof SocketTimeoutException)) {
                SamplerLog.warn("支付宝调用异常", t);
            }
        }

        return null;
    }

    private AlipayDsp.BidResponse doHttpInvoke(String url, AlipayRelDspReq bidRequest) {
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/x-protobuf");
        headers.add("Content-Encoding", "gzip");
        headers.add("Accept", "application/x-protobuf");
        headers.add("Accept-Encoding", "gzip");
        headers.add("keep-alive", "true");
        //使用gzip压缩
        byte[] requestBytes = gzipCompress(bidRequest.getBidRequest().toByteArray());
        if (requestBytes == null) {
            return null;
        }
        HttpEntity<byte[]> httpEntity = new HttpEntity<>(requestBytes, headers);
        ResponseEntity<byte[]> resEntity = null;
        try {
            resEntity = alipayRestTemplate.exchange(url, HttpMethod.POST, httpEntity, byte[].class);
            SamplerLog.info("支付宝 dsp请求 {} DSP返回 {} url is {}", httpEntity, resEntity, url);
        } catch (Exception e) {
            if (!(e instanceof RestClientException)) {
                log.warn("支付宝调用异常 response is {}", resEntity, e);
            }
        }
        if (resEntity == null) {
            return null;
        }
        byte[] body = resEntity.getBody();
        AlipayDsp.BidResponse response = null;
        if (body != null) {
            //响应头包含gzip 再解压
            String first = resEntity.getHeaders().getFirst("Content-Encoding");
            if ("gzip".equals(first)) {
                byte[] decompress = gzipDecompress(body);
                if (decompress != null) {
                    try {
                        response = AlipayDsp.BidResponse.parseFrom(decompress);
                    } catch (Exception e) {
                        //
                        return null;
                    }
                }

            } else {
                try {
                    response = AlipayDsp.BidResponse.parseFrom(body);
                } catch (InvalidProtocolBufferException e) {
                    //
                    return null;
                }
            }
        }
        if (response != null) {
            try {
                SamplerLog.info("支付宝dsp返回 {}", JsonFormat.printer().print(response));
            } catch (InvalidProtocolBufferException e) {
                //ignore
            }
        }
        if (response != null && CollectionUtils.isNotEmpty(response.getSeatBidList()) && CollectionUtils.isNotEmpty(response.getSeatBidList().get(0).getBidList())) {
            Cat.logMetricForCount("支付宝DSP返回");
            return response;
        }
        if (response != null && response.getNbr() == 9) {
            //到这个时间戳过期
            stringRedisTemplate03.opsForValue().set("alipay_nbr9_" + bidRequest.getDeviceId(), "", response.getColdEndTime(), TimeUnit.SECONDS);
        }
        return null;

    }

    @Override
    public String priceEncryption(BigDecimal price) {
        try {
            SecretKeySpec secretKey = new SecretKeySpec("ysDusbwFux32RAfJ".getBytes(StandardCharsets.UTF_8), "AES");
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            byte[] bytes = cipher.doFinal(price.toString().getBytes(StandardCharsets.UTF_8));
            //2进制转换成16进制
            StringBuilder sb = new StringBuilder();
            for (byte aByte : bytes) {
                String hex = Integer.toHexString(aByte & 0xFF);
                if (hex.length() == 1) {
                    hex = '0' + hex;
                }
                sb.append(hex);
            }
            return sb.toString();
        } catch (Exception e) {
            log.warn("支付宝加密失败", e);
        }
        return null;
    }

    @Override
    protected Integer getDspId() {
        return DspEnum.DSP_25.getDspId();
    }

    @Override
    protected DspEnum getDsp() {
        return DspEnum.DSP_25;
    }

    @Override
    protected String replaceUrl(String price, String url) {
        if (url.contains("${AUCTION_PRICE}") && price != null) {
            return url.replace("${AUCTION_PRICE}", price);
        }
        return url;
    }

    @Override
    public AlipayRelDspReq convertReq(AdxCommonBidRequest adxCommonBidRequest, DspInfo dspInfo) throws DspException {
        AlipayDsp.BidRequest request = alipayDspReqConvert.convert(adxCommonBidRequest, dspInfo);
        AlipayRelDspReq alipayRelDspReq = new AlipayRelDspReq();
        alipayRelDspReq.setBidRequest(request);
        return alipayRelDspReq;
    }

    @Override
    public AdxCommonBidResponse convertResp(AlipayDsp.BidResponse bidResponse) throws DspException {
        if (bidResponse == null) {
            return null;
        }
        AdxCommonBidResponse convert = alipayDspRespConvert.convert(bidResponse);
        convert.setDspId(getDspId());
        CommonSeatBid commonSeatBid = convert.getCommonSeatBidList().get(0);
        CommonCreative commonCreative = commonSeatBid.getCommonCreative();
        String deepLinkUrl = commonCreative.getDeepLinkUrl();
        String taskId = getTaskId(deepLinkUrl);
        commonCreative.setDspTaskId(taskId);
        return convert;
    }

    private String getTaskId(String deepLinkUrl) {
        if (StringUtils.isBlank(deepLinkUrl)) {
            return null;
        }
        try {
            deepLinkUrl = URLDecoder.decode(deepLinkUrl, StandardCharsets.UTF_8.toString());
            if (deepLinkUrl.contains("waitouScene")) {
                String[] params = deepLinkUrl.split("&");
                for (String param : params) {
                    if (param.startsWith("waitouScene")) {
                        String[] keyValue = param.split("=");
                        if (keyValue.length == 2) {
                            return keyValue[1];
                        }
                    }
                }
            }
        } catch (Exception e) {
            //
        }
        return null;
    }

    //gzip压缩
    private byte[] gzipCompress(byte[] data) {
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream(); GZIPOutputStream gzipOS = new GZIPOutputStream(bos)) {
            gzipOS.write(data);
            gzipOS.finish();
            return bos.toByteArray();
        } catch (IOException e) {
            log.warn("gzip压缩失败", e);
        }
        return null;
    }

    //gzip解压
    private byte[] gzipDecompress(byte[] data) {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(data); GZIPInputStream gis = new GZIPInputStream(bis); ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
            byte[] buffer = new byte[4096];
            int len;
            // 4. 循环读取解压后的数据，写入输出流
            while ((len = gis.read(buffer)) != -1) {
                bos.write(buffer, 0, len);
            }
            // 5. 将缓存的解压数据转为字节数组返回
            return bos.toByteArray();
        } catch (IOException e) {
            log.warn("gzip解压失败", e);
        }
        return null;
    }

//    public static void main(String[] args) throws Exception {
//        String json = "{\"app\":{\"verion\":\"7.11.0.711002\",\"name\":\"快影\",\"bundle\":\"com.kwai.videoeditor\"},\"id\":\"10153-343c64a2-70e5-401c-be58-384945a264c4\",\"imp\":[{\"adType\":1,\"tagId\":\"2001\",\"id\":\"10153-676c945d-09be-4efe-b274-b6c35793ec05\",\"adStyles\":[1,2,3,4],\"asset\":[{\"templateId\":\"2_1\"}],\"cpmBidFloor\":\"17\",\"bidType\":0}],\"device\":{\"deviceType\":4,\"oaidMd5\":\"03a9679a12cb29f561118fbdbb6f3f72\",\"carrier\":1,\"osv\":\"13\",\"os\":\"android\",\"ip\":\"39.144.147.200\",\"model\":\"V2279A\",\"ua\":\"Mozilla/5.0 (Linux; Android 13; V2279A Build/TP1A.220624.014; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/121.0.6167.212 KsWebView/1.8.121.843 (rel) Mobile Safari/537.36\",\"make\":\"vivo\",\"connectionType\":7,\"oaid\":\"64cc0ee97b0b25c1da63cd108ea890015e9fa12043a99789f117a00b576fe1f7\"},\"timeout\":200}";
//        AlipayDspInvoker alipayDspInvoker = new AlipayDspInvoker();
//        AlipayDsp.BidRequest.Builder builder = AlipayDsp.BidRequest.newBuilder();
//        JsonFormat.parser().merge(json, builder);
//        AlipayRelDspReq alipayRelDspReq = new AlipayRelDspReq();
//        alipayRelDspReq.setBidRequest(builder.build());
//        alipayRestTemplate = new RestTemplate();
//        AlipayDsp.BidResponse bidResponse = alipayDspInvoker.doHttpInvoke("https://ugapi.alipay.com/dsp/bidding/tuiartb", alipayRelDspReq);
//        System.out.println(bidResponse);
//    }
}
