package cn.com.duiba.duiba.base.service.api.duibaboot.oss.template;

import cn.com.duiba.duiba.base.service.api.duibaboot.oss.template.operation.MultipartTask;
import cn.com.duiba.duiba.base.service.api.duibaboot.oss.template.operation.MultipartTaskImpl;
import cn.com.duiba.duiba.base.service.api.duibaboot.oss.template.util.OssUtils;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.utils.IOUtils;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectResult;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.InitializingBean;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;

/**
 * @author dugq
 * @date 2021/7/21 4:03 下午
 */
@Data
@Slf4j
public abstract class AbstractOssTemplate implements OssTemplate, InitializingBean {
    public static final String canOverwriteKey = "canOverWrite";
    /**
     * OSS 客户端
     */
    protected OSSClient ossClient;

    /**
     * bucket的端点，也是访问域名。
     */
    protected String endPoint;

    /**
     * 当前客户端操作的分区名称
     */
    protected String bucketName;

    /**
     * bucket的加速访问域名
     */
    protected String cdnDomain;

    /**
     * bucket的加速域名
     */
    protected String cdnHost;

    /**
     * bucket的访问域名附带协议： http://endPoint.bucketName
     */
    protected String bucketDomain;

    /**
     * bucket的访问域名 ： endPoint.bucketName
     */
    protected String bucketHost;

    @Resource
    private ExecutorService ossExecutorService;

    @Override
    public MultipartTask createMultipartTask() {
        return new MultipartTaskImpl(this);
    }

    @Override
    public <R> R downloadStream(String objectName, @NotNull Function<OSSObject, R> callback) {
        if (StringUtils.isBlank(objectName)) {
            return null;
        }
        String key;
        URI uri = URI.create(objectName);
        //兼容传入的是文件地址的场景，对域名进行校验后，再进行下载操作
        if (Objects.nonNull(uri.getHost())) {
            // 1、cdn加速域名拼接的地址 或者
            if (Objects.equals(uri.getHost(), cdnHost)) {
                //objectName要求不能以/开头
                key = uri.getPath().replaceFirst("/", "");
            }
            // 2、oss endpoint拼接的地址。同时也包含对bucket的验证
            else if (Objects.equals(uri.getHost(), bucketHost)) {
                key = uri.getPath().replaceFirst("/", "");
            }
            // 3、其他一律不允许下载
            else {
                log.error("非法的Oss下载地址：{}.  this template cdnDomain={} endpoint={} bucket={}", objectName, cdnDomain, endPoint, bucketName);
                return null;
            }

        } else {
            //没有加域名的，默认直接进行下载
            key = objectName;
        }
        OssUtils.validatorObjectName(key);
        final OSSObject object = ossClient.getObject(bucketName, key);
        try {
            return callback.apply(object);
        } finally {
            final InputStream objectContent = object.getObjectContent();
            try {
                objectContent.close();
            } catch (IOException e) {
                //ignore.如论如何，都得把inputStream关闭了。
                log.warn("close input stream error after upload input stream oss to oss!", e);
            }
        }
    }

    @Override
    public String getObjectName(String ossUrl) {
        if (StringUtils.isBlank(ossUrl)) {
            return ossUrl;
        }
        URI uri = URI.create(ossUrl);
        //没有域名的情况时，直接返回path
        if (Objects.isNull(uri.getHost())) {
            if (uri.getPath().startsWith("/")) {
                return uri.getPath().replaceFirst("/", "");
            }
            return uri.getPath();
        } else if (StringUtils.equals(uri.getHost(), cdnHost)) {
            return StringUtils.substringAfterLast(ossUrl, cdnHost + "/");
        } else if (StringUtils.equals(uri.getHost(), bucketHost)) {
            return StringUtils.substringAfterLast(ossUrl, bucketHost + "/");
        } else {
            return null;
        }
    }

    @Override
    public String uploadStream(InputStream inputStream, String objectName, ObjectMetadata meta) {
        OssUtils.validatorObjectName(objectName);
        final PutObjectResult result = ossClient.putObject(bucketName, objectName, inputStream, meta);
        if (OssUtils.successful(result)) {
            return getUrl(objectName);
        }
        log.error("上传oss发生错误.object name = {} msg = {}", objectName, OssUtils.getErrorMessage(result));
        return null;
    }

    @Override
    public String uploadFile(File file, String objectName, ObjectMetadata meta) {
        OssUtils.validatorObjectName(objectName);
        final PutObjectResult result = ossClient.putObject(bucketName, objectName, file, meta);
        if (OssUtils.successful(result)) {
            return getUrl(objectName);
        }
        log.error("上传oss发生错误.object name = {} msg = {}", objectName, OssUtils.getErrorMessage(result));
        return null;
    }

    @Override
    public String uploadString(String content, String objectName, ObjectMetadata meta) {
        if (StringUtils.isBlank(content)) {
            log.error("上传oss发生错误:文件内容为空.object name = {}", objectName);
            return null;
        }
        final byte[] bytes = content.getBytes();
        if (meta.getContentLength() == 0) {
            meta.setContentLength(bytes.length);
        }
        return uploadStream(new ByteArrayInputStream(bytes), objectName, meta);
    }

    protected String getBucketUri() {
        return bucketDomain + "/";
    }

    @Override
    public boolean doesObjectExist(String url) {
        final String objectName = getObjectName(url);
        if (StringUtils.isBlank(url)) {
            throw new UnsupportedOperationException("objectName 非法！objectName=" + url);
        }
        return ossClient.doesObjectExist(bucketName, objectName);
    }

    @Override
    public void deleteObject(String url) {
        final String objectName = getObjectName(url);
        if (StringUtils.isBlank(url)) {
            throw new UnsupportedOperationException("objectName 非法！objectName=" + url);
        }
        ossClient.deleteObject(bucketName, objectName);
    }


    @Override
    public String getUrlWithOutScheme(String objectName) {
        return getUrl(objectName);
    }

    @Override
    public String getCDNHost() {
        return cdnHost;
    }

    @Override
    public String getContentByObjectName(String objectName) {
        if (StringUtils.isBlank(objectName)) {
            return null;
        }
        // 文件不存在
        if (!doesObjectExist(objectName)) {
            return null;
        }
        try {
            OSSObject ossObject = ossClient.getObject(bucketName, objectName);
            return IOUtils.readStreamAsString(ossObject.getObjectContent(), StandardCharsets.UTF_8.name());
        } catch (Exception e) {
            log.error("getContentByObjectName error! objectName={}", objectName, e);
            return null;
        }
    }
}
