package cn.com.duiba.duiba.base.service.api.duibaboot.oss.template.metadata;

import cn.com.duiba.duiba.base.service.api.duibaboot.oss.template.AbstractOssTemplate;
import com.aliyun.oss.model.ObjectMetadata;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.util.Date;

/**
 * @author dugq
 * @date 2021/7/23 11:37 上午
 */
@Slf4j
public class ObjectMetadataHelper {

    public static final String USER_DATE_PREFIX = "x-oss-meta-";

    private ObjectMetadata objectMetadata;

    public static ObjectMetadataHelper create() {
        ObjectMetadataHelper builder = new ObjectMetadataHelper();
        builder.objectMetadata = new ObjectMetadata();
        return builder;
    }

    /**
     * 这是文件类型。对象http请求的contentType
     * 对于浏览器可预览的文件，此属性比较重要，浏览器会根据httpResponse的contentType做出相应的处理。
     *
     * @param contentType 文件类型
     */
    public ObjectMetadataHelper setContentType(String contentType) {
        objectMetadata.setContentType(contentType);
        return this;
    }

    /**
     * 文件加密算法
     * 对于加密或者压缩的文件，浏览器预览时缺少此属性，将无法预览
     *
     * @param encoding 加密算法
     */
    public ObjectMetadataHelper setContentEncoding(String encoding) {
        objectMetadata.setContentEncoding(encoding);
        return this;
    }

    /**
     * 设置文件为强制下载。关闭浏览器预览模式
     *
     * @param fileName 下载后的文件名称
     */
    public ObjectMetadataHelper setAsDownloadFile(String fileName) {
        if (StringUtils.isBlank(fileName)) {
            throw new UnsupportedOperationException("强制下载模式，请指定文件名！");
        }
        objectMetadata.setContentDisposition("attachment; filename=\"" + fileName + "\"");
        return this;
    }

    /**
     * 设置文件大小。
     *
     * @param contentLength 文件大小
     */
    public ObjectMetadataHelper setContentLength(Long contentLength) {
        objectMetadata.setContentLength(contentLength);
        return this;
    }

    /**
     * 设置过期时间
     *
     * @param expirationTime 过期时间
     */
    public ObjectMetadataHelper setExpirationTime(Date expirationTime) {
        objectMetadata.setExpirationTime(expirationTime);
        return this;
    }

    /**
     * 追加自定义的元空间数据
     *
     * @param key   key 会自动补充{@link #USER_DATE_PREFIX}在前面
     * @param value value
     */
    public ObjectMetadataHelper addUserDate(String key, String value) {
        if (StringUtils.isBlank(key) || StringUtils.isBlank(value)) {
            throw new UnsupportedOperationException("自定义属性不可以为空");
        }
        if (!StringUtils.startsWith(key, USER_DATE_PREFIX)) {
            key = USER_DATE_PREFIX + key;
        }
        objectMetadata.getUserMetadata().put(key, value);
        return this;
    }

    /**
     * 设置文件为禁止覆盖。
     * 默认为允许。
     * 开启禁止覆盖后，请求有100QPS显示
     */
    public ObjectMetadataHelper forbidOverWrite() {
        objectMetadata.setHeader("x-oss-forbid-overwrite", true);
        return this;
    }

    /**
     * 设置文件允许覆盖。
     */
    public ObjectMetadataHelper canOverWrite() {
        objectMetadata.addUserMetadata(AbstractOssTemplate.canOverwriteKey, "true");
        return this;
    }

    /**
     * 获取元空间对象
     *
     * @return 元空间对象
     */
    public ObjectMetadata getInstance() {
        if (objectMetadata.getContentLength() == 0) {
            log.info("我们推荐元空间数据对象含有文件大小，以保证文件的完整性");
        }
        return objectMetadata;
    }
}
