package cn.com.duiba.duiba.base.service.api.id.generator;

import cn.com.duiba.duiba.base.service.api.id.generator.bean.IdGeneratorKey;
import cn.com.duiba.duiba.base.service.api.id.generator.configuration.IdGeneratorProperties;
import cn.com.duiba.duiba.base.service.api.id.generator.configuration.Scene;
import cn.com.duiba.duiba.base.service.api.id.generator.exception.IdGeneratorException;
import cn.com.duiba.duiba.base.service.api.id.generator.service.IdGeneratorCache;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.scheduling.annotation.Scheduled;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author lizhi
 * @date 2024/12/30 13:49
 */
@Slf4j
public class IdGeneratorClient {

    @Resource(name = "redisTemplate")
    private RedisAtomicClient redisAtomicClient;
    @Resource
    private IdGeneratorProperties idGeneratorProperties;

    private final LoadingCache<IdGeneratorKey, IdGeneratorCache> cache = Caffeine.newBuilder().build(new CacheLoader<IdGeneratorKey, IdGeneratorCache>() {
        @Override
        public IdGeneratorCache load(@NonNull IdGeneratorKey key) {
            IdGeneratorCache idGeneratorCache = new IdGeneratorCache(redisAtomicClient, idGeneratorProperties);
            idGeneratorCache.setScene(idGeneratorProperties.getScenes().get(key.getSceneKey()));
            idGeneratorCache.setKey(key);
            idGeneratorCache.init();
            return idGeneratorCache;
        }
    });

    /**
     * 生成一个全局唯一Id
     * @param sceneKey 场景标识
     * @return 唯一ID
     */
    public Long idGenerator(String sceneKey) {
        return generator(sceneKey,new Date());
    }

    private Long generator(String sceneKey, Date time){
        Scene scene = idGeneratorProperties.getScenes().get(sceneKey);
        IdGeneratorCache idGeneratorCache = getCache(sceneKey, scene, time);
        return Objects.requireNonNull(idGeneratorCache).get();
    }
    
    private IdGeneratorCache getCache(String sceneKey, Scene scene, Date time) {
        if(Objects.isNull(scene)){
            throw new IdGeneratorException("发号场景["+sceneKey+"]未配置");
        }
        String formatTime = scene.getTimeLevel().formatTime(time);
        IdGeneratorKey key = new IdGeneratorKey();
        key.setSceneKey(sceneKey);
        key.setTime(formatTime);
        return cache.get(key);
    }

    /**
     * 扫描过期的发号Cache
     */
    @Scheduled(cron = "0 0 1/1 * * ?")
    public void scanCache(){
        log.info("IdGenerator, scanCache, cache.size={}", cache.asMap().size());
        Set<IdGeneratorKey> keys = Sets.newHashSet();
        for(Map.Entry<IdGeneratorKey,IdGeneratorCache> entry:cache.asMap().entrySet()){
            IdGeneratorCache idGeneratorCache = entry.getValue();
            if(idGeneratorCache.isInvalidate()){
                keys.add(entry.getKey());
            }
        }
        for (IdGeneratorKey key:keys){
            cache.invalidate(key);
        }
    }

    @PostConstruct
    public void loadInit() {
        Map<String, Scene> scenes = idGeneratorProperties.getScenes();
        if (scenes == null || scenes.isEmpty()) {
            return;
        }
        log.info("IdGenerator, init, scenes.size={}", scenes.size());
        for(Map.Entry<String, Scene> entry : idGeneratorProperties.getScenes().entrySet()){
            getCache(entry.getKey(), entry.getValue(), new Date());
            log.info("IdGenerator, init key={}", entry.getKey());
        }
    }
}
