package cn.com.duiba.duiba.base.service.api.id.generator.service;

import cn.com.duiba.duiba.base.service.api.id.generator.bean.IdGeneratorKey;
import cn.com.duiba.duiba.base.service.api.id.generator.configuration.IdGeneratorProperties;
import cn.com.duiba.duiba.base.service.api.id.generator.configuration.Scene;
import cn.com.duiba.duiba.base.service.api.id.generator.configuration.TimeLevel;
import cn.com.duiba.duiba.base.service.api.id.generator.exception.IdGeneratorException;
import cn.com.duiba.wolf.redis.RedisAtomicClient;
import cn.com.duiba.wolf.redis.RedisLock;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Date;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author lizhi
 * @date 2024/12/30 14:09
 */
@Slf4j
@Data
public class IdGeneratorCache {

    private RedisAtomicClient redisAtomicClient;
    
    private IdGeneratorProperties idGeneratorProperties;

    private Scene scene;
    
    private IdGeneratorKey key;

    private ExecutorService executorService;

    private int loadingThreshold;

    private final LinkedBlockingQueue<Long> idQueue = new LinkedBlockingQueue<>();

    private static final int WARN_TIME = 500;
    
    private static final int INFO_TIME = 100;

    public IdGeneratorCache(RedisAtomicClient redisAtomicClient, IdGeneratorProperties idGeneratorProperties) {
        this.redisAtomicClient = redisAtomicClient;
        this.idGeneratorProperties = idGeneratorProperties;
    }

    public void init(){
        int nThreads = Math.max(idGeneratorProperties.getScenes().size(),1);
        this.executorService = new ThreadPoolExecutor(nThreads, nThreads,
                0L, TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("id-generator"));
        loadingThreshold = scene.getStep() / 2;
        initStart();
        loadId();
    }


    public boolean isInvalidate(){
        String time = scene.getTimeLevel().formatTime(new Date());
        return !StringUtils.equals(key.getTime(),time);
    }
    
    public Long get() throws IdGeneratorException {
        if(Objects.equals(1, scene.getStep())){
            return incrBy(getRedisKey(), 1);
        }
        long start = System.currentTimeMillis();
        try {
            if (idQueue.isEmpty()) {
                // 队列已经没有ID了，同步加载
                loadId();
            } else if(idQueue.size() < loadingThreshold){
                //当发放队列保有量小于阈值时，异步进行补充
                executorService.execute(this::loadId);
            }
        } finally {
            log(start, "loadId");
        }
        start = System.currentTimeMillis();
        try {
            return idQueue.poll(1, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IdGeneratorException("发号超时");
        } finally {
            log(start, "poll");
        }
    }
    
    private void log(long start, String biz) {
        long cost = System.currentTimeMillis() - start;
        if (cost > WARN_TIME) {
            log.warn("IdGenerator, {} limit {} ms, cost={}ms", biz, WARN_TIME, cost);
        } else if (cost > INFO_TIME){
            log.info("IdGenerator, {} limit {} ms, cost={}ms", biz, INFO_TIME, cost);
        }
    }
    
    private String getRedisKey() {
        return key.getSceneKey() + "_" + key.getTime();
    }
    
    private String getLockKey() {
        return key.getSceneKey();
    }

    private synchronized void loadId(){
        if(Objects.equals(1L, scene.getStep())){
            return;
        }
        if (idQueue.size() >= scene.getStep()) {
            // 已经加载过了
            return;
        }
        long delta = scene.getStep();
        Long result = incrBy(getRedisKey(), delta);
        if (result == null || result < delta) {
            log.error("IdGenerator, incr error, key={}, delta={}, result={}", key.getSceneKey(), delta, result);
            return;
        }
        for (long i = result - delta; i < result; i++) {
            idQueue.add(i + 1);
        }
    }
    
    private void initStart() {
        if (scene.getStartId() == 0) {
            return;
        }
        try(RedisLock lock = redisAtomicClient.getLock(getLockKey(), 10, 5, 50)) {
            if (lock == null) {
                // 长时间获取不到分布式锁，也触发初始化，结果也只是会浪费一些id
                log.warn("IdGenerator, initStart lock is null, key={}, ", key.getSceneKey());
            }
            doInitStart();
        } catch (Exception e) {
            // 异常，也触发初始化，结果也只是会浪费一些id
            log.error("IdGenerator, initStart error, key={}, ", key.getSceneKey(), e);
            doInitStart();
        }
    }
    
    private void doInitStart() {
        String redisKey = getRedisKey();
        Long exist = redisAtomicClient.getLong(redisKey);
        if (exist != null && exist >= scene.getStartId()) {
            return;
        }
        long delta = exist == null ? scene.getStartId() : scene.getStartId() - exist;
        Long result = incrBy(redisKey, delta);
        if (result == null || result < scene.getStartId()) {
            throw new IdGeneratorException("发号场景["+key.getSceneKey()+"]初始化启始值失败");
        }
        log.info("IdGenerator, start init, startId={}, redisKey={}, delta={}", scene.getStartId(), redisKey, delta);
    }

    private Long incrBy(String redisKey, long delta) {
        long start = System.currentTimeMillis();
        try {
            return redisAtomicClient.incrBy(redisKey, delta, getTimeout() + 1, TimeUnit.HOURS);
        } catch (Throwable e) {
            log.error("IdGenerator, incr error, redisKey={}, delta={}", redisKey, delta);
            return null;
        } finally {
            log.info("IdGenerator, incr cost={}", System.currentTimeMillis() - start);
        }
    }
    
    private long getTimeout() {
        TimeLevel timeLevel = scene.getTimeLevel();
        switch (timeLevel) {
            case NONE:
                // 100*366*24
                return 878400;
            case YEAR:
                // 366*24
                return 8784;
            case MONTH:
                // 31*24
                return 744;
            default:
                return 24;
        }
    }
}
