package cn.com.duiba.cat.model.configuration.client.entity;

import cn.com.duiba.cat.model.configuration.client.BaseEntity;
import cn.com.duiba.cat.model.configuration.client.Constants;
import cn.com.duiba.cat.model.configuration.client.IVisitor;
import cn.com.duiba.cat.model.configuration.client.transform.DefaultXmlBuilder;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class ClientConfig extends BaseEntity<ClientConfig> {

    private String              mode;
    private boolean             enabled        = true;
    private Boolean             dumpLocked;
    private List<Server>        servers        = new ArrayList<Server>();
    private Map<String, Domain> domains        = new LinkedHashMap<String, Domain>();
    private Bind                bind;
    private List<Property>      properties     = new ArrayList<Property>();
    private String              domain         = "unknown";
    private int                 maxMessageSize = 5000;

    public ClientConfig() {
    }

    public ClientConfig(String domain) {
        if (domain != null) {
            this.domain = domain;
        }
    }

    @Override
    public void accept(IVisitor visitor) {
        visitor.visitConfig(this);
    }

    public ClientConfig addDomain(Domain domain) {
        domains.put(domain.getId(), domain);
        return this;
    }

    public ClientConfig addProperty(Property property) {
        properties.add(property);
        return this;
    }

    public ClientConfig addServer(Server server) {
        servers.add(server);
        return this;
    }

    public Domain findDomain(String id) {
        return domains.get(id);
    }

    public Server findServer(String ip) {
        for (Server server : servers) {
            if (!server.getIp().equals(ip)) {
                continue;
            }

            return server;
        }

        return null;
    }

    @Override
    public void mergeAttributes(ClientConfig other) {
        assertAttributeEquals(other, Constants.ENTITY_CONFIG, Constants.ATTR_DOMAIN, domain, other.getDomain());

        if (other.getMode() != null) {
            mode = other.getMode();
        }

        enabled = other.isEnabled();

        if (other.getDumpLocked() != null) {
            dumpLocked = other.getDumpLocked();
        }

        maxMessageSize = other.getMaxMessageSize();
    }

    public boolean removeDomain(String id) {
        if (domains.containsKey(id)) {
            domains.remove(id);
            return true;
        }

        return false;
    }

    public boolean removeServer(String ip) {
        int len = servers.size();

        for (int i = 0; i < len; i++) {
            Server server = servers.get(i);

            if (!server.getIp().equals(ip)) {
                continue;
            }

            servers.remove(i);
            return true;
        }

        return false;
    }

    @Override
    public String toString() {
        return new DefaultXmlBuilder().buildXml(this);
    }

    public String getMode() {
        return mode;
    }

    public ClientConfig setMode(String mode) {
        this.mode = mode;
        return this;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public ClientConfig setEnabled(boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    public Boolean getDumpLocked() {
        return dumpLocked;
    }

    public ClientConfig setDumpLocked(Boolean dumpLocked) {
        this.dumpLocked = dumpLocked;
        return this;
    }

    public List<Server> getServers() {
        return servers;
    }

    public void setServers(List<Server> servers) {
        this.servers = servers;
    }

    public Map<String, Domain> getDomains() {
        return domains;
    }

    public void setDomains(Map<String, Domain> domains) {
        this.domains = domains;
    }

    public Bind getBind() {
        return bind;
    }

    public ClientConfig setBind(Bind bind) {
        this.bind = bind;
        return this;
    }

    public List<Property> getProperties() {
        return properties;
    }

    public void setProperties(List<Property> properties) {
        this.properties = properties;
    }

    public String getDomain() {
        return domain;
    }

    public ClientConfig setDomain(String domain) {
        this.domain = domain;
        return this;
    }

    public int getMaxMessageSize() {
        return maxMessageSize;
    }

    public ClientConfig setMaxMessageSize(int maxMessageSize) {
        this.maxMessageSize = maxMessageSize;
        return this;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;

        if (o == null || getClass() != o.getClass()) return false;

        ClientConfig that = (ClientConfig) o;

        return new EqualsBuilder().append(enabled, that.enabled).append(maxMessageSize, that.maxMessageSize).append(mode, that.mode).append(dumpLocked, that.dumpLocked).append(servers, that.servers).append(domains, that.domains).append(bind, that.bind).append(properties, that.properties).append(domain, that.domain).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37).append(mode).append(enabled).append(dumpLocked).append(servers).append(domains).append(bind).append(properties).append(domain).append(maxMessageSize).toHashCode();
    }
}
