package cn.com.duiba.galaxy.console.controller;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.dto.appextra.AppConfigDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppExtraService;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.devops.web.api.RemoteAmiyaProjectService;
import cn.com.duiba.devops.web.model.amiya.AmiyaSimpleProjectRes;
import cn.com.duiba.devops.web.model.common.PageRes;
import cn.com.duiba.devops.web.param.amiya.ProjectSearchDto;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.manager.CommonBackendManager;
import cn.com.duiba.galaxy.console.manager.FileUploadManager;
import cn.com.duiba.galaxy.console.manager.ProjectManager;
import cn.com.duiba.galaxy.console.manager.RegionManager;
import cn.com.duiba.galaxy.console.model.param.CheckParam;
import cn.com.duiba.galaxy.console.model.param.IdentityQueryParam;
import cn.com.duiba.galaxy.console.model.param.project.DayanQueryParam;
import cn.com.duiba.galaxy.console.model.vo.EnumVo;
import cn.com.duiba.galaxy.console.model.vo.IdentityVo;
import cn.com.duiba.galaxy.console.model.vo.RegionVo;
import cn.com.duiba.galaxy.console.model.vo.SsoUserVo;
import cn.com.duiba.galaxy.console.model.vo.project.DayanSearchVo;
import cn.com.duiba.galaxy.console.model.vo.project.PageDayanProjectVo;
import cn.com.duiba.galaxy.console.model.vo.project.SimpleAppInfoVo;
import cn.com.duiba.galaxy.core.activity.CheckMode;
import cn.com.duiba.galaxy.core.activity.Configable;
import cn.com.duiba.galaxy.core.enums.ConfigType;
import cn.com.duiba.galaxy.core.enums.PlaywayTypeEnum;
import cn.com.duiba.galaxy.sdk.base.JsonResult;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.domain.dto.AdminInfoDto;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;


/**
 * 通用工具
 *
 * @author zhangdaqing
 * @date 2022/10/09
 */
@RestController
@RequestMapping("/common")
@Slf4j
public class CommonBackendController {

    @Resource
    private FileUploadManager fileUploadManager;

    @Resource
    private ProjectManager projectManager;

    @Resource
    private RegionManager regionManager;

    @Resource
    private RemoteAppService remoteAppService;

    @Resource
    private RemoteAmiyaProjectService remoteAmiyaProjectService;

    @Resource
    private RemoteAppExtraService remoteAppExtraServiceNew;

    @Resource
    private RemoteAdminService remoteAdminService;

    @Autowired
    private CommonBackendManager commonBackendManager;

    /**
     * 查询玩法类型
     *
     * @return {@link List}<{@link JSONObject}>
     */
    @GetMapping(name = "查询玩法类型", value = "/listPlayways")
    public List<JSONObject> getAttributesByConditions() {
        return Arrays.stream(PlaywayTypeEnum.values()).map(value -> new JSONObject().fluentPut(value.getCode().toString(), value.getDesc())).collect(Collectors.toList());
    }

    /**
     * 查询appIds信息
     *
     * @param appIds 应用id
     * @return {@link List}<{@link Map}<{@link Long}, {@link String}>>
     */
    @GetMapping(name = "查询appIds信息", value = "/listSimpleAppInfo")
    public JsonResult<List<SimpleAppInfoVo>> listSimpleAppInfo(String appIds) {
        List<SimpleAppInfoVo> appInfoVos = new ArrayList<>();
        if (StringUtils.isBlank(appIds)) {
            return JsonResult.success(appInfoVos);
        }
        List<Long> appIdList = Arrays.stream(appIds.split(",")).map(Long::valueOf).collect(Collectors.toList());
        List<AppSimpleDto> apps = remoteAppService.findByIds(appIdList);
        apps.forEach(u -> {
            SimpleAppInfoVo simpleAppInfoVo = SimpleAppInfoVo.builder().appId(u.getId())
                    .appName(u.getName())
                    .appKey(u.getAppKey())
                    .build();
            appInfoVos.add(simpleAppInfoVo);
        });
        return JsonResult.success(appInfoVos);
    }

    /**
     * 通过appName(前模糊)查询app应用信息
     *
     * @param appName 应用名称
     * @return {@link JsonResult}<{@link List}<{@link SimpleAppInfoVo}>>
     */
    @GetMapping(name = "通过appName(前模糊)查询app应用信息", value = "/listSimpleAppInfoByNameLike")
    public JsonResult<List<SimpleAppInfoVo>> listSimpleAppInfoByNameLike(String appName) {
        List<SimpleAppInfoVo> appInfoVos = new ArrayList<>();
        if (StringUtils.isBlank(appName)) {
            return JsonResult.success(appInfoVos);
        }
        DubboResult<List<AppSimpleDto>> simpleDtoResult = remoteAppService.findAppNameByNameLike(appName);
        List<AppSimpleDto> simpleDtoList = simpleDtoResult.getResult();
        simpleDtoList.forEach(u -> {
            SimpleAppInfoVo simpleAppInfoVo = SimpleAppInfoVo.builder().appId(u.getId())
                    .appName(u.getName())
                    .appKey(u.getAppKey())
                    .build();
            appInfoVos.add(simpleAppInfoVo);
        });
        return JsonResult.success(appInfoVos);
    }


    /**
     * 上传文件
     *
     * @return {@link JsonResult}<{@link String}>
     */
    @PostMapping("/uploadFile")
    public JsonResult<String> uploadFile(@RequestParam MultipartFile file) {
        try {
            return JsonResult.success(fileUploadManager.uploadFile(file));
        } catch (Exception e) {
            throw new BizRuntimeException(PlatformConsoleErrorEnum.FILE_UNKNOWN_ERROR, e);
        }
    }

    /**
     * 根据枚举类名获取枚举常量
     * <p>
     * 传参返回对应枚举常量，不传参则返回管理后台所有枚举类常量，以下是目前已有枚举：
     * <p>PrototypeTypeEnum    原型类型枚举
     * <p>TemplateTypeEnum     模版类型枚举
     * <p>TemplateSyncStateEnum 模版同步状态枚举
     * <p>PlaywayTypeEnum      玩法类型枚举
     * <p>ProjectStateEnum     项目状态枚举
     * <p>PrototypeStateEnum   原型状态枚举
     *
     * @param name 枚举类名
     * @return {@link JsonResult}<{@link List}<{@link EnumVo}>>
     */
    @GetMapping("/getEnumByName")
    public JsonResult<List<EnumVo>> getEnumByName(@RequestParam(required = false) String name) {
        return JsonResult.success(commonBackendManager.getEnumByName(name));
    }

    /**
     * 获取省和城市
     *
     * @return {@link JsonResult}<{@link List}<{@link RegionVo}>>
     */
    @GetMapping(name = "查询省份和城市信息", value = "/getProvinceAndCity")
    public JsonResult<List<RegionVo>> getProvinceAndCity(HttpServletResponse response) {
        return JsonResult.success(regionManager.getProvinceAndCity());
    }

    /**
     * 查询大雁项目
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link PageDayanProjectVo}>
     */
    @PostMapping(name = "查询大雁项目", value = "/searchDayan")
    public JsonResult<PageDayanProjectVo> searchDayan(@Validated @RequestBody DayanQueryParam param) {
        ProjectSearchDto projectSearchDto = BeanUtil.copyProperties(param, ProjectSearchDto.class);
        PageRes<AmiyaSimpleProjectRes> amiyaSimpleProjectResPageRes = remoteAmiyaProjectService.pageList(projectSearchDto);
        PageDayanProjectVo pageDayanProjectVo = new PageDayanProjectVo();
        pageDayanProjectVo.setTotalCount(amiyaSimpleProjectResPageRes.getTotal());
        pageDayanProjectVo.setList(BeanUtil.copyToList(amiyaSimpleProjectResPageRes.getData(), DayanSearchVo.class));
        return JsonResult.success(pageDayanProjectVo);
    }


    @GetMapping(name = "查询sso用户信息", value = "/searchSsoUser")
    public JsonResult<List<SsoUserVo>> searchSsoUser(String name) {
        List<SsoUserVo> ssoUserVos = new ArrayList<>();

        if (StringUtils.isBlank(name)) {
            return JsonResult.success(ssoUserVos);
        }
        List<AdminDto> adminDtos = remoteAdminService.findByNameLike(name);
        return JsonResult.success(BeanUtil.copyToList(adminDtos, SsoUserVo.class));
    }

    @GetMapping(name = "查询sso用户信息", value = "/batchFindAdminInfoByIds")
    public JsonResult<List<SsoUserVo>> batchFindAdminInfoByIds(String ids) {
        if (StringUtils.isBlank(ids)) {
            return JsonResult.success();
        }
        List<Long> list = Arrays.stream(ids.split(","))
                .map(Long::valueOf).collect(Collectors.toList());
        List<AdminInfoDto> adminInfoDtos = remoteAdminService.batchFindAdminInfoByIds(list);
        return JsonResult.success(BeanUtil.copyToList(adminInfoDtos, SsoUserVo.class));
    }

    /**
     * 根据类型获取身份信息
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link IdentityVo}>
     */
    @PostMapping(name = "根据类型获取身份信息", value = "/listAllIdentityByType")
    public JsonResult<IdentityVo> listAllIdentityByType(@Validated @RequestBody IdentityQueryParam param) {
        return JsonResult.success(projectManager.listAllIdentityByType(param.getType(), param.getAppId()));
    }

    @PostMapping(name = "校验", value = "/check")
    public JsonResult<String> check(@Validated @RequestBody CheckParam param) {
        Configable configable = ConfigType.ofCode(param.getType()).getCreator().create(null, JSONObject.toJSONString(param.getContent()));
        // 校验
        Conditions.expectNotNull(configable, PlatformConsoleErrorEnum.CONFIG_CHECK_ERROR);
        CheckMode checkMode = new CheckMode(param.getMode() == 1, true);
        configable.findConfigErrors(checkMode);
        return JsonResult.success();
    }

    /**
     * 生成id
     *
     * @param type 类型
     * @return {@link JsonResult}<{@link String}>
     */
    @GetMapping("/generateID")
    public JsonResult<String> generateID(@RequestParam Integer type) {
        ConfigType configType = ConfigType.ofCode(type);
        Conditions.expectNotNull(configType, PlatformConsoleErrorEnum.INVALID_TYPE);
        return JsonResult.success(configType.getCreator().generateId());
    }

    /**
     * 获取导航栏代码
     *
     * @param appId 应用id
     * @return {@link JsonResult}<{@link String}>
     */
    @GetMapping("/getNavProgram")
    public JsonResult<String> getNavProgram(@NotNull(message = "appId必传") Long appId) {
        return JsonResult.success(Optional.ofNullable(remoteAppExtraServiceNew.findAppConfigDto(appId))
                .map(DubboResult::getResult)
                .map(AppConfigDto::getNavProgram)
                .orElse(null));
    }


}
