package cn.com.duiba.galaxy.console.controller;


import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.galaxy.basic.enums.OptionTypeEnum;
import cn.com.duiba.galaxy.basic.enums.PrototypeTypeEnum;
import cn.com.duiba.galaxy.basic.model.entity.ProjectEntity;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeEntity;
import cn.com.duiba.galaxy.basic.model.jsonfield.OptionJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.PlayAttributesJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.StrategyJson;
import cn.com.duiba.galaxy.basic.service.ProjectService;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.manager.ProjectManager;
import cn.com.duiba.galaxy.console.model.param.project.AccountConfigUpdateParam;
import cn.com.duiba.galaxy.console.model.param.project.OpenbsUpdateParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectCopyParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectNewParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectQueryParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectUpdateParam;
import cn.com.duiba.galaxy.console.model.param.project.SimpleProjectParam;
import cn.com.duiba.galaxy.console.model.param.project.UpdateAppIdParam;
import cn.com.duiba.galaxy.console.model.param.valid.GroupByProjectType;
import cn.com.duiba.galaxy.console.model.vo.project.PageProjectVo;
import cn.com.duiba.galaxy.console.model.vo.project.ProjectQueryVo;
import cn.com.duiba.galaxy.console.model.vo.project.ProjectSimpleVo;
import cn.com.duiba.galaxy.core.util.ValidateUtils;
import cn.com.duiba.galaxy.sdk.base.JsonResult;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.groups.Default;
import java.util.List;
import java.util.Objects;

/**
 * 活动管理
 *
 * @author fangxiaorun
 * @date 2022/09/26
 */
@RestController
@Slf4j
@RequestMapping("/project")
public class ProjectConsoleController {

    @Resource
    private ProjectManager projectManager;

    @Autowired
    private ProjectService projectService;
    @Autowired
    private PrototypeService prototypeService;

    @Resource
    private RemoteAppService remoteAppService;


    /**
     * 新建项目
     *
     * @param param 参数
     * @return {@link Boolean}
     */
    @PostMapping(name = "新建项目", value = "/newProject")
    public JsonResult<Boolean> newProject(@RequestBody ProjectNewParam param) throws BizException {

        Long prototypeId = param.getPrototypeId();
        PrototypeEntity prototypeEntity = prototypeService.getById(prototypeId);


        if (Objects.equals(PrototypeTypeEnum.CUSTOMIZATION.getCode(), prototypeEntity.getPrototypeType())) {
            ValidateUtils.validate(param, Default.class, GroupByProjectType.Custom.class);
        } else {
            ValidateUtils.validate(param, Default.class, GroupByProjectType.Template.class);
        }
        Long projectId = projectManager.newProject(param);
        if (projectId == null) {
            return JsonResult.fail(PlatformConsoleErrorEnum.NEW_PROJECT_ERROR);
        }
        return JsonResult.success();
    }

    /**
     * 根据项目id查询项目信息
     *
     * @param projectId 项目id
     * @return {@link ProjectQueryVo}
     */
    @GetMapping(name = "查询项目信息", value = "/getProject")
    public JsonResult<ProjectQueryVo> getProject(@RequestParam Long projectId) {
        return JsonResult.success(projectManager.getProjectByProjectId(projectId));
    }

    /**
     * 更新项目
     *
     * @param param 参数
     * @return {@link Boolean}
     */
    @PostMapping(name = "更新项目", value = "/update")
    public JsonResult<String> update(@RequestBody ProjectUpdateParam param) throws BizException {
        // 更新
        Boolean result = projectManager.updateProject(param);
        if (Boolean.FALSE.equals(result)) {
            return JsonResult.fail(PlatformConsoleErrorEnum.SYSTEM_ERROR);
        }
        return JsonResult.success();

    }

    /**
     * 获取简单项目
     *
     * @param projectId 项目id
     * @return {@link JsonResult}<{@link ProjectQueryVo}>
     */
    @GetMapping(name = "获取简单项目", value = "/getSimpleProject")
    public JsonResult<ProjectSimpleVo> getSimpleProject(Long projectId) {
        // 更新
        ProjectEntity projectEntity = projectService.getById(projectId);
        ProjectSimpleVo simpleVo = BeanUtil.copyProperties(projectEntity, ProjectSimpleVo.class);
        return JsonResult.success(simpleVo);
    }


    /**
     * 更新状态
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping(name = "更新项目状态", value = "/updateState")
    public JsonResult<Boolean> updateState(@RequestBody SimpleProjectParam param) {
        /**
         * 模板活动：积分商城创建活动，活动平台创建活动。
         * 定制活动：定制活动创建活动
         */

        return JsonResult.success(projectManager.updateState(param.getId(), param.getState()));
    }

    /**
     * 项目列表(分页)
     *
     * @param queryParam 查询参数
     * @return {@link PageProjectVo}
     */
    @GetMapping(name = "查询列表", value = "/list")
    public JsonResult<PageProjectVo> list(ProjectQueryParam queryParam) {
        return JsonResult.success(projectManager.listProjects(queryParam));
    }


    /**
     * 更新应用id
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping(name = "定向appId", value = "/updateAppId")
    public JsonResult<Boolean> updateAppId(@Validated @RequestBody UpdateAppIdParam param) {

        ProjectEntity projectEntity = projectService.getById(param.getId());
        Conditions.expectNotNull(projectEntity, PlatformConsoleErrorEnum.NULL_PROJECT);

        Long appId = param.getAppId();
        // appId 未变更
        if (Objects.equals(appId, projectEntity.getAppId())) {
            return JsonResult.success();
        }

        AppSimpleDto appSimpleDto = remoteAppService.getSimpleAppNoCache(appId);
        Conditions.expectNotNull(appSimpleDto, PlatformConsoleErrorEnum.APP_NOT_FOUNT);

        PlayAttributesJson playAttributes = projectEntity.getPlayAttributes();
        List<StrategyJson> rules = playAttributes.getRules();
        // 直接更新
        if (CollectionUtils.isNotEmpty(rules)) {
            for (StrategyJson rule : rules) {
                // 模板活动未启用的发奖规则暂时不进行校验。如果启用，在发布的时候会再次校验
                if (Boolean.FALSE.equals(rule.getStrategyEnable())) {
                    continue;
                }
                List<OptionJson> options = rule.getOptions();
                if (CollectionUtils.isNotEmpty(options)) {
                    // 是否存在采购商品
                    boolean haveProcurement = options.stream().anyMatch(u -> Objects.equals(u.getType(), OptionTypeEnum.PROCUREMENT.getCode()));
                    Conditions.expectFalse(haveProcurement, PlatformConsoleErrorEnum.NOT_ALLOW_UPDATE_APP);
                }
            }
        }
        return JsonResult.success(projectManager.updateAppId(param.getId(), param.getAppId()));
    }


    /**
     * 更新openbs状态
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping("/updateOpenBs")
    public JsonResult<Boolean> updateOpenBs(@Validated @RequestBody OpenbsUpdateParam param) {
        LambdaUpdateWrapper<ProjectEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(ProjectEntity::getId, param.getProjectId());
        updateWrapper.set(ProjectEntity::getOpenbs, param.getState());
        return JsonResult.success(projectService.update(updateWrapper));
    }


    /**
     * 查询扣费配置状态
     *
     * @param projectId 项目id
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @GetMapping("/getAccountSwitchConfig")
    public JsonResult<Boolean> getAccountSwitchConfig(@RequestParam("projectId") String projectId) {
        // todo 待实现 prize.center.api-> RemoteSwitchConfigService  是否需要适配
        return JsonResult.success();
    }


    /**
     * 更新扣费配置
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping("/updateAccountSwitchConfig")
    public JsonResult<Boolean> updateAccountSwitchConfig(@Validated @RequestBody AccountConfigUpdateParam param) {
        // todo 待实现
        return JsonResult.success();
    }

    /**
     * 获取项目链接
     *
     * @param projectId 项目id
     * @return {@link JsonResult}<{@link String}>
     */
    @GetMapping("/getIndexUrl")
    public JsonResult<String> getIndexUrl(@RequestParam Long projectId) {
        ProjectEntity projectEntity = projectService.getById(projectId);
        if (projectEntity == null || projectEntity.getAppId() == null) {
            return JsonResult.success();
        }
        return JsonResult.success(projectManager.getIndexUrl(projectId, projectEntity.getAppId()));
    }

    /**
     * 复制项目
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link String}>
     */
    @PostMapping("/copyProject")
    public JsonResult<Boolean> copyProject(@Validated @RequestBody ProjectCopyParam param) {
        return JsonResult.success(projectManager.copyProject(param));

    }

    @GetMapping("/getReadyCopyProject")
    public JsonResult<ProjectQueryVo> getReadyCopyProject(Long projectId) {
        return JsonResult.success(projectManager.getReadyCopyProject(projectId));
    }


    /**
     * 获取项目预览链接
     *
     * @param projectId 项目id
     * @return {@link JsonResult}<{@link String}>
     */
    @GetMapping("/getPreviewUrl")
    public JsonResult<String> getPreviewUrl(Long projectId) {
        return JsonResult.success(projectManager.getPreviewUrl(projectId));
    }
}