package cn.com.duiba.galaxy.console.controller;


import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeJarEntity;
import cn.com.duiba.galaxy.basic.service.PrototypeJarService;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.console.enums.EnvironmentTypeEnum;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.manager.PrototypeManager;
import cn.com.duiba.galaxy.console.model.param.prototype.CrossPushPrototypeParam;
import cn.com.duiba.galaxy.console.model.param.prototype.PrototypeQueryParam;
import cn.com.duiba.galaxy.console.model.param.prototype.PrototypeSaveParam;
import cn.com.duiba.galaxy.console.model.param.prototype.PrototypeUpdateParam;
import cn.com.duiba.galaxy.console.model.param.prototype.SimplePrototypeParam;
import cn.com.duiba.galaxy.console.model.param.prototype.UpdatePrototypeStateParam;
import cn.com.duiba.galaxy.console.model.param.valid.GroupByResponsibility;
import cn.com.duiba.galaxy.console.model.vo.PagePrototypeVo;
import cn.com.duiba.galaxy.console.model.vo.prototype.PrototypeVo;
import cn.com.duiba.galaxy.core.load.InstallJarService;
import cn.com.duiba.galaxy.core.util.ValidateUtils;
import cn.com.duiba.galaxy.sdk.base.JsonResult;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.com.duiba.sso.api.web.interceptor.annotation.CanAccess;
import cn.com.duibaboot.ext.autoconfigure.security.dpefensivepolicy.CsrfOff;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.groups.Default;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.Map;
import java.util.concurrent.Future;


/**
 * 原型管理
 *
 * @author fangxiaorun
 * @date 2022/09/26
 */
@RestController
@Slf4j
@RequestMapping("/prototype")
public class PrototypeController {


    @Autowired
    private PrototypeService prototypeService;
    @Resource
    private PrototypeManager prototypeManager;
    @Resource
    private InstallJarService installJarService;
    @Resource
    private PrototypeJarService prototypeJarService;

    /**
     * 新建原型
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping(name = "新建原型", value = "/newPrototype")
    public JsonResult<Boolean> newPrototype(@Validated @RequestBody PrototypeSaveParam param) {
        return JsonResult.success(prototypeManager.newPrototype(param));
    }

    /**
     * 获取原型信息
     *
     * @param param 参数
     * @return {@link PrototypeVo}
     */
    @GetMapping(name = "获取原型信息", value = "/getPrototype")
    public JsonResult<PrototypeVo> getPrototype(SimplePrototypeParam param) {
        return JsonResult.success(prototypeManager.getPrototype(param));
    }

    /**
     * 更新原型
     * 后期优化
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping(name = "更新原型", value = "/update")
    public JsonResult<String> update(@RequestBody PrototypeUpdateParam param) throws Exception {

        //提交构建任务
        String gitlab = param.getPlayAttributes().getJavaCode();
        Long prototypeId = param.getId();
        ValidateUtils.validate(param, GroupByResponsibility.Developer.class, Default.class);

        Future<PrototypeJarEntity> task = installJarService.submitJarTask(prototypeId, gitlab);

        Boolean result = prototypeManager.updatePrototype(param, task);

        if (Boolean.FALSE.equals(result)) {
            return JsonResult.fail(PlatformConsoleErrorEnum.SYSTEM_ERROR);
        }

        return JsonResult.success();
    }


    /**
     * 更新状态
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link Void}>
     */
    @PostMapping(name = "更新原型状态", value = "/updateState")
    public JsonResult<Void> updateState(@Validated @RequestBody UpdatePrototypeStateParam param) {
        prototypeManager.updatePrototypeState(param.getPrototypeId(), param.getState());
        return JsonResult.success();
    }


    /**
     * 原型列表(分页)
     *
     * @param param 参数
     * @return {@link JsonResult}<{@link PagePrototypeVo}>
     */
    @GetMapping(name = "原型列表(分页)", value = "/list")
    public JsonResult<PagePrototypeVo> list(PrototypeQueryParam param) {
        Conditions.expectTrue(param != null, PlatformConsoleErrorEnum.PARAM_ERROR);
        return JsonResult.success(prototypeManager.listPrototypes(param));
    }

    /**
     * 上传原型文件
     *
     * @param file 文件
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping(name = "上传原型文件", value = "/importConfig")
    public JsonResult<JSONObject> importConfig(@RequestParam("file") MultipartFile file) throws BizException {

        try (InputStream in = file.getInputStream()) {
            byte[] bytes = new byte[in.available()];
            in.read(bytes);
            String content = new String(bytes);
            //检查是否为json
            JSONObject attribute = JSONObject.parseObject(content);
//            fileUploadManager.uploadFile(file);
            return JsonResult.success(attribute);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 主页皮肤预览
     *
     * @param prototypeId 原型id
     */
    @CsrfOff
    @GetMapping("previewSkinOfIndex")
    public void previewSkinOfIndex(@RequestParam Long prototypeId, HttpServletResponse response) {
        try (PrintWriter out = response.getWriter()) {
            String html = prototypeManager.getSkinOfIndex(prototypeId);
            response.setCharacterEncoding("UTF-8");
            response.setContentType("text/html");
            out.print(html);
        } catch (IOException e) {
            throw new BizRuntimeException(PlatformConsoleErrorEnum.SYSTEM_ERROR);
        }
    }


    /**
     * 跨域推送原型
     *
     * @param param   参数
     * @param request 请求
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @PostMapping("pushPrototype")
    public JsonResult<Boolean> pushPrototype(@Validated @RequestBody CrossPushPrototypeParam param
            , HttpServletRequest request) {
        EnvironmentTypeEnum environmentTypeEnum = EnvironmentTypeEnum.getEnvironmentTypeEnumByCode(param.getEnvironment());
        String targetUri = StringUtils.replace(request.getRequestURI(), "pushPrototype", "receiveAndSavePrototype");
        String url = String.format("http://galaxy.%s.com.cn%s", environmentTypeEnum.getDesc(), targetUri);
        param.setUrl(url);
        prototypeManager.pushPrototype(param);
        return JsonResult.success();
    }

    /**
     * 接收并保存原型
     *
     * @param map 地图
     * @return {@link JsonResult}<{@link Boolean}>
     */
    @CsrfOff
    @CanAccess
    @PostMapping("receiveAndSavePrototype")
    public JsonResult<Boolean> receiveAndSavePrototype(@RequestBody Map<String, String> map) {
        return JsonResult.success(prototypeManager.receiveAndSavePrototype(map));

    }

}



