package cn.com.duiba.galaxy.console.manager.impl;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.galaxy.adapter.base.adapter.StockAdapter;
import cn.com.duiba.galaxy.adapter.base.dto.UpdateStockDto;
import cn.com.duiba.galaxy.adapter.base.enums.AdapterTypeEnum;
import cn.com.duiba.galaxy.adapter.base.handler.GalaxyAdapter;
import cn.com.duiba.galaxy.adapter.base.handler.GalaxyAdapterFactory;
import cn.com.duiba.galaxy.basic.enums.OptionTypeEnum;
import cn.com.duiba.galaxy.basic.enums.ProjectCreateSourceEnum;
import cn.com.duiba.galaxy.basic.enums.ProjectStateEnum;
import cn.com.duiba.galaxy.basic.enums.PrototypeTypeEnum;
import cn.com.duiba.galaxy.basic.model.entity.ProjectEntity;
import cn.com.duiba.galaxy.basic.model.entity.ProjectExtEntity;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeEntity;
import cn.com.duiba.galaxy.basic.model.entity.SpEntity;
import cn.com.duiba.galaxy.basic.model.jsonfield.DayanProjectJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.LayeredRuleJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.OptionJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.PlayAttributesJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.StrategyJson;
import cn.com.duiba.galaxy.basic.service.ProjectExtService;
import cn.com.duiba.galaxy.basic.service.ProjectService;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.basic.service.SpService;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.enums.ProjectUpdateButtonTypeEnum;
import cn.com.duiba.galaxy.console.enums.PrototypeStateEnum;
import cn.com.duiba.galaxy.console.manager.OptionManager;
import cn.com.duiba.galaxy.console.manager.ProjectManager;
import cn.com.duiba.galaxy.console.manager.SpManager;
import cn.com.duiba.galaxy.console.manager.SsoExtManager;
import cn.com.duiba.galaxy.console.model.param.LayeredRuleJsonParm;
import cn.com.duiba.galaxy.console.model.param.OperationalOptionParam;
import cn.com.duiba.galaxy.console.model.param.OptionJsonParam;
import cn.com.duiba.galaxy.console.model.param.PlayAttributesJsonParam;
import cn.com.duiba.galaxy.console.model.param.ProjectExtParam;
import cn.com.duiba.galaxy.console.model.param.StrategyJsonParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectCopyParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectNewParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectUpdateParam;
import cn.com.duiba.galaxy.console.model.param.valid.GroupByProjectCreateSource;
import cn.com.duiba.galaxy.console.model.param.valid.GroupByPrototypeType;
import cn.com.duiba.galaxy.console.model.param.valid.GroupByResponsibility;
import cn.com.duiba.galaxy.console.model.vo.PlayAttributesJsonVo;
import cn.com.duiba.galaxy.console.model.vo.ProjectExtraBaseVo;
import cn.com.duiba.galaxy.console.model.vo.SpVo;
import cn.com.duiba.galaxy.console.model.vo.StrategyJsonVo;
import cn.com.duiba.galaxy.console.model.vo.project.ProjectQueryVo;
import cn.com.duiba.galaxy.core.activity.BizProjectConfigMap;
import cn.com.duiba.galaxy.core.activity.CheckMode;
import cn.com.duiba.galaxy.core.activity.ClientBizProjectConfigMap;
import cn.com.duiba.galaxy.core.enums.PlatformCoreErrorEnum;
import cn.com.duiba.galaxy.core.factory.ObjectBuildFactory;
import cn.com.duiba.galaxy.core.load.custom.BizProject;
import cn.com.duiba.galaxy.core.util.ValidateUtils;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import javax.annotation.Resource;
import javax.validation.groups.Default;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * 项目公共服务实现
 *
 * @author fangxiaorun
 * @date 2022/9/30
 */
@Slf4j
public abstract class AbstractProjectManagerImpl implements ProjectManager {
    @Autowired
    private ProjectService projectService;

    @Autowired
    private PrototypeService prototypeService;

    @Autowired
    private ProjectExtService projectExtService;

    @Autowired
    protected SpService spService;

    @Autowired
    protected SpManager spManager;

    @Autowired
    protected OptionManager optionManager;

    @Autowired
    private SsoExtManager ssoExtManager;


    @Autowired
    private ClientBizProjectConfigMap clientBizProjectConfigMap;

    @Autowired
    private GalaxyAdapterFactory<GalaxyAdapter> galaxyAdapterFactory;

    @Resource
    private TransactionTemplate transactionTemplate;


    @Override
    public Long newProject(ProjectNewParam param) {
        Long prototypeId = param.getPrototypeId();

        // 校验原型
        PrototypeEntity prototypeEntity = prototypeService.getById(prototypeId);
        Conditions.expectNotNull(prototypeEntity, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);

        // 校验原型状态 ：启用状态方可创建项目
        Conditions.expectTrue(Objects.equals(prototypeEntity.getState(), PrototypeStateEnum.ENABLED.getCode()), PlatformConsoleErrorEnum.PROTOTYPE_NOT_ENABLED);

        //创建项目选择定制原型，则关联大雁项目为必填项；创建项目选择模版原型，则关联大雁项目为非必填项
        // 保存project
        ProjectEntity projectEntity = new ProjectEntity();
        // project不保存javaCode, cfUrl, skins
        PlayAttributesJson playAttributes = prototypeEntity.getPlayAttributes();
        if (playAttributes != null) {
            playAttributes.setJavaCode(null);
            playAttributes.setSkins(null);
            playAttributes.setCfUrl(null);
            projectEntity.setPlayAttributes(playAttributes);
        }

        if (prototypeEntity.getViewAttributes() != null) {
            projectEntity.setViewAttributes(prototypeEntity.getViewAttributes());
        }
        projectEntity.setProjectName(param.getProjectName());
        projectEntity.setPrototypeId(prototypeId);
        projectEntity.setState(ProjectStateEnum.INIT.getCode());
        Integer source = param.getSource();
        projectEntity.setOperator(Objects.equals(ProjectCreateSourceEnum.ACTIVITY_PLATFORM.getCode(), source) ? ssoExtManager.getAdminInfo().getName() : "system");
        projectEntity.setSource(source);
        projectEntity.setProd(param.getProd());

        Boolean execute = transactionTemplate.execute(t -> {
            try {
                boolean projectSave = projectService.save(projectEntity);

                boolean projectExtraSave = true;

                // 是否是活动平台来源
                boolean isPlatform = Objects.equals(source, ProjectCreateSourceEnum.ACTIVITY_PLATFORM.getCode());

                if (isPlatform && projectSave) {
                    // 保存projectExtra
                    ProjectExtEntity projectExtEntity = new ProjectExtEntity();
                    if (StringUtils.isNotBlank(param.getDayanProjectId()) && StringUtils.isNotBlank(param.getDayanProjectName())) {
                        DayanProjectJson dayanProjectJson = new DayanProjectJson(Long.valueOf(param.getDayanProjectId()), param.getDayanProjectName());
                        projectExtEntity.setDayanProject(dayanProjectJson);
                    }
                    projectExtEntity.setProjectId(projectEntity.getId());
                    projectExtraSave = projectExtService.save(projectExtEntity);
                }
                return projectSave && projectExtraSave;
            } catch (Exception e) {
                t.setRollbackOnly();
                log.warn("Exception [{}]:", e.getMessage(), e);
                return false;
            }
        });

        if (Boolean.TRUE.equals(execute)) {
            return projectEntity.getId();
        }
        return null;
    }


    @Override
    public Boolean updateProject(ProjectUpdateParam param) throws BizException {
        List<UpdateStockDto> updateStockDtos = new ArrayList<>();
        // 查询更新前的project信息
        ProjectEntity projectEntity = projectService.getById(param.getId());
        Conditions.expectNotNull(projectEntity, PlatformConsoleErrorEnum.NULL_PROJECT);

        // 项目更新前被他人修改
        Conditions.expectTrue(param.getGmtModified().getTime() == projectEntity.getGmtModified().getTime()
                , PlatformConsoleErrorEnum.NOT_NEWEST_PROJECT);

        // 保存并发布需要定向appid
        if (Objects.equals(param.getButtonType(), ProjectUpdateButtonTypeEnum.SAVE_PUBLISH.getCode())) {
            Conditions.expectNotNull(projectEntity.getAppId(), PlatformConsoleErrorEnum.NOT_DIRECTED_APP);
        }


        // 查询关联原型信息
        PrototypeEntity prototypeEntity = prototypeService.getById(projectEntity.getPrototypeId());
        Conditions.expectNotNull(prototypeEntity, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);

        // 校验
        validatedParam(prototypeEntity, projectEntity, param);

        // 入参param 转 entity
        ProjectEntity projectEntityNew = BeanUtil.copyProperties(projectEntity, ProjectEntity.class);
        BeanUtil.copyProperties(param, projectEntityNew);

        // 根据条件设置字段
        if (!SpringEnvironmentUtils.isProdEnv() && param.getProd() != null) {
            projectEntityNew.setProd(param.getProd());
        }
        // 获取要更新的状态
        ProjectStateEnum stateEnum = getProjectState(projectEntity, param.getButtonType());
        if (stateEnum != null) {
            projectEntityNew.setState(stateEnum.getCode());
        }


        // 获取projectId
        Long projectId = param.getId();

        /**
         * 过滤库和前端传过来的奖项。将不存在的奖项逻辑删除
         */
        List<String> errorMsg = new ArrayList<>();
        Boolean result = transactionTemplate.execute(t -> {
            try {
                handleOption(projectEntityNew.getAppId(), projectId, param.getPlayAttributes(), updateStockDtos);

                BizProject bizProject = new BizProject(projectEntityNew);
                BizProjectConfigMap bizProjectConfigMap = bizProject.getBizProjectConfigMap();
                bizProjectConfigMap.findConfigErrors(new CheckMode(false, false));

                // 根据projectId更新project
                LambdaUpdateWrapper<ProjectEntity> projectUpdateWrapper = new LambdaUpdateWrapper<>();
                projectUpdateWrapper.eq(ProjectEntity::getId, projectId);
                Integer source = projectEntity.getSource();
                projectEntityNew.setOperator(Objects.equals(ProjectCreateSourceEnum.ACTIVITY_PLATFORM.getCode(), source) ? ssoExtManager.getAdminInfo().getName() : "system");
                // 手动设置更新时间
                projectEntityNew.setGmtModified(new Date());
                boolean updateProject = projectService.update(projectEntityNew, projectUpdateWrapper);


                // 是否是定制定制项目
                // 来源是活动平台的项目需要更新extra

                // 是否是活动平台来源
                boolean isPlatform = Objects.equals(projectEntity.getSource(), ProjectCreateSourceEnum.ACTIVITY_PLATFORM.getCode());

                // 保存或更新projectExt
                ProjectExtParam projectExt = param.getProjectExtra();
                boolean updateProjectExtra = true;
                // 活动平台才保存额外扩展信息
                if (isPlatform && projectExt != null) {
                    ProjectExtEntity extEntity = BeanUtil.copyProperties(projectExt, ProjectExtEntity.class);
                    LambdaUpdateWrapper<ProjectExtEntity> projectExtEntityWrapper = new LambdaUpdateWrapper<>();
                    projectExtEntityWrapper.eq(ProjectExtEntity::getProjectId, projectId);
                    extEntity.setEffectTags(JSONObject.toJSONString(projectExt.getEffectTags()));
                    updateProjectExtra = projectExtService.saveOrUpdate(extEntity, projectExtEntityWrapper);
                }
                return updateProject && updateProjectExtra;
            } catch (Exception e) {
                t.setRollbackOnly();
                log.warn("保存活动失败", e);
                if (e instanceof BizException && Objects.equals(((BizException) e).getCode(), "2003004019")) {
                    log.warn("保存活动失败:三方库存不足", e);

                } else if (e instanceof BizRuntimeException && Objects.equals(((BizRuntimeException) e).getCode(), PlatformCoreErrorEnum.PRIZE_STOCK_NOT_ENOUGH.getCode())) {
                    log.warn("保存活动失败:三方库存不足", e);
                } else if (CollectionUtils.isNotEmpty(updateStockDtos)) {
                    updateStockDtos.forEach(u -> u.setType(Objects.equals(UpdateStockDto.INCREASE, u.getType()) ? UpdateStockDto.DEDUCT : UpdateStockDto.INCREASE));
                    StockAdapter stockAdapter = galaxyAdapterFactory.getAdapterByType(AdapterTypeEnum.CREDIT, "StockAdapter");
                    try {
                        stockAdapter.batchDeductStock(projectEntity.getAppId().toString(), updateStockDtos);
                    } catch (BizException ex) {
                        log.warn("保存活动失败:三方库存不足", e);
                    }

                }
                errorMsg.add(e.getMessage());
            }
            return false;
        });
        if (CollectionUtils.isNotEmpty(errorMsg)) {
            throw new BizRuntimeException(PlatformConsoleErrorEnum.UPDATE_PROJECT_ERROR.setDesc(errorMsg.get(0)));
        }
        return result;
    }


    /**
     * 处理奖项
     *
     * @param appId                   应用id
     * @param projectId               项目id
     * @param playAttributesJsonParam 玩法属性json参数
     */
    private void handleOption(Long appId, Long projectId, PlayAttributesJsonParam playAttributesJsonParam, List<UpdateStockDto> updateStockDtos) throws BizException {

        if (CollectionUtils.isEmpty(playAttributesJsonParam.getRules())) {
            optionManager.batchOperationalOption(appId, projectId, new ArrayList<>(), updateStockDtos);
        } else {
            List<OperationalOptionParam> optionParams = buildOperationalOptionParams(playAttributesJsonParam.getRules(), projectId);
            // 发奖规则存在采购商品需要定向appId
            if (optionParams.stream().anyMatch(u -> u.getOptionType().equals(OptionTypeEnum.PROCUREMENT.getCode()))) {
                Conditions.expectNotNull(appId, PlatformConsoleErrorEnum.NOT_DIRECTED_APP);
            }
            optionManager.batchOperationalOption(appId, projectId, optionParams, updateStockDtos);
        }
    }


    /**
     * 构建操作奖项参数个数
     * 将前端传过来的参数构建OptionParam并入库
     *
     * @param strategyJsonParams 策略json参数个数
     * @param projectId          项目id
     * @return {@link List}<{@link OperationalOptionParam}>
     */
    private List<OperationalOptionParam> buildOperationalOptionParams(List<StrategyJsonParam> strategyJsonParams, Long projectId) {


        List<OperationalOptionParam> optionParams = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(strategyJsonParams)) {
            // 过滤有变更操作的发奖规则
            //List<StrategyJsonParam> changeRus = strategyJsonParams.stream().filter(u -> !Objects.equals(u.getChangeType(), ChangeTypeEnum.NO_CHANGE.getCode())).collect(Collectors.toList());
            // 遍历发奖规则
            for (StrategyJsonParam rus : strategyJsonParams) {
                String ruId = rus.getId();
                List<OptionJsonParam> options = rus.getOptions();
                for (OptionJsonParam option : options) {
                    OperationalOptionParam operationalOptionParam = transformOption(option, projectId, ruId, rus.getName());
                    optionParams.add(operationalOptionParam);
                }
                //  开启了用户分层出奖
                if (rus.isLayeredEnable()) {
                    List<LayeredRuleJsonParm> layeredRules = rus.getLayeredRules();
                    if (CollectionUtils.isNotEmpty(layeredRules)) {
                        for (LayeredRuleJsonParm layeredRule : layeredRules) {
                            List<OptionJsonParam> optionsLayered = layeredRule.getOptions();
                            for (OptionJsonParam optionJsonParam : optionsLayered) {
                                OperationalOptionParam operationalOptionParam = transformOption(optionJsonParam, projectId, layeredRule.getId(), layeredRule.getName());
                                optionParams.add(operationalOptionParam);
                            }
                        }
                    }
                }
            }
        }
        return optionParams;
    }


    private OperationalOptionParam transformOption(OptionJsonParam param, Long projectId, String ruId, String ruName) {
        OperationalOptionParam operationalOptionParam = new OperationalOptionParam();
        operationalOptionParam.setProjectId(projectId);
        operationalOptionParam.setId(param.getId());
        operationalOptionParam.setRuleId(ruId);
        operationalOptionParam.setRuleName(ruName);
        operationalOptionParam.setOptionName(param.getName());
        operationalOptionParam.setOptionType(param.getType());
        operationalOptionParam.setOptionImg1(param.getIcon());
        operationalOptionParam.setOptionImg2(param.getIcon2());
        operationalOptionParam.setPrizeId(param.getPrizeId());
        if (!Objects.equals(param.getType(), OptionTypeEnum.THANKS.getCode())) {
            operationalOptionParam.setRate(new BigDecimal(param.getRate()));
            if (param.getTotalStock() != null) {
                operationalOptionParam.setTotalStock(Math.toIntExact(param.getTotalStock()));
            }
        }
        operationalOptionParam.setUserLimit(param.getUserLimit());
        operationalOptionParam.setUserLimitType(param.getUserLimitType());
        operationalOptionParam.setAwardLimit(param.getAwardLimit());
        operationalOptionParam.setAwardLimitType(param.getAwardLimitType());
        operationalOptionParam.setSendCount(param.getSendCount());
        operationalOptionParam.setShowIndexType(param.getShowIndexType());
        operationalOptionParam.setAtleastTimes(param.getAtleastTimes());
        operationalOptionParam.setMultipleOption(param.isMultipleOption() ? 1 : 0);
        operationalOptionParam.setItemId(param.getItemId());
        operationalOptionParam.setBizType(param.getBizType());
        return operationalOptionParam;
    }


    /**
     * 获取项目状态
     *
     * @param projectEntity 项目entity
     * @param buttonType    按钮类型
     */
    private ProjectStateEnum getProjectState(ProjectEntity projectEntity, Integer buttonType) {
        /**
         *
         * 未上线条件：
         *  1.当前状态是待发布，待编辑，已发布待上线时，点击保存，状态变更为待发布；
         *  2.点击保存并发布，状态变更已发布待上线。
         *
         * 已上线的条件：用户只能操作保存并发布
         *  1.状态不变更
         *
         */
        // 默认不更改
        ProjectStateEnum stateEnum = null;
        // 已上线
        if (Objects.equals(projectEntity.getState(), ProjectStateEnum.ON_LINE.getCode())) {
            // 只能操作保存并发布按钮
            Conditions.expectTrue(Objects.equals(buttonType, ProjectUpdateButtonTypeEnum.SAVE_PUBLISH.getCode()), PlatformConsoleErrorEnum.NOT_ALLOW_SAVE);
        } else {
            if (Objects.equals(0, projectEntity.getSource())) {
                // 未上线
                stateEnum = ProjectStateEnum.WAIT_PUBLISH;
                if (Objects.equals(buttonType, ProjectUpdateButtonTypeEnum.SAVE_PUBLISH.getCode())) {
                    // 待发布状态
                    stateEnum = ProjectStateEnum.WAIT_ON_LINE;
                }
            } else {
                //  积分商城活动状态，区别于活动平台（积分商城活动工具，发布即上线）
                stateEnum = Objects.equals(buttonType, ProjectUpdateButtonTypeEnum.SAVE_PUBLISH.getCode()) ? ProjectStateEnum.ON_LINE : ProjectStateEnum.WAIT_ON_LINE;
            }
        }
        return stateEnum;
    }


    /**
     * 验证参数
     *
     * @param prototypeEntity 原型entity
     * @param oldEntity       老entity
     * @param param           参数
     * @throws BizException 业务异常
     */
    private void validatedParam(PrototypeEntity prototypeEntity, ProjectEntity oldEntity, ProjectUpdateParam param) throws BizException {
        List<Class<?>> classes = new ArrayList<>();
        // 默认校验
        classes.add(Default.class);
        // 运营操作校验
        classes.add(GroupByResponsibility.Operator.class);

        // 校验定制活动  或者 模板活动分组
        if (Objects.equals(prototypeEntity.getPrototypeType(), PrototypeTypeEnum.CUSTOMIZATION.getCode())) {
            // 定制活动校验
            classes.add(GroupByPrototypeType.Custom.class);
        }

        // 活动来源分组校验
        if (Objects.equals(oldEntity.getSource(), ProjectCreateSourceEnum.CREDITS_STORE.getCode())) {
            // 积分商城来源校验
            classes.add(GroupByProjectCreateSource.CreditsStore.class);
        }
        Class<?>[] classesNew = new Class<?>[classes.size()];

        // 校验
        ValidateUtils.validate(param, classes.toArray(classesNew));

    }


    @Override
    public ProjectQueryVo getProjectByProjectId(Long projectId) {
        LambdaQueryWrapper<ProjectEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(ProjectEntity::getId, projectId);
        // project
        ProjectEntity projectEntity = projectService.getOne(wrapper);
        if (projectEntity == null) {
            return null;
        }
        PrototypeEntity prototypeEntity = prototypeService.getById(projectEntity.getPrototypeId());

        // 合并 prototype 和project 的配置
        ObjectBuildFactory.mergeConfig(prototypeEntity, projectEntity);

        ProjectQueryVo projectQueryVo = BeanUtil.copyProperties(projectEntity, ProjectQueryVo.class);
        // 目前只有活动平台来源的活动projectExtra表中才有数据
        if (Objects.equals(projectEntity.getSource(), ProjectCreateSourceEnum.ACTIVITY_PLATFORM.getCode())) {
            // projectExtra
            LambdaQueryWrapper<ProjectExtEntity> projectExtWrapper = new LambdaQueryWrapper<>();
            projectExtWrapper.eq(ProjectExtEntity::getProjectId, projectId);
            ProjectExtEntity extEntity = projectExtService.getOne(projectExtWrapper);
            ProjectExtraBaseVo projectExtraBaseVo = BeanUtil.copyProperties(extEntity, ProjectExtraBaseVo.class);
            projectExtraBaseVo.setEffectTags(JSONObject.parseArray(extEntity.getEffectTags(), String.class));
            projectQueryVo.setProjectExtra(projectExtraBaseVo);
        }

        // 设置道具
        List<SpEntity> spEntities = spManager.listAllSps(prototypeEntity.getId());
        List<SpVo> spVos = BeanUtil.copyToList(spEntities, SpVo.class);
        projectQueryVo.setSps(spVos);

        return projectQueryVo;

    }


    /**
     * 更新项目状态
     *
     * @param projectId 项目id
     * @param state     状态
     * @return {@link Boolean}
     */
    @Override
    public Boolean updateState(Long projectId, Integer state) {

        // 该接口只能从上线、下线两个状态变更。其他状态在更新、创建project时自行变更
        /**
         *  从下线->上线
         *     1.判断活动访问时间（活动访问结束时间要大于当前时间）
         *     2.判断是否定向appId
         *  从已发布未上线->上线
         *      1.判断是否定向appId
         */
        ProjectEntity projectEntity = projectService.getById(projectId);
        // 校验项目是否存在
        Conditions.expectNotNull(projectEntity, PlatformConsoleErrorEnum.NULL_PROJECT);

        Integer stateOld = projectEntity.getState();
        long now = System.currentTimeMillis();

        // 上线的项目只能变更-下线
        // 上线  -> 下线
        Integer stateNew = null;
        if (Objects.equals(state, ProjectStateEnum.OFF_LINE.getCode())) {
            // 下线就下线吧，没啥好校验的
            stateNew = state;
        } else if (Objects.equals(stateOld, ProjectStateEnum.OFF_LINE.getCode()) && Objects.equals(state, ProjectStateEnum.ON_LINE.getCode())) {
            // 下线->上线
            // 请先定向app
            Conditions.expectNotNull(projectEntity.getAppId(), PlatformConsoleErrorEnum.NOT_DIRECTED_APP);
            // 项目访问的结束时间小于当前时间
            Conditions.expectTrue(projectEntity.getEndTime().getTime() > now, PlatformConsoleErrorEnum.END_TIME_ERROR);
            stateNew = state;
        } else if (Objects.equals(stateOld, ProjectStateEnum.WAIT_ON_LINE.getCode()) && Objects.equals(state, ProjectStateEnum.ON_LINE.getCode())) {
            stateNew = state;
        }
        // 期望stateNew！=null
        Conditions.expectNotNull(stateNew, PlatformConsoleErrorEnum.STATE_CHANGE_ERROR);
        LambdaUpdateWrapper<ProjectEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(ProjectEntity::getId, projectId).set(ProjectEntity::getState, stateNew);
        return projectService.update(updateWrapper);
    }

    @Override
    public boolean copyProject(ProjectCopyParam param) {

        ProjectEntity projectEntity = projectService.getById(param.getProjectId());
        ProjectEntity projectEntityCopy = BeanUtil.copyProperties(projectEntity, ProjectEntity.class);
        projectEntityCopy.setId(null);
        projectEntityCopy.setProjectName(projectEntity.getProjectName() + "_copy");
        // 发奖规则奖项清楚
        List<StrategyJson> rules = projectEntity.getPlayAttributes().getRules();
        List<StrategyJson> resetRules = resetRules(rules);
        projectEntity.getPlayAttributes().setRules(resetRules);
        projectEntityCopy.setPlayAttributes(projectEntity.getPlayAttributes());
        projectEntityCopy.setState(ProjectStateEnum.INIT.getCode());
        projectEntityCopy.setGmtCreate(new Date());
        projectEntityCopy.setGmtModified(new Date());
        projectEntityCopy.setOperator(ssoExtManager.getAdminInfo().getName());
        projectEntityCopy.setProd(param.getProd());
        projectEntityCopy.setStartTime(null);
        projectEntityCopy.setEndTime(null);
        projectEntityCopy.setOpenbs(0);

        // 拷贝projectExt
        return Boolean.TRUE.equals(transactionTemplate.execute(t -> {
            boolean saveProject = false;
            boolean saveProjectExtState = true;
            try {
                Long num = projectService.saveProject(projectEntityCopy);
                saveProject = num > 0;
                boolean isPlatform = Objects.equals(projectEntity.getSource(), ProjectCreateSourceEnum.ACTIVITY_PLATFORM.getCode());
                if (saveProject && isPlatform) {
                    // 拷贝projectExt
                    LambdaQueryWrapper<ProjectExtEntity> entityLambdaQueryWrapper = new LambdaQueryWrapper<>();
                    entityLambdaQueryWrapper.eq(ProjectExtEntity::getProjectId, param.getProjectId());
                    ProjectExtEntity extEntity = projectExtService.getOne(entityLambdaQueryWrapper);
                    ProjectExtEntity projectExtEntityCopy = BeanUtil.copyProperties(extEntity, ProjectExtEntity.class);
                    projectExtEntityCopy.setId(null);
                    projectExtEntityCopy.setProjectId(projectEntityCopy.getId());
                    projectExtEntityCopy.setGmtCreate(new Date());
                    projectExtEntityCopy.setGmtModified(new Date());
                    saveProjectExtState = projectExtService.save(projectExtEntityCopy);
                }
            } catch (Exception e) {
                log.warn("Exception [{}]:", e.getMessage(), e);
                t.setRollbackOnly();
            }
            return saveProject && saveProjectExtState;
        }));
    }

    @Override
    public ProjectQueryVo getReadyCopyProject(Long projectId) {

        ProjectQueryVo projectQuerySourceVo = getProjectByProjectId(projectId);

        ProjectQueryVo projectQueryTargetVo = BeanUtil.copyProperties(projectQuerySourceVo, ProjectQueryVo.class);
        projectQueryTargetVo.setId(null);
        projectQueryTargetVo.setProjectName(String.format("%s_copy", projectQuerySourceVo.getProjectName()));
        projectQueryTargetVo.setState(ProjectStateEnum.INIT.getCode());
        projectQueryTargetVo.setStartTime(null);
        projectQueryTargetVo.setEndTime(null);

        PlayAttributesJsonVo playAttributes = projectQueryTargetVo.getPlayAttributes();
        if (playAttributes == null || CollectionUtils.isEmpty(playAttributes.getRules())) {
            return projectQueryTargetVo;
        }

        List<StrategyJsonVo> strategyJsonVos = playAttributes.getRules();
        List<StrategyJson> strategyJsons = BeanUtil.copyToList(strategyJsonVos, StrategyJson.class);
        List<StrategyJson> resetRules = resetRules(strategyJsons);
        projectQueryTargetVo.getPlayAttributes().setRules(BeanUtil.copyToList(resetRules, StrategyJsonVo.class));

        return projectQueryTargetVo;
    }


    private List<StrategyJson> resetRules(List<StrategyJson> rules) {
        if (CollectionUtils.isEmpty(rules)) {
            return new ArrayList<>();
        }
        for (StrategyJson rule : rules) {
            List<OptionJson> options = rule.getOptions();
            if (CollectionUtils.isNotEmpty(options)) {
                for (OptionJson option : options) {
                    option.setId(null);
                    option.setTotalStock(0L);
                    option.setUsedStock(0L);
                }
            }
            List<LayeredRuleJson> layeredRules = rule.getLayeredRules();
            if (CollectionUtils.isEmpty(layeredRules)) {
                continue;
            }

            for (LayeredRuleJson layeredRule : layeredRules) {
                List<OptionJson> optionsLayered = layeredRule.getOptions();
                if (CollectionUtils.isNotEmpty(optionsLayered)) {
                    continue;
                }
                for (OptionJson optionLayered : optionsLayered) {
                    optionLayered.setId(null);
                    optionLayered.setTotalStock(0L);
                    optionLayered.setUsedStock(0L);
                }
            }
        }
        return rules;
    }


}