package cn.com.duiba.galaxy.console.manager.impl;

import cn.com.duiba.galaxy.api.model.dto.PrototypeDto;
import cn.com.duiba.galaxy.api.model.dto.SpDto;
import cn.com.duiba.galaxy.basic.enums.DeleteFieldTypeEnum;
import cn.com.duiba.galaxy.basic.enums.PrototypeTypeEnum;
import cn.com.duiba.galaxy.basic.model.entity.ProjectEntity;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeEntity;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeJarEntity;
import cn.com.duiba.galaxy.basic.model.entity.SpEntity;
import cn.com.duiba.galaxy.basic.model.jsonfield.PlayAttributesJson;
import cn.com.duiba.galaxy.basic.model.jsonfield.SkinJson;
import cn.com.duiba.galaxy.basic.service.ProjectService;
import cn.com.duiba.galaxy.basic.service.PrototypeJarService;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.basic.service.SpService;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.enums.PrototypeStateEnum;
import cn.com.duiba.galaxy.console.manager.PrototypeManager;
import cn.com.duiba.galaxy.console.manager.SpManager;
import cn.com.duiba.galaxy.console.manager.SsoExtManager;
import cn.com.duiba.galaxy.console.model.param.SpParam;
import cn.com.duiba.galaxy.console.model.param.prototype.CrossPushPrototypeParam;
import cn.com.duiba.galaxy.console.model.param.prototype.PrototypeQueryParam;
import cn.com.duiba.galaxy.console.model.param.prototype.PrototypeSaveParam;
import cn.com.duiba.galaxy.console.model.param.prototype.PrototypeUpdateParam;
import cn.com.duiba.galaxy.console.model.param.prototype.SimplePrototypeParam;
import cn.com.duiba.galaxy.console.model.vo.PagePrototypeVo;
import cn.com.duiba.galaxy.console.model.vo.SpVo;
import cn.com.duiba.galaxy.console.model.vo.project.ProjectSimpleVo;
import cn.com.duiba.galaxy.console.model.vo.prototype.PrototypeListVo;
import cn.com.duiba.galaxy.console.model.vo.prototype.PrototypeVo;
import cn.com.duiba.galaxy.core.activity.BizPrototypeConfigMapImpl;
import cn.com.duiba.galaxy.core.activity.CheckMode;
import cn.com.duiba.galaxy.core.activity.skin.SkinTypeEnum;
import cn.com.duiba.galaxy.core.util.http.HttpClient;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import javax.annotation.Resource;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

/**
 * 原型公共服务实现
 *
 * @author fangxiaorun
 * @date 2022/9/30
 */
@Slf4j
public abstract class AbstractPrototypeManagerImpl implements PrototypeManager {
    @Autowired
    protected PrototypeService prototypeService;

    @Autowired
    protected ProjectService projectService;

    @Autowired
    protected SpService spService;

    @Autowired
    protected SpManager spManager;

    @Autowired
    protected SsoExtManager ssoExtManager;

    @Resource
    private PrototypeJarService prototypeJarService;

    @Resource
    private TransactionTemplate transactionTemplate;


    private static final String PUSH_PROTOTYPE_KEY1 = "prototypeDto";
    private static final String PUSH_PROTOTYPE_KEY2 = "spDtos";


    @Override
    public boolean newPrototype(PrototypeSaveParam param) {
        // 原型名称重复项校验
        LambdaQueryWrapper<PrototypeEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(PrototypeEntity::getPrototypeName, param.getPrototypeName());
        PrototypeEntity prototypeEntityOld = prototypeService.getOne(wrapper);
        Conditions.expectTrue(prototypeEntityOld == null, PlatformConsoleErrorEnum.PROTOTYPE_NAME_REPEAT);
        PrototypeEntity prototypeEntityNew = BeanUtil.copyProperties(param, PrototypeEntity.class);
        // 设置操作人
        prototypeEntityNew.setOperator(ssoExtManager.getAdminInfo().getName());
        return prototypeService.save(prototypeEntityNew);
    }

    @Override
    public PrototypeVo getPrototype(SimplePrototypeParam param) {
        PrototypeEntity entity = prototypeService.getPrototype(param.getId(), param.getPrototypeName());
        if (entity == null) {
            return null;
        }
        LambdaQueryWrapper<SpEntity> spEntityLambdaQueryWrapper = new LambdaQueryWrapper<>();
        // 过滤出原型下，未删除的道具
        spEntityLambdaQueryWrapper.eq(SpEntity::getPrototypeId, param.getId()).eq(SpEntity::getDeleted, DeleteFieldTypeEnum.UN_DELETE.getCode());
        List<SpEntity> spEntities = spService.list(spEntityLambdaQueryWrapper);
        PrototypeVo prototypeVo = BeanUtil.copyProperties(entity, PrototypeVo.class);
        if (CollectionUtils.isNotEmpty(spEntities)) {
            prototypeVo.setSps(BeanUtil.copyToList(spEntities, SpVo.class));
        }
        return prototypeVo;
    }

    public PagePrototypeVo listPrototypes(PrototypeQueryParam param) {


        // 查询原型
        Page<PrototypeEntity> entityPage = prototypeService.conditionAndPageQuery(param.getPrototypeCondition(), param.getProjectCondition()
                , param.getPageNum(), param.getPageSize(), param.getPrototypeType(), param.getState());
        Map<Long, List<ProjectEntity>> projectGroup = new HashMap<>();
        // 定制需要设置项目id
        boolean isCustomization = PrototypeTypeEnum.CUSTOMIZATION.getCode().equals(param.getPrototypeType());
        if (isCustomization) {
            List<Long> prototypeIds = entityPage.getRecords().stream().map(PrototypeEntity::getId).collect(Collectors.toList());
            LambdaQueryWrapper<ProjectEntity> wrapper = new LambdaQueryWrapper<>();
            wrapper.in(CollectionUtils.isNotEmpty(prototypeIds), ProjectEntity::getPrototypeId, prototypeIds);
            List<ProjectEntity> projectEntityList = projectService.list(wrapper);
            projectGroup = projectEntityList.stream().collect(Collectors.groupingBy(ProjectEntity::getPrototypeId, Collectors.toList()));
        }
        // 根据项目查询条件不空，
        List<PrototypeEntity> prototypeEntities = entityPage.getRecords();

        List<PrototypeListVo> prototypeBaseVos = BeanUtil.copyToList(prototypeEntities, PrototypeListVo.class);

        if (Objects.equals(param.getPrototypeType(), PrototypeTypeEnum.CUSTOMIZATION.getCode())) {
            for (PrototypeListVo u : prototypeBaseVos) {
                List<ProjectSimpleVo> projectSimpleVos = null;
                List<ProjectEntity> projectEntities = projectGroup.get(u.getId());
                if (CollectionUtils.isNotEmpty(projectEntities)) {
                    List<ProjectSimpleVo> projectSimpleVos1 = BeanUtil.copyToList(projectEntities, ProjectSimpleVo.class);
                    projectSimpleVos = projectSimpleVos1.stream()
                            .sorted(Comparator.comparing(ProjectSimpleVo::getId)
                                    .reversed())
                            .collect(Collectors.toList());
                }

                u.setProjectSimpleVos(projectSimpleVos);
            }
        }
        return PagePrototypeVo.newBuilder().list(prototypeBaseVos).totalCount(entityPage.getTotal()).build();
    }

    @Override
    public Boolean updatePrototype(PrototypeUpdateParam param, Future<PrototypeJarEntity> task) throws Exception {

        // 简单校验
        PrototypeEntity prototypeEntity = prototypeService.getById(param.getId());
        // 项目不存在
        Conditions.expectNotNull(prototypeEntity, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);

        // 校验原型名称是否重复
        if (!Objects.equals(param.getPrototypeName(), prototypeEntity.getPrototypeName())) {
            PrototypeEntity prototypeQuery = prototypeService.getPrototype(null, param.getPrototypeName());
            Conditions.expectTrue(prototypeQuery == null, PlatformConsoleErrorEnum.PROTOTYPE_NAME_REPEAT);
        }

        // 确定原型状态
        PrototypeStateEnum prototypeStateEnum = getPrototypeState(prototypeEntity);

        // 是为了加载和校验参数
        PrototypeEntity prototypeEntityNew = BeanUtil.copyProperties(prototypeEntity, PrototypeEntity.class, "gmtCreate", "gmtModified");
        BeanUtil.copyProperties(param, prototypeEntityNew);
        prototypeEntityNew.setGmtModified(new Date());

        if (prototypeStateEnum != null) {
            prototypeEntityNew.setState(prototypeStateEnum.getCode());
        }

        boolean update = transactionTemplate.execute(t -> {
            try {
                // 保存sp
                saveSps(param.getSps(), prototypeEntity.getId());
                // 校验
                BizPrototypeConfigMapImpl bizPrototypeConfigMap = new BizPrototypeConfigMapImpl(prototypeEntityNew);
                bizPrototypeConfigMap.findConfigErrors(new CheckMode(true, Objects.equals(prototypeEntity.getPrototypeType(), PrototypeTypeEnum.CUSTOMIZATION.getCode())));

                LambdaUpdateWrapper<PrototypeEntity> updateWrapper = new LambdaUpdateWrapper<>();
                updateWrapper.eq(PrototypeEntity::getId, param.getId());
                prototypeEntityNew.setOperator(ssoExtManager.getAdminInfo().getName());
                String commitId = savePrototypeJarEntity(task == null ? null : task.get());
                if (StringUtils.isNotBlank(commitId)) {
                    prototypeEntityNew.setCommitId(commitId);
                }
                return prototypeService.update(prototypeEntityNew, updateWrapper);
            } catch (Exception e) {
                log.warn("Exception [{}]:", e.getMessage(), e);
                t.setRollbackOnly();
                return false;
            }
        });
        return update;
    }


    /**
     * 保存原型javacode jar
     *
     * @param prototypeJarEntity 原型jarentity
     * @return boolean
     */
    private String savePrototypeJarEntity(PrototypeJarEntity prototypeJarEntity) {
        //如果任务返回的entity存在jar说明代码变更，更新jar表数据
        if (prototypeJarEntity != null && prototypeJarEntity.getPrototypeJar() != null) {
            LambdaQueryWrapper<PrototypeJarEntity> wrapper = new LambdaQueryWrapper<>();
            Long protoytypeId = prototypeJarEntity.getPrototypeId();
            String commitId = prototypeJarEntity.getCommitId();
            wrapper.eq(PrototypeJarEntity::getPrototypeId, protoytypeId);
            PrototypeJarEntity jarEntity = new PrototypeJarEntity();
            jarEntity.setPrototypeId(protoytypeId);
            jarEntity.setSourceId("1");
            jarEntity.setCommitId(commitId);
            jarEntity.setPrototypeJar(prototypeJarEntity.getPrototypeJar());
            prototypeJarService.saveOrUpdate(jarEntity, wrapper);
            return prototypeJarEntity.getCommitId();
        }
        return null;
    }

    /**
     * 保存sps
     *
     * @param params      参数个数
     * @param prototypeId 原型id
     * @return boolean
     */
    private void saveSps(List<SpParam> params, Long prototypeId) {

        List<SpEntity> entitiesOld = spManager.listAllSps(prototypeId);
        // 默认全删
        List<SpEntity> delEntities = entitiesOld;
        // 保存或更新sp
        if (CollectionUtils.isNotEmpty(params)) {
            List<SpEntity> entities = BeanUtil.copyToList(params, SpEntity.class);
            entities.forEach(u -> u.setPrototypeId(prototypeId));
            spService.saveOrUpdateBatch(entities);
            // 删除旧的sp
            List<Long> idsNew = params.stream().map(SpParam::getId).collect(Collectors.toList());
            // 所有的
            delEntities = entitiesOld.stream().filter(u -> !idsNew.contains(u.getId())).collect(Collectors.toList());
        }

        // 设置逻辑删除
        if (CollectionUtils.isNotEmpty(delEntities)) {
            delEntities.forEach(u -> u.setDeleted(DeleteFieldTypeEnum.DELETE.getCode()));
        }
        spService.updateBatchById(delEntities);
    }


    /**
     * 获取原型状态
     *
     * @param prototypeEntity 原型entity
     * @return {@link PrototypeStateEnum}
     */
    private PrototypeStateEnum getPrototypeState(PrototypeEntity prototypeEntity) {
        PrototypeStateEnum prototypeStateEnum = null;
        if (Objects.equals(prototypeEntity.getPrototypeType(), PrototypeTypeEnum.TEMPLATE.getCode())) {
            // 判断首次发布: 变成未启用
            if (prototypeEntity.getGmtCreate().getTime() == prototypeEntity.getGmtModified().getTime()) {
                prototypeStateEnum = PrototypeStateEnum.NOT_ENABLED;
            } else if (prototypeEntity.getViewAttributes() != null && prototypeEntity.getViewAttributes().getViewConfig() != null) {
                prototypeStateEnum = PrototypeStateEnum.ENABLED;
            }
        } else {
            prototypeStateEnum = PrototypeStateEnum.ENABLED;
        }
        return prototypeStateEnum;
    }

    @Override
    public String getSkinOfIndex(Long prototypeId) {
        //  查询对应原型信息
        PrototypeEntity prototype = prototypeService.getById(prototypeId);
        Conditions.expectNotNull(prototype, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);

        return Optional.of(prototype)
                .map(PrototypeEntity::getPlayAttributes)
                .map(PlayAttributesJson::getSkins)
                .flatMap(v -> v.stream().filter(s -> Objects.equals(SkinTypeEnum.INDEX.getType(), s.getType())).findFirst())
                .map(SkinJson::getHtml)
                .orElseThrow(() -> new BizRuntimeException(PlatformConsoleErrorEnum.SKIN_NOT_FOUND));
    }

    @Override
    public Boolean copyPrototype(Long prototypeId) {


        return null;

    }

    @Override
    public Boolean pushPrototype(CrossPushPrototypeParam param) {

        PrototypeEntity prototypeEntity = prototypeService.getById(param.getPrototypeId());
        if (prototypeEntity == null) {
            throw new BizRuntimeException(PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);
        }

        PrototypeDto prototypeDto = BeanUtil.copyProperties(prototypeEntity, PrototypeDto.class);
        prototypeDto.setOperator("system");
        prototypeDto.setState(PrototypeStateEnum.TO_EDIT.getCode());
        // 随机生成原型名称
        prototypeDto.setPrototypeName(String.format(prototypeDto.getPrototypeName() + "_%s", RandomUtil.randomString(6)));
        Map<String, String> dtoMap = new HashMap<>();
        dtoMap.put(PUSH_PROTOTYPE_KEY1, JSONObject.toJSONString(prototypeDto, SerializerFeature.DisableCircularReferenceDetect));

        // 道具信息导入
        List<SpEntity> spEntities = spService.listPrototypeSp(param.getPrototypeId());
        if (CollectionUtils.isNotEmpty(spEntities)) {
            List<SpDto> spDtos = BeanUtil.copyToList(spEntities, SpDto.class);
            spDtos.forEach(u -> u.setId(null));
            dtoMap.put(PUSH_PROTOTYPE_KEY2, JSONObject.toJSONString(spDtos, SerializerFeature.DisableCircularReferenceDetect));
        }
        log.info("原型:[{}]推送request:{},url:{}", prototypeEntity.getId(), JSONObject.toJSONString(dtoMap), param.getUrl());
        try {
            StringEntity requestBody = new StringEntity(JSON.toJSONString(dtoMap), ContentType.APPLICATION_JSON);
            String result = HttpClient.doPost(param.getUrl(), requestBody);
            log.info("原型:{}，推送结果:{}", prototypeEntity.getId(), result);
        } catch (Exception e) {
            log.info("原型推送失败,prototypeId:{}", e, Optional.ofNullable(prototypeEntity.getId()).orElse(null));
            return false;
        }
        return true;

    }

    /**
     * 接收并保存原型
     *
     * @param map 地图
     * @return {@link Boolean}
     */
    @Override
    public Boolean receiveAndSavePrototype(Map<String, String> map) {
        if (map == null) {
            return false;
        }

        if (map.get(PUSH_PROTOTYPE_KEY1) == null) {
            return false;
        }
        PrototypeDto prototypeDto = JSONObject.parseObject(map.get(PUSH_PROTOTYPE_KEY1), PrototypeDto.class);
        PrototypeEntity prototypeEntity = BeanUtil.copyProperties(prototypeDto, PrototypeEntity.class);
        prototypeEntity.setId(null);
        prototypeEntity.setGmtModified(null);
        prototypeEntity.setGmtCreate(null);
        prototypeEntity.setState(PrototypeStateEnum.TO_EDIT.getCode());

        return transactionTemplate.execute(t -> {
            boolean saveSp = true;
            boolean savePrototype = false;
            try {
                savePrototype = prototypeService.save(prototypeEntity);
                if (map.get(PUSH_PROTOTYPE_KEY2) != null) {
                    List<SpDto> spDtos = JSONObject.parseArray(map.get(PUSH_PROTOTYPE_KEY2), SpDto.class);
                    List<SpEntity> spEntities = BeanUtil.copyToList(spDtos, SpEntity.class);
                    if (savePrototype && CollectionUtils.isNotEmpty(spEntities)) {
                        for (SpEntity u : spEntities) {
                            u.setId(null);
                            u.setPrototypeId(prototypeEntity.getId());
                        }
                        saveSp = spService.saveBatch(spEntities);
                    }
                }
            } catch (Exception e) {
                log.warn("Exception [{}]:", e.getMessage(), e);
                t.setRollbackOnly();
            }
            return saveSp && savePrototype;
        });
    }

    public void updatePrototypeState(Long prototypeId, Integer state) {
        //  0, "待编辑",1, "未启用",2, "已启用"
        PrototypeEntity oldEntity = prototypeService.getById(prototypeId);
        Conditions.expectNotNull(oldEntity, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);
        Conditions.expectTrue(ObjectUtils.notEqual(state, oldEntity.getState()) && ObjectUtils.notEqual(state, PrototypeStateEnum.TO_EDIT.getCode()), PlatformConsoleErrorEnum.INVALID_CHANGE);

//        PrototypeStateEnum prototypeStateEnum = null;
//        // 停用-> 启用 ： 校验原型文件
//        if (Objects.equals(oldEntity.getState(), PrototypeStateEnum.NOT_ENABLED.getCode())
//                && Objects.equals(state, PrototypeStateEnum.ENABLED.getCode())) {
//
//            ViewAttributesJson viewAttributes = oldEntity.getViewAttributes();
//            if (viewAttributes != null && viewAttributes.getViewValueConfig() != null) {
//                prototypeStateEnum = PrototypeStateEnum.ENABLED;
//            }
//
//        }
//
//        // 启用-> 停用：校验管理的活动是否有在线的
//        if (Objects.equals(oldEntity.getState(), PrototypeStateEnum.ENABLED.getCode())
//                && Objects.equals(state, PrototypeStateEnum.NOT_ENABLED.getCode())) {
//            LambdaQueryWrapper<ProjectEntity> projectQuery = new LambdaQueryWrapper<>();
//            projectQuery.eq(ProjectEntity::getPrototypeId, prototypeId)
//                    .eq(ProjectEntity::getState, ProjectStateEnum.ON_LINE.getCode());
//            List<ProjectEntity> projectEntities = projectService.list(projectQuery);
//            if (CollectionUtils.isEmpty(projectEntities)) {
//                prototypeStateEnum = PrototypeStateEnum.NOT_ENABLED;
//            }
//        }
//
//        if (prototypeStateEnum != null) {
        LambdaUpdateWrapper<PrototypeEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(PrototypeEntity::getId, prototypeId)
                .set(PrototypeEntity::getState, state)
                .set(PrototypeEntity::getOperator, ssoExtManager.getAdminInfo().getName());
        Conditions.expectTrue(prototypeService.update(updateWrapper), PlatformConsoleErrorEnum.STATE_CHANGE_ERROR);
    }

}