package cn.com.duiba.galaxy.console.manager.impl;

import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.manager.CommonBackendManager;
import cn.com.duiba.galaxy.console.model.vo.EnumFieldVo;
import cn.com.duiba.galaxy.console.model.vo.EnumVo;
import cn.com.duiba.galaxy.console.util.PackageUtils;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.message.BizError;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestParam;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * 通用服务
 *
 * @author fangxiaorun
 * @date 2022/9/30
 */
@Slf4j
@Service
public class CommonBackendManagerImpl implements CommonBackendManager {

    @Override
    public List<EnumVo> getEnumByName(@RequestParam(required = false) String name) {
        String[] enumPaths = {"cn.com.duiba.galaxy.console.enums", "cn.com.duiba.galaxy.basic.enums"};

        List<EnumVo> resultList = Lists.newArrayList();

        for (String enumPath : enumPaths) {
            Set<Class<?>> scan = new HashSet<>();
            try {
                if (StringUtils.isNotBlank(name)) {
                    scan.add(Class.forName(String.format("%s.%s", enumPath, name)));
                } else {
                    scan = PackageUtils.getClasses(enumPath);
                }
            } catch (ClassNotFoundException e) {
                throw new BizRuntimeException(PlatformConsoleErrorEnum.ENUM_NOT_FOUND, e);
            } catch (IOException e) {
                throw new BizRuntimeException(PlatformConsoleErrorEnum.GET_ENUM_ERROR, e);
            }

            for (Class<?> classes : scan) {
                //  错误枚举跳过
                if (BizError.class.isAssignableFrom(classes)) {
                    continue;
                }
                //  得到所有枚举常量
                Object[] objects = classes.getEnumConstants();
                //  得到所有枚举常量
                Method getCode = getQualifiedMethod(classes, "getCode");
                Method getDesc = getQualifiedMethod(classes, "getDesc");
                if (Objects.isNull(getCode) || Objects.isNull(getDesc)) {
                    continue;
                }
                List<EnumFieldVo> valueList = Lists.newArrayList();
                try {
                    for (Object obj : objects) {
                        valueList.add(EnumFieldVo.builder()
                                .code((Integer) getCode.invoke(obj))
                                .desc((String) getDesc.invoke(obj))
                                .build());
                    }
                    resultList.add(EnumVo.builder().enumName(classes.getSimpleName()).valueList(valueList).build());
                } catch (Exception e) {
                    log.info("根据枚举类名获取枚举常量,枚举类[{}]方法[{}]执行失败,方法将忽略该枚举类,如有需要请先按照接口规范调整属性名！", classes.getSimpleName(), name, e);
                }
            }
        }
        return resultList;
    }

    /**
     * 获取限定方法
     *
     * @param classes classes
     * @param name    获取编码
     * @return {@link Method}
     */
    private static Method getQualifiedMethod(Class<?> classes, String name) {
        try {
            return classes.getMethod(name);
        } catch (Exception e) {
            log.info("根据枚举类名获取枚举常量,枚举类[{}]缺失方法[{}],方法将忽略该枚举类,如有需要请先按照接口规范调整属性名！", classes.getSimpleName(), name);
            return null;
        }
    }
}