package cn.com.duiba.galaxy.console.manager.impl;

import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.galaxy.basic.enums.PrototypeTypeEnum;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeEntity;
import cn.com.duiba.galaxy.basic.model.entity.TemplateEntity;
import cn.com.duiba.galaxy.basic.model.jsonfield.ViewAttributesJson;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.basic.service.TemplateService;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.enums.PrototypeStateEnum;
import cn.com.duiba.galaxy.console.enums.TemplateSyncStateEnum;
import cn.com.duiba.galaxy.console.enums.TemplateTypeEnum;
import cn.com.duiba.galaxy.console.manager.TemplateBackendManager;
import cn.com.duiba.galaxy.console.model.param.template.TemplatePageQueryParam;
import cn.com.duiba.galaxy.console.model.param.template.TemplateSaveParam;
import cn.com.duiba.galaxy.console.model.param.template.TemplateSyncParam;
import cn.com.duiba.galaxy.console.model.vo.PageList;
import cn.com.duiba.galaxy.console.model.vo.template.TemplateVo;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 模版管理
 *
 * @author fangxiaorun
 * @date 2022/9/30
 */
@Slf4j
@Service
public class TemplateBackendManagerImpl implements TemplateBackendManager {
    @Autowired
    private PrototypeService prototypeService;

    @Autowired
    private TemplateService templateService;

    @Resource
    private RemoteAppService remoteAppService;

    @Override
    public TemplateVo getById(Long id) {
        //  模版是否存在；
        TemplateEntity template = templateService.getById(id);
        Conditions.expectNotNull(template, PlatformConsoleErrorEnum.TEMPLATE_NOT_FOUND);

        //  查询对应原型信息
        PrototypeEntity prototype = prototypeService.getById(template.getPrototypeId());
        Conditions.expectNotNull(prototype, PlatformConsoleErrorEnum.TEMPLATE_PROTOTYPE_NOT_FOUND);

        TemplateVo templateVO = BeanUtil.copyProperties(template, TemplateVo.class);
        templateVO.setPrototypeState(prototype.getState());
        templateVO.setTemplatePoster(Optional.of(prototype)
                .map(PrototypeEntity::getViewAttributes)
                .map(ViewAttributesJson::getTemplatePoster)
                .orElse(""));
        return templateVO;
    }

    @Override
    public PageList<TemplateVo> list(TemplatePageQueryParam param) {
        //  查询模版列表
        LambdaQueryWrapper<TemplateEntity> templateQueryWrapper = new LambdaQueryWrapper<TemplateEntity>()
                .like(StringUtils.isNotBlank(param.getTemplateName()), TemplateEntity::getTemplateName, param.getTemplateName())
                .eq(Objects.nonNull(param.getTemplateType()), TemplateEntity::getTemplateType, param.getTemplateType())
                .orderByDesc(TemplateEntity::getId);
        IPage<TemplateEntity> page = templateService.page(new Page<>(param.getPageNum(), param.getPageSize()), templateQueryWrapper);
        if (CollectionUtils.isEmpty(page.getRecords())) {
            return PageList.of(page.getTotal(), page.getSize(), Collections.emptyList());
        }

        //  查询对应原型信息
        List<Long> prototypeIdList = page.getRecords().stream().map(TemplateEntity::getPrototypeId).collect(Collectors.toList());
        List<PrototypeEntity> prototypeList = prototypeService.listByIds(prototypeIdList);
        Map<Long, PrototypeEntity> templatePosterMap = prototypeList.stream().collect(Collectors.toMap(PrototypeEntity::getId, Function.identity(), (o, n) -> n));

        // 最终返回分页对象
        return PageList.of(page, result -> {
            TemplateVo templateVO = BeanUtil.copyProperties(result, TemplateVo.class);
            Optional.ofNullable(templatePosterMap.get(templateVO.getPrototypeId())).ifPresent(prototype -> {
                templateVO.setPrototypeState(prototype.getState());
                templateVO.setTemplatePoster(Optional.of(prototype)
                        .map(PrototypeEntity::getViewAttributes)
                        .map(ViewAttributesJson::getTemplatePoster)
                        .orElse(""));
            });
            return templateVO;
        });
    }

    @Override
    public Boolean save(TemplateSaveParam param) {
        //  校验模版类型是否合法；
        Conditions.expectNotNull(TemplateTypeEnum.valueOfType(param.getTemplateType()), PlatformConsoleErrorEnum.TEMPLATE_TYPE_NOT_FOUND);

        //  校验关联原型是否存在、是否启用、类型是否匹配；
        PrototypeEntity prototype = prototypeService.getById(param.getPrototypeId());
        Conditions.expectNotNull(prototype, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);
        Conditions.expectTrue(Objects.equals(prototype.getState(), PrototypeStateEnum.ENABLED.getCode()), PlatformConsoleErrorEnum.PROTOTYPE_NOT_ENABLED);
        Conditions.expectTrue(Objects.equals(prototype.getPrototypeType(), PrototypeTypeEnum.TEMPLATE.getCode()), PlatformConsoleErrorEnum.PROTOTYPE_NOT_MATCH);

        //  所选择原型是否已创建模版；
        LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<TemplateEntity>().eq(TemplateEntity::getPrototypeId, param.getPrototypeId());
        Conditions.expectTrue(templateService.count(queryWrapper) <= 0, PlatformConsoleErrorEnum.PROTOTYPE_HAS_BEEN_ASSOCIATED);

        //  模版名称默认使用所关联的原型名称；
        TemplateEntity entity = BeanUtil.copyProperties(param, TemplateEntity.class);
        entity.setTemplateName(prototype.getPrototypeName());
        return templateService.save(entity);
    }

    @Override
    public Boolean sync(TemplateSyncParam param) {
        //  模版是否存在；
        TemplateEntity template = templateService.getById(param.getTemplateId());
        Conditions.expectNotNull(template, PlatformConsoleErrorEnum.TEMPLATE_NOT_FOUND);

        // 无效状态变更拦截；
        TemplateSyncStateEnum paramTypeEnum = TemplateSyncStateEnum.valueOfType(param.getSyncType());
        TemplateSyncStateEnum currentTypeEnum = TemplateSyncStateEnum.valueOfType(template.getSyncType());
        Conditions.expectFalse(paramTypeEnum == currentTypeEnum && (paramTypeEnum == TemplateSyncStateEnum.INIT || paramTypeEnum == TemplateSyncStateEnum.FULL_SYNC)
                , PlatformConsoleErrorEnum.INVALID_CHANGE);

        //  校验关联原型是否存在、是否启用；
        PrototypeEntity prototype = prototypeService.getById(template.getPrototypeId());
        Conditions.expectNotNull(prototype, PlatformConsoleErrorEnum.PROTOTYPE_NOT_FOUND);
        Conditions.expectTrue(Objects.equals(prototype.getState(), PrototypeStateEnum.ENABLED.getCode()), PlatformConsoleErrorEnum.PROTOTYPE_NOT_ENABLED);

        //  局部同步时，校验APP信息
        String appIdListStr = null;
        if (paramTypeEnum == TemplateSyncStateEnum.LOCAL_SYNC) {
            List<Long> appIdList = param.getSyncApp().stream().map(Long::valueOf).collect(Collectors.toList());
            List<Long> appIdListFromDb = remoteAppService.findByIds(appIdList).stream().map(AppSimpleDto::getId).collect(Collectors.toList());
            if (appIdList.size() != appIdListFromDb.size()) {
                List<Long> subtract = (List<Long>) CollectionUtils.subtract(appIdList, appIdListFromDb);
                log.warn("同步模版失败，以下app信息未检索到：{}", subtract);
                throw new BizRuntimeException(PlatformConsoleErrorEnum.INVALID_APP.setDesc(String.format("请输入正确的应用ID，应用ID%s不存在", subtract)));
            }
            appIdListStr = Optional.ofNullable(param.getSyncApp()).map(Object::toString).orElse(null);
        }

        //  保存模版同步信息；
        LambdaUpdateWrapper<TemplateEntity> lambdaUpdateWrapper = new LambdaUpdateWrapper<>();
        lambdaUpdateWrapper.eq(TemplateEntity::getId, param.getTemplateId())
                .set(TemplateEntity::getSyncType, param.getSyncType())
                .set(TemplateEntity::getSyncApp, appIdListStr);
        return templateService.update(null, lambdaUpdateWrapper);
    }
}