package cn.com.duiba.galaxy.console.manager.impl.credits;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.consumer.center.api.dto.ConsumerDto;
import cn.com.duiba.credits.sdk.CreditTool;
import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.domain.dto.DomainConfigDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.developer.center.api.remoteservice.RemoteDomainConfigService;
import cn.com.duiba.galaxy.api.model.param.ProjectNewOrUpdateParam;
import cn.com.duiba.galaxy.basic.model.entity.ProjectEntity;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeEntity;
import cn.com.duiba.galaxy.basic.model.entity.TemplateEntity;
import cn.com.duiba.galaxy.basic.service.ProjectExtService;
import cn.com.duiba.galaxy.basic.service.ProjectService;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.basic.service.TemplateService;
import cn.com.duiba.galaxy.basic.util.Environment;
import cn.com.duiba.galaxy.console.enums.IdentityTypeEnum;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.console.manager.ProjectManager;
import cn.com.duiba.galaxy.console.manager.impl.AbstractProjectManagerImpl;
import cn.com.duiba.galaxy.console.model.param.project.ProjectNewParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectQueryParam;
import cn.com.duiba.galaxy.console.model.param.project.ProjectUpdateParam;
import cn.com.duiba.galaxy.console.model.vo.IdentityChildrenVo;
import cn.com.duiba.galaxy.console.model.vo.IdentityVo;
import cn.com.duiba.galaxy.console.model.vo.project.PageProjectVo;
import cn.com.duiba.galaxy.console.model.vo.project.ProjectListVo;
import cn.com.duiba.galaxy.core.annotation.ConditionalOnBizScene;
import cn.com.duiba.galaxy.core.enums.BizSceneEnum;
import cn.com.duiba.galaxy.core.enums.OpenbsStateEnum;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;


/**
 * 项目差异化服务实现（积分商城1.0）
 *
 * @author fangxiaorun
 * @date 2022/09/30
 */
@Service
@Slf4j
@ConditionalOnBizScene(BizSceneEnum.CREDITS_MALL_V1)
public class CreditsProjectManagerImpl extends AbstractProjectManagerImpl implements ProjectManager {

    @Autowired
    private ProjectService projectService;
    @Autowired
    private PrototypeService prototypeService;
    @Autowired
    private TemplateService templateService;
    @Autowired
    private ProjectExtService projectExtService;
    @Resource
    private RemoteDomainConfigService remoteDomainConfigService;
    @Resource
    private RemoteAppService remoteAppService;

    @Override
    public Boolean updateAppId(Long projectId, Long appId) {
        LambdaUpdateWrapper<ProjectEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(ProjectEntity::getId, projectId);
        updateWrapper.set(ProjectEntity::getAppId, appId);
        return projectService.update(updateWrapper);
    }

    @Override
    public PageProjectVo listProjects(ProjectQueryParam queryParam) {

        Page<ProjectEntity> projectEntityPage = projectService.conditionAndPageQuery(queryParam.getPrototypeCondition(), queryParam.getProjectCondition(),
                queryParam.getProd(), queryParam.getOperator(), queryParam.getPageNum(), queryParam.getPageSize());
        PageProjectVo pageProjectVo = new PageProjectVo();

        if (projectEntityPage.getTotal() == 0) {
            return pageProjectVo;
        }

        // 找到对应的prototypeId
        List<ProjectEntity> projectEntities = projectEntityPage.getRecords();
        Map<Long, ProjectEntity> projectEntityMap = projectEntities.stream().collect(Collectors.toMap(ProjectEntity::getId, x -> x));

        Set<Long> prototypeIds = projectEntities.stream().map(ProjectEntity::getPrototypeId).collect(Collectors.toSet());

        // 查询对应的原型名称和appId
        List<PrototypeEntity> prototypeEntities = prototypeService.listPrototypeEntity(prototypeIds);
        Map<Long, PrototypeEntity> prototypeEntityMap = prototypeEntities.stream().collect(Collectors.toMap(PrototypeEntity::getId, s -> s));

        // 设置prototypeName
        List<ProjectListVo> projectListVos = BeanUtil.copyToList(projectEntityPage.getRecords(), ProjectListVo.class);
        for (ProjectListVo projectListVo : projectListVos) {
            PrototypeEntity prototypeEntity = prototypeEntityMap.get(projectListVo.getPrototypeId());
            // 原型名称
            projectListVo.setPrototypeName(prototypeEntity.getPrototypeName());
            // 原型类型
            projectListVo.setPrototypeType(prototypeEntity.getPrototypeType());
            // 设置openbs开关状态
            ProjectEntity projectEntity = projectEntityMap.get(projectListVo.getId());

            projectListVo.setEnableOpenBs(Objects.equals(projectEntity.getOpenbs(), OpenbsStateEnum.CLOSE.getCode()));
            // 1正式 0非正式
            projectListVo.setProd(projectEntity.getProd() == 1);

        }
        projectListVos.forEach(u -> u.setPrototypeName(String.valueOf(prototypeEntityMap.get(u.getPrototypeId()).getPrototypeName())));

        pageProjectVo.setTotalCount(projectEntityPage.getTotal());
        pageProjectVo.setList(projectListVos);
        return pageProjectVo;
    }

    @Override
    @Transactional
    public Long createForMall(ProjectNewOrUpdateParam param) throws BizException {
        Long projectId = param.getId();
        if (Objects.isNull(projectId)) {
            //  新建,对应模版是否存在、是否可用
            // 创建活动的流程区别于活动平台，并非先创建活动，再填充活动内容，而是同时完成创建和更新
            TemplateEntity template = templateService.listByAppId(param.getAppId())
                    .stream().filter(v -> Objects.equals(param.getTemplateId(), v.getId()))
                    .findFirst()
                    .orElse(null);
            Conditions.expectNotNull(template, PlatformConsoleErrorEnum.TEMPLATE_NOT_FOUND);

            projectId = this.newProject(BeanUtil.copyProperties(param, ProjectNewParam.class));
            Conditions.expectNotNull(projectId, PlatformConsoleErrorEnum.NEW_PROJECT_ERROR);

            // 预设部分字段，便于后续执行更新逻辑
            ProjectEntity projectEntity = projectService.getById(projectId);
            param.setId(projectId);
            param.setGmtModified(projectEntity.getGmtModified());

            //  定向app
            this.updateAppId(projectId, param.getAppId());
        }
        //  编辑
        Conditions.expectTrue(updateForMall(param), PlatformConsoleErrorEnum.NEW_PROJECT_ERROR);
        return projectId;
    }

    @Override
    public Boolean updateForMall(ProjectNewOrUpdateParam param) throws BizException {
        return this.updateProject(BeanUtil.copyProperties(param, ProjectUpdateParam.class));
    }


    /**
     * 获取首页访问url
     *
     * @param projectId 项目id
     * @return {@link String}
     */
    @Override
    public String getIndexUrl(Long projectId, Long appId) {
        DubboResult<DomainConfigDto> domainConfig = remoteDomainConfigService.findDomainConfig(appId);
        DomainConfigDto config = domainConfig.getResult();
        return StringUtils.join(getScheme(), config.getActivityDomain(), "/galaxy/app/project/", projectId, "/index.html");
    }

    @Override
    public IdentityVo listAllIdentityByType(Integer type, Long appIp) {
        IdentityVo identityVo = new IdentityVo();
        identityVo.setType(type);
        identityVo.setName(IdentityTypeEnum.getIdentityTypeEnumByCode(type).getDesc());

        if (Objects.equals(type, IdentityTypeEnum.MEMBER_LEVEL.getCode())) {
            DubboResult<AppSimpleDto> result = remoteAppService.getSimpleApp(appIp);
            if (result.isSuccess()) {
                AppSimpleDto dto = result.getResult();
                String vipLevelConfig = dto.getVipLevelConfig();
                List<IdentityChildrenVo> childrenVos = JSONObject.parseArray(vipLevelConfig, IdentityChildrenVo.class);
                identityVo.setChildrenList(childrenVos);
                return identityVo;
            }
        } else if (Objects.equals(type, IdentityTypeEnum.USER_TAG.getCode())) {
            // todo 预留，待实现
        }
        return identityVo;
    }

    @Override
    public String getPreviewUrl(Long projectId) {
        ProjectEntity projectEntity = projectService.getById(projectId);
        Conditions.expectNotNull(projectEntity, PlatformConsoleErrorEnum.NULL_PROJECT);
        Conditions.expectNotNull(projectEntity.getAppId(), PlatformConsoleErrorEnum.NOT_DIRECTED_APP);

        AppSimpleDto app = Optional.ofNullable(remoteAppService.getSimpleApp(projectEntity.getAppId()))
                .map(DubboResult::getResult)
                .orElseThrow(() -> new BizRuntimeException(PlatformConsoleErrorEnum.APP_NOT_FOUNT));

        Long appId = app.getId();

        DomainConfigDto domian = Optional.ofNullable(remoteDomainConfigService.findDomainConfig(appId))
                .map(DubboResult::getResult)
                .orElseThrow(() -> new BizRuntimeException(PlatformConsoleErrorEnum.DOMIAN_NOT_FOUND));

        Map<String, String> params = Maps.newHashMap();
        params.put("uid", ConsumerDto.PREVIEWUSERID);
        params.put("timestamp", Long.toString(System.currentTimeMillis()));
        params.put("redirect", getIndexUrl(projectId, appId));
        return new CreditTool(app.getAppKey(), app.getAppSecret()).buildUrlWithSign(String.format("%s%s/autoLogin/autologin?", getScheme(), domian.getActivityDomain()), params);
    }


    private String getScheme() {
        return Environment.isProdEnv() ? "https:" : "http:";
    }
}
