package cn.com.duiba.galaxy.console.manager.impl.credits;

import cn.com.duiba.galaxy.console.manager.impl.AbstractRegionManagerImpl;
import cn.com.duiba.galaxy.console.model.vo.RegionVo;
import cn.com.duiba.galaxy.core.annotation.ConditionalOnBizScene;
import cn.com.duiba.galaxy.core.enums.BizSceneEnum;
import cn.com.duiba.geo.api.dto.AdministrativeDivisionDto;
import cn.com.duiba.geo.api.remoteservice.RemoteAdministrativeDivisionService;
import cn.hutool.core.bean.BeanUtil;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author zhangdaqing
 * @Date 2022/10/20 14:02
 **/
@Service
@ConditionalOnBizScene(BizSceneEnum.CREDITS_MALL_V1)
public class CreditsRegionManagerImpl extends AbstractRegionManagerImpl {
    @Autowired
    private RemoteAdministrativeDivisionService remoteAdministrativeDivisionService;
    //新疆编码前缀
    private static final String XINJIANGPREFIXCODE = "65";

    //新疆省直辖县级市编码前缀
    private static final String SPECIALPREFIXCODE = "6590";

    private static final String REGION = "region";

    private Cache<String, List<RegionVo>> regionCache =
            CacheBuilder.newBuilder().expireAfterWrite(1, TimeUnit.DAYS).maximumSize(50).build();


    /**
     * 获取省市列表，直辖市取到三级
     *
     * @return {@link List}<{@link RegionVo}>
     */
    @Override
    public List<RegionVo> getProvinceAndCity() {
        List<RegionVo> cache = regionCache.getIfPresent(REGION);
        if (CollectionUtils.isNotEmpty(cache)) {
            return cache;
        }
        List<RegionVo> regionVos = Lists.newArrayList();
        List<AdministrativeDivisionDto> provinceList = remoteAdministrativeDivisionService.findRegionDivision();
        if (CollectionUtils.isEmpty(provinceList)) {
            return regionVos;
        }
        provinceList.forEach(divisionDto -> {
            RegionVo regionVo = new RegionVo();
            regionVo.setAdCode(divisionDto.getAdCode());
            regionVo.setName(divisionDto.getName());
            List<AdministrativeDivisionDto> cityList = remoteAdministrativeDivisionService.findAdministrativeDivisionByParentAdCode(divisionDto.getAdCode());
            if (CollectionUtils.isNotEmpty(cityList)) {
                if (divisionDto.getAdCode().startsWith(XINJIANGPREFIXCODE)) {
                    cityList = removeSpecialXinJiang(cityList);
                }
                regionVo.setChildren(BeanUtil.copyToList(cityList, RegionVo.class));
            }
            regionVos.add(regionVo);
        });
        //放入缓存
        regionCache.put(REGION, regionVos);
        return regionVos;
    }


    //移除新疆特殊的省直辖县级市。如昆玉市等
    public List<AdministrativeDivisionDto> removeSpecialXinJiang(List<AdministrativeDivisionDto> cityList) {
        List<AdministrativeDivisionDto> administrativeDivisionDtos = Lists.newArrayList();
        for (AdministrativeDivisionDto administrativeDivisionDto : cityList) {
            if (!administrativeDivisionDto.getAdCode().startsWith(SPECIALPREFIXCODE)) {
                administrativeDivisionDtos.add(administrativeDivisionDto);
            }
        }
        return administrativeDivisionDtos;
    }
}
