package cn.com.duiba.galaxy.console.model.vo;

import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.metadata.IPage;
import lombok.AllArgsConstructor;
import lombok.Getter;

import java.io.Serializable;
import java.util.List;
import java.util.function.Function;

import static java.util.stream.Collectors.toList;

/**
 * 简单分页模型
 *
 * @author fangxiaorun
 * @date 2022/10/09
 */
@Getter
@AllArgsConstructor
public class PageList<T> implements Serializable {

    private static final long serialVersionUID = 5001287295240910627L;

    /**
     * 查询数据列表
     */
    private List<T> list;

    /**
     * 记录总数
     */
    private Long totalCount;

    /**
     * 总页数
     */
    private Long totalPages;


    /**
     * 静态构造器
     *
     * @param totalCount 记录总数
     * @param pageSize   每页记录数
     * @param list       数据列表
     * @return {@link PageList}<{@link T}>
     */
    public static <T> PageList<T> of(long totalCount, long pageSize, List<T> list) {
        long totalPages = getTotalPages(totalCount, pageSize);
        return new PageList<>(list, totalCount, totalPages);
    }

    /**
     * 静态构造器
     *
     * @param page      mybatis-plus 分页对象
     * @param destClass 目标类型
     * @return {@link PageList}<{@link R}>
     */
    public static <T, R> PageList<R> of(IPage<T> page, Class<R> destClass) {
        long totalCount = page.getTotal();
        long pageSize = page.getSize();
        long totalPages = getTotalPages(totalCount, pageSize);
        return new PageList<>(BeanUtil.copyToList(page.getRecords(), destClass), totalCount, totalPages);
    }

    /**
     * 静态构造器
     *
     * @param page   mybatis-plus 分页对象
     * @param mapper 自定义类型转换函数
     * @return {@link PageList}<{@link R}>
     */
    public static <T, R> PageList<R> of(IPage<T> page, Function<? super T, ? extends R> mapper) {
        long totalCount = page.getTotal();
        long pageSize = page.getSize();
        long totalPages = getTotalPages(totalCount, pageSize);
        List<R> list = page.getRecords().stream().map(mapper).collect(toList());
        return new PageList<>(list, totalCount, totalPages);
    }

    /**
     * 获取页数
     *
     * @param totalCount 总记录数
     * @param pageSize   每页数量
     * @return long
     */
    private static long getTotalPages(long totalCount, long pageSize) {
        if (pageSize == 0) {
            return 0L;
        }
        long totalPages = totalCount / pageSize;
        if (totalCount % pageSize != 0) {
            totalPages++;
        }
        return totalPages;
    }
}
