package cn.com.duiba.galaxy.console.remote;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.galaxy.api.model.dto.TemplateDto;
import cn.com.duiba.galaxy.api.remote.RemoteTemplateBackendService;
import cn.com.duiba.galaxy.basic.model.entity.PrototypeEntity;
import cn.com.duiba.galaxy.basic.model.entity.TemplateEntity;
import cn.com.duiba.galaxy.basic.model.jsonfield.ViewAttributesJson;
import cn.com.duiba.galaxy.basic.service.PrototypeService;
import cn.com.duiba.galaxy.basic.service.TemplateService;
import cn.com.duiba.galaxy.console.enums.PlatformConsoleErrorEnum;
import cn.com.duiba.galaxy.sdk.exception.BizRuntimeException;
import cn.com.duiba.galaxy.sdk.utils.Conditions;
import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * "模版"远程服务实现
 *
 * @author fangxiaorun
 * @date 2022/10/11
 */
@RestController
@Slf4j
public class RemoteTemplateBackendServiceImpl implements RemoteTemplateBackendService {

    @Autowired
    private TemplateService templateService;

    @Autowired
    private PrototypeService prototypeService;

    @Override
    public List<TemplateDto> listByAppId(Long appId) throws BizException {
        try {
            //  查询模版列表
            List<TemplateEntity> list = templateService.listByAppId(appId);
            Conditions.expectTrue(CollectionUtils.isNotEmpty(list), PlatformConsoleErrorEnum.NO_AUTH_TEMPLATE);

            //  查询对应原型信息
            List<Long> prototypeIdList = list.stream().map(TemplateEntity::getPrototypeId).collect(Collectors.toList());
            List<PrototypeEntity> prototypeList = prototypeService.listByIds(prototypeIdList);
            Map<Long, PrototypeEntity> templatePosterMap = prototypeList.stream().collect(Collectors.toMap(PrototypeEntity::getId, Function.identity(), (o, n) -> n));

            List<TemplateDto> dtos = BeanUtil.copyToList(list, TemplateDto.class);
            dtos.forEach(v -> v.setTemplatePoster(Optional.ofNullable(templatePosterMap.get(v.getPrototypeId()))
                    .map(PrototypeEntity::getViewAttributes)
                    .map(ViewAttributesJson::getTemplatePoster)
                    .orElse(null)));
            return dtos;
        } catch (BizRuntimeException e) {
            throw new BizException(e.getMessage()).withCode(e.getCode());
        } catch (Exception e) {
            log.error("Exception [{}]\nappId={}:", e.getMessage(), appId, e);
            PlatformConsoleErrorEnum consoleEx = PlatformConsoleErrorEnum.SYSTEM_ERROR;
            throw new BizException(consoleEx.getDesc()).withCode(consoleEx.getCode());
        }
    }

    @Override
    public TemplateDto getByPrototypeId(Long prototypeId) throws BizException {
        try {
            //  查询模版列表
            LambdaQueryWrapper<TemplateEntity> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(TemplateEntity::getPrototypeId, prototypeId);
            return Optional.ofNullable(templateService.getOne(queryWrapper))
                    .map(v -> BeanUtil.copyProperties(v, TemplateDto.class))
                    .orElse(null);
        } catch (BizRuntimeException e) {
            throw new BizException(e.getMessage()).withCode(e.getCode());
        } catch (Exception e) {
            log.error("Exception [{}]\nprototypeId={}:", e.getMessage(), prototypeId, e);
            PlatformConsoleErrorEnum consoleEx = PlatformConsoleErrorEnum.SYSTEM_ERROR;
            throw new BizException(consoleEx.getDesc()).withCode(consoleEx.getCode());
        }
    }
}