package cn.com.duiba.galaxy.sdk.component.inviteassist;

import cn.com.duiba.galaxy.sdk.PlatformSdkErrorEnum;
import cn.com.duiba.galaxy.sdk.UserRequestApi;
import cn.com.duiba.galaxy.sdk.annotation.CustomRequestAction;
import cn.com.duiba.galaxy.sdk.annotation.ProjectCycleCheck;
import cn.com.duiba.galaxy.sdk.annotation.UserConcurrentLock;
import cn.com.duiba.galaxy.common.exception.BizRuntimeException;
import cn.com.duiba.galaxy.common.utils.PageList;
import cn.com.duiba.galaxy.sdk.annotation.enums.ProjectCycleCheckTypeEnum;
import cn.com.duiba.galaxy.sdk.api.tools.inner.lock.DistributedLock;
import cn.com.duiba.wolf.utils.DateUtils;

import java.util.Date;


/**
 * 邀请助力组件
 * <p>
 * 注意📢：此处需要自行处理并发问题
 *
 * @author 刘佳辉
 * @since 2023/2/1
 */
public class InviteAssistComponentActionDefault extends InviteAssistComponentAction {

    private final String componentId;

    public InviteAssistComponentActionDefault(String componentId) {
        this.componentId = componentId;
    }

    @Override
    @UserConcurrentLock
    @ProjectCycleCheck
    @CustomRequestAction(id = "getInviteCode", desc = "获取邀请码")
    public InviteResult getInviteCode(UserRequestApi api) {
        return api.getComponentApi().getInviteAssistApi().getInviteCode(componentId);
    }

    @Override
    @UserConcurrentLock
    @ProjectCycleCheck
    @CustomRequestAction(id = "getInviteCodeCoverPeriod", desc = "获取邀请码并刷新已过期邀请码时间")
    public InviteResult getInviteCodeCoverPeriod(UserRequestApi api) {
        return api.getComponentApi().getInviteAssistApi().getInviteCodeCoverPeriod(componentId);
    }

    @Override
    @ProjectCycleCheck
    @CustomRequestAction(id = "doAssist", desc = "助力")
    public AssistResult doAssist(UserRequestApi api) {
        String inviteCode = api.getStringParameterCheckBlank("inviteCode");
        String extra = api.getStringParameter("extra");
        DistributedLock distributedLock = api.getLockApi().newLock(componentId+inviteCode);
        if(distributedLock.tryLock()){
            return api.getComponentApi().getInviteAssistApi().doAssist(componentId, inviteCode, extra);
        }
        throw new BizRuntimeException(PlatformSdkErrorEnum.LOCK_ERROR.setDesc("助力失败，请稍后再试"));
    }

    @Override
    @CustomRequestAction(id = "queryInviteRecords", desc = "查询邀请记录")
    public PageList<InviteRecordResult> queryInviteRecords(UserRequestApi api) {
        InviteAssistQueryParam param = getRecordQueryParam(api);

        return api.getComponentApi().getInviteAssistApi().queryInviteRecords(param);
    }

    @Override
    @CustomRequestAction(id = "queryAssistRecords", desc = "查询助力记录")
    public PageList<AssistRecordResult> queryAssistRecords(UserRequestApi api) {

        InviteAssistQueryParam param = getRecordQueryParam(api);

        return api.getComponentApi().getInviteAssistApi().queryAssistRecords(param);
    }


    @Override
    @CustomRequestAction(id = "queryNewInviteCount", desc = "查询未读邀请数量")
    public int queryNewInviteCount(UserRequestApi api) {
        return api.getComponentApi().getInviteAssistApi().queryNewInviteCount(componentId, api.getUserId());
    }


    private InviteAssistQueryParam getRecordQueryParam(UserRequestApi api) {
        Integer pageNum = api.getIntegerParameterCheckNull("pageNum");
        Integer pageSize = api.getIntegerParameterCheckNull("pageSize");
        Long startDate = api.getLongParameter("startDate");
        Long endDate = api.getLongParameter("endDate");
        Integer distinctAssistUserId = api.getIntegerParameter("distinctAssistUserId", 0);
        Integer distinctShareUserId = api.getIntegerParameter("distinctShareUserId", 0);

        InviteAssistQueryParam param = new InviteAssistQueryParam();
        param.setComponentId(componentId);
        param.setProjectId(api.getActionContext().getProjectId());
        param.setDistinctAssistUserId(distinctAssistUserId.equals(1));
        param.setDistinctShareUserId(distinctShareUserId.equals(1));
        param.setPageNum(pageNum);
        param.setPageSize(pageSize);
        if (startDate != null) {
            param.setStartTime(new Date(startDate));
        } else {
            param.setStartTime(DateUtils.getDayStartTime(new Date()));
        }
        if (endDate != null) {
            param.setEndTime(new Date(endDate));
        } else {
            param.setEndTime(DateUtils.getDayEndTime(new Date()));
        }

        if (param.getStartTime().after(param.getEndTime())) {
            throw new BizRuntimeException("开始时间不能大于结束时间");
        }
        if (param.getEndTime().getTime() - param.getStartTime().getTime() > 100L * 24 * 3600 * 1000) {
            throw new BizRuntimeException("时间周期不能大于100天");
        }
        return param;
    }


}
