/**
 * Project Name:goods-center-biz
 * File Name:GoodsBatchDao.java
 * Package Name:cn.com.duiba.goods.center.biz.dao
 * Date:2016年5月23日上午11:52:58
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.dao;

import java.util.Date;
import java.util.List;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.enums.CouponType;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;

/**
 * ClassName:GoodsBatchDao <br/>
 * Date:     2016年5月23日 上午11:52:58 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see      
 */
public interface GoodsBatchDao {
    /**
     * 
     * insert:创建一个批次. <br/>
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchType 批次类型
     * @param startDay 开始日期
     * @param endDay 结束日期
     * @param stockId 库存id
     * @return 批次id
     * @since JDK 1.6
     */
    Long insert(GoodsTypeEnum gtype,Long gid,CouponType batchType,Date startDay,Date endDay, Long stockId);
    /**
     * 
     * select:根据id查询一条批次信息. <br/>
     *
     * @author xuhengfei
     * @param id 批次id
     * @return 批次信息
     * @since JDK 1.6
     */
    GoodsBatchEntity select(Long id);
    /**
     * 不包含已删除的批次
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @return 结果集
     * @since JDK 1.6
     */
    List<GoodsBatchEntity> selectNotDeletedByGtypeGid(GoodsTypeEnum gtype,long gid);
    /**
     * 不包含已删除和已过期的批次
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @return 结果集
     * @since JDK 1.6
     */
    List<GoodsBatchEntity> selectNotDeletedNotExpiredGtypeGid(GoodsTypeEnum gtype,long gid);
    
    /**
     * 
     * updateImportingYes:标记为正在导入中. <br/>
     *
     * @author xuhengfei
     * @param id 批次id
     * @since JDK 1.6
     */
    void updateImportingYes(Long id);
    /**
     * 
     * updateImportingNo:标记为没有在导入中. <br/>
     *
     * @author xuhengfei
     * @param id 批次id
     * @since JDK 1.6
     */
    void updateImportingNo(Long id);
    /**
     * 更新为已使用状态
     *
     * @author xuhengfei
     * @param id 批次id
     * @return 影响行数
     * @since JDK 1.6
     */
    int updateStatusUsed(Long id);
    /**
     * 更新为使用中状态
     *
     * @author xuhengfei
     * @param id 批次id
     * @return 影响行数
     * @since JDK 1.6
     */
    int updateStatusUsing(Long id);
    /**
     * 更新为未使用状态
     *
     * @author xuhengfei
     * @param id 批次id
     * @return 影响行数
     * @since JDK 1.6
     */
    int updateStatusNotUsed(Long id);
    /**
     * 逻辑删除一个批次
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param goodsBatchId 批次id
     * @return 影响行数
     * @since JDK 1.6
     */
    int deleteBatch(GoodsTypeEnum gtype,long gid,Long goodsBatchId);
    
    /**
     * 更新库存id字段
     *
     * @author xuhengfei
     * @param gtype 商品类型
     * @param gid 商品id
     * @param goodsBatchId 批次id
     * @param stockId 新库存id
     * @return 影响行数
     * @since JDK 1.6
     */
    int updateStockId(GoodsTypeEnum gtype,long gid,Long goodsBatchId,Long stockId);
    
    /**
     * 根据批次日期查询批次
     * @param gtype
     * @param gid
     * @param startDay
     * @param endDay
     * @return
     */
    GoodsBatchEntity selectByStartDayAndDay(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay);
    
    /**
     * 更新批次有效时间.
     *
     * @param id 批次id
     * @param startDay 生效期
     * @param endDay 失效期
     * @return true, if update valid date
     */
    int updateValidDate(long id,Date startDay, Date endDay);
    
    /**
     * 
     * 更新批次类型,仅针对广告券商品使用. <br/>
     *
     * @author sunjiangrong
     * @param gid 商品id
     * @param dstType
     * @return 影响行数
     */
    int updateBatchType(long gid, int dstType);
    
    /**
     * 因为唯一索引限制，物理删除批次数据
     *
     * @author zufengze
     * @param gtype 商品类型
     * @param gid 商品id
     * @param batchId 批次id
     * @return 影响行数
     * @since JDK 1.6
     */
    int physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId);
    /**
     * 查询多个商品批次
     * @param gtype 商品类型
     * @param gids 商品id列表
     * @return 结果集
     */
    List<GoodsBatchEntity> selectGoodsBatchs(GoodsTypeEnum gtype, List<Long> gids);
}

