/**
 * Project Name:goods-center-biz File Name:GoodsBatchServiceImpl.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午4:43:49 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.enums.CouponType;
import cn.com.duiba.goods.center.biz.bo.GoodsBatchCouponBackendBO;
import cn.com.duiba.goods.center.biz.dao.GoodsBatchDao;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * ClassName:GoodsBatchServiceImpl <br/>
 * Date: 2016年5月25日 下午4:43:49 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
public class GoodsBatchServiceImpl implements GoodsBatchService {

    private static Logger             log = LoggerFactory.getLogger(GoodsBatchServiceImpl.class);
    @Autowired
    private GoodsBatchDao             goodsBatchDao;
    @Autowired
    private RemoteStockService        remoteStockService;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;

    @Override
    public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay) {

        // 只允许精确到日，不允许精确到秒
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            startDay = sdf.parse(sdf.format(startDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        try {
            endDay = sdf.parse(sdf.format(endDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }
        GoodsBatchEntity entity = goodsBatchDao.selectByStartDayAndDay(gtype, gid, startDay, endDay);
        if(entity != null){
        	if(entity.getStatus()==GoodsBatchEntity.StatusDeleted){
        		goodsBatchDao.updateStatusNotUsed(entity.getId());
        	}
        	return entity.getId();
        }
        return goodsBatchDao.insert(gtype, gid, CouponType.Normal, startDay, endDay, null);
    }

    @Override
    public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay, Long stockId) {
        // 只允许精确到日，不允许精确到秒
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            startDay = sdf.parse(sdf.format(startDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        try {
            endDay = sdf.parse(sdf.format(endDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }
        GoodsBatchEntity entity = goodsBatchDao.selectByStartDayAndDay(gtype, gid, startDay, endDay);
        if(entity != null){
        	if(entity.getStatus()==GoodsBatchEntity.StatusDeleted){
        		goodsBatchDao.updateStatusNotUsed(entity.getId());
        	}
        	return entity.getId();
        }
        return goodsBatchDao.insert(gtype, gid, CouponType.Link, startDay, endDay, stockId);
    }

    @Override
    public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date startDay, Date endDay, Long stockId) {
        // 只允许精确到日，不允许精确到秒
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");

        try {
            startDay = sdf.parse(sdf.format(startDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }

        try {
            endDay = sdf.parse(sdf.format(endDay));
        } catch (ParseException e) {
            log.error("date parse error", e);
        }
        GoodsBatchEntity entity = goodsBatchDao.selectByStartDayAndDay(gtype, gid, startDay, endDay);
        if(entity != null){
        	if(entity.getStatus()==GoodsBatchEntity.StatusDeleted){
        		goodsBatchDao.updateStatusNotUsed(entity.getId());
        	}
        	return entity.getId();
        }
        return goodsBatchDao.insert(gtype, gid, CouponType.Repeat, startDay, endDay, stockId);
    }

    @Override
    public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid) {
        return goodsBatchDao.selectNotDeletedNotExpiredGtypeGid(gtype, gid);
    }

    @Override
    public List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid) {
        return goodsBatchDao.selectNotDeletedByGtypeGid(gtype, gid);
    }

    @Override
    public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        GoodsBatchEntity e = goodsBatchDao.select(batchId);
        if (e.getImporting()) {
            return false;
        }
		if (e.getBatchType() == GoodsBatchEntity.BatchTypeNormal) {
			boolean existUsedCoupon = goodsCouponService.findBatchExsitUsedCoupon(gtype, gid, batchId);
			if (existUsedCoupon) {
				return false;
			}
		}else{
            DubboResult<Long> stockRet=remoteStockBackendService.find(e.getStockId());
            DubboResult<Long> totalStockRet=remoteStockBackendService.findTotalStock(e.getStockId());
            if(totalStockRet.isSuccess() && stockRet.isSuccess()){
                Long stock = stockRet.getResult();
                Long totalStock = stockRet.getResult();
                if(stock != totalStock){
                	return false;
                }
            }else{
                throw new RuntimeGoodsException(ErrorCode.E9999999);
            }
		}
        int ret=goodsBatchDao.deleteBatch(gtype,gid,batchId);
        if(ret==1){
            return true;
        } else {
            return false;
        }
    }
    /*
    @Override
    public Boolean fillStockId(GoodsTypeEnum gtype, long gid, long goodsBatchId, long stockId) {
        int ret = goodsBatchDao.updateStockId(gtype, gid, goodsBatchId, stockId);
        if (ret == 1) {
            return true;
        } else {
            return false;
        }
    }*/

    @Override
    public GoodsBatchEntity find(Long goodsBatchId) {
        return goodsBatchDao.select(goodsBatchId);
    }

    @Autowired
    private GoodsCouponService goodsCouponService;

    @Override
    public GoodsBatchEntity getUsingBatch(GoodsTypeEnum gtype, long gid) {
        try {
            DBTimeProfile.enter(getClass().getSimpleName() + ".refreshBatchUsing");

            List<GoodsBatchEntity> batchs = findNormalBatchs(gtype, gid);

            for (GoodsBatchEntity e : batchs) {
                long stock = goodsCouponService.findNotUsedCount(e);
                if (stock <= 0) {
                    // 标记无库存
                    goodsBatchDao.updateStatusUsed(e.getId());
                    e.setStatus(GoodsBatchEntity.StatusUsed);
                }
            }
            
            //选当前使用用的批次
            for(GoodsBatchEntity e:batchs){
	            if(e.getStatus()==GoodsBatchEntity.StatusUsing){
	        		return e;
	        	}
            }
            
            //选下一个批次
            batchs=findNormalBatchs(gtype, gid);
            List<GoodsBatchEntity> backups=new ArrayList<>();
            Date min=null;
            for(GoodsBatchEntity e:batchs){
                if(e.getStatus()==GoodsBatchEntity.StatusNotUse){
                    backups.add( e);
                    if(min==null){
                        min=e.getStartDay();
                    }else{
                        if(min.getTime()>e.getStartDay().getTime()){
                            min=e.getStartDay();
                        }
                    }
                }
            }

            for (GoodsBatchEntity e : backups) {
                if (e.getStartDay().equals(min)) {
                    // 选中
                    goodsBatchDao.updateStatusUsing(e.getId());
                    return find(e.getId());
                }
            }

            return null;
        } finally {
            DBTimeProfile.release();
        }
    }

    private Long getOrigionTotalBatchStock(GoodsTypeEnum gtype, long gid, long goodsBatchId) {
        GoodsBatchEntity batch = goodsBatchDao.select(goodsBatchId);
        if (batch.getBatchType() == GoodsBatchEntity.BatchTypeLink
            || batch.getBatchType() == GoodsBatchEntity.BatchTypeRepeat) {
            if (batch.getStockId() == null) {
                // 如果没有填写stockId，认为0库存
                return 0L;
            }
            DubboResult<Long> ret = remoteStockBackendService.findTotalStock(batch.getStockId());
            if (ret.isSuccess()) {
                return ret.getResult();
            } else {
                throw new RuntimeGoodsException(ErrorCode.E0203003);
            }

        } else {
            return goodsCouponService.findTotalAllCount(gtype, gid, goodsBatchId);
        }
    }

    @Override
    public Long getSumBatchStock(GoodsTypeEnum gtype, long gid) {
        List<GoodsBatchEntity> list = findNormalBatchs(gtype, gid);
        long total = 0L;
        for (GoodsBatchEntity e : list) {
            //已删除，或者已使用的，跳过
            if (e.getStatus() != GoodsBatchEntity.StatusDeleted &&
                    e.getStatus()!=GoodsBatchEntity.StatusUsed) {
                total += goodsCouponService.findNotUsedCount(e);
            }
        }
        return total;
    }

    @Override
    public Long getTotalAllBatchStock(GoodsTypeEnum gtype, long gid) {
        List<GoodsBatchEntity> list = findNotDeletedBatchs(gtype, gid);
        long total = 0L;
        for (GoodsBatchEntity e : list) {
            if (e.getStatus() != GoodsBatchEntity.StatusDeleted) {
                total += getOrigionTotalBatchStock(gtype, gid, e.getId());
            }
        }

        return total;
    }

    @Override
    public void markBatchStatusUsed(long batchId) {
        goodsBatchDao.updateStatusUsed(batchId);
    }

    @Override
    public void markBatchStatusUsing(long batchId) {
        goodsBatchDao.updateStatusUsing(batchId);
    }

    @Override
    public void markBatchStatusNotUsed(long batchId) {
        goodsBatchDao.updateStatusNotUsed(batchId);
    }

    @Override
    public Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay) {
        // 只允许精确到日，不允许精确到秒
        startDay = new DateTime(startDay).withTimeAtStartOfDay().toDate();
        endDay = new DateTime(endDay).withTimeAtStartOfDay().toDate();

        if (goodsBatchDao.updateValidDate(batchId, startDay, endDay) == 1) {
            return true;
        }
        return false;
    }

    @Override
    public boolean updateGoodsType(GoodsTypeEnum gtype, long gid, int dstType) {
        if (gtype.equals(GoodsTypeEnum.ADVERT)) {
            int row = goodsBatchDao.updateBatchType(gid, dstType);
            if (row >= 1) {
                return true;
            }
            return false;
        }
        return false;
    }

    @Override
    public Boolean physicalDeleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        
        int row = goodsBatchDao.physicalDeleteBatch(gtype, gid, batchId);
        if (row >= 1) {
            return true;
        }
        return false;
    }

	@Override
	public List<GoodsBatchEntity> findGoodsBatchs(GoodsTypeEnum gtype,
			List<Long> gids) {
		return goodsBatchDao.selectGoodsBatchs(gtype,gids);
	}

	@Override
	public void refreshUsingBatch(GoodsTypeEnum gtype, long gid) {
		try {
			DBTimeProfile.enter(getClass().getSimpleName() + ".refreshBatchUsing");

			List<GoodsBatchEntity> batchs = findNormalBatchs(gtype, gid);

			for (GoodsBatchEntity e : batchs) {
				long stock = goodsCouponService.findNotUsedCount(e);
				if (stock <= 0) {
					// 标记无库存
					goodsBatchDao.updateStatusUsed(e.getId());
					e.setStatus(GoodsBatchEntity.StatusUsed);
				}
			}

			// 选当前使用用的批次
			for (GoodsBatchEntity e : batchs) {
				if (e.getStatus() == GoodsBatchEntity.StatusUsing) {
					return;
				}
			}

			// 选下一个批次
			batchs = findNormalBatchs(gtype, gid);
			List<GoodsBatchEntity> backups = new ArrayList<>();
			Date min = null;
			for (GoodsBatchEntity e : batchs) {
				if (e.getStatus() == GoodsBatchEntity.StatusNotUse) {
					backups.add(e);
					if (min == null) {
						min = e.getStartDay();
					} else {
						if (min.getTime() > e.getStartDay().getTime()) {
							min = e.getStartDay();
						}
					}
				}
			}

			for (GoodsBatchEntity e : backups) {
				if (e.getStartDay().equals(min)) {
					// 选中
					goodsBatchDao.updateStatusUsing(e.getId());
					return;
				}
			}
		} finally {
			DBTimeProfile.release();
		}
	}

}
