/**
 * Project Name:goods-center-biz File Name:GoodsCouponServiceImpl.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午1:08:07 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.api.remoteservice.tool.Page;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao;
import cn.com.duiba.goods.center.biz.dao.GoodsCouponDao.CouponFormat;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.biz.service.stock.StockService;
import cn.com.duiba.goods.center.biz.util.ConsumeStockTypeUtil;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.idmaker.service.api.remoteservice.RemoteIDMakerBackendService;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.service.remoteservice.RemoteConsumerService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;

/**
 * ClassName:GoodsCouponServiceImpl <br/>
 * Date: 2016年5月25日 下午1:08:07 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
@SuppressWarnings("deprecation")
@Service("goodsCouponService")
public class GoodsCouponServiceImpl implements GoodsCouponService {

    private static Logger               log = LoggerFactory.getLogger(GoodsCouponServiceImpl.class);
    @Autowired
    private GoodsCouponDao              goodsCouponDao;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;
    @Autowired
    private RemoteStockService          remoteStockService;
    @Autowired
    private StockService                stockService;
    @Resource
    protected RemoteConsumerService       remoteConsumerService;
    @Autowired
    protected GoodsCouponService        goodsCouponService;
    @Autowired
    private RemoteIDMakerBackendService remoteIDMakerBackendService;

    @Override
    public GoodsCouponEntity find(Long goodsCouponId) {
        return goodsCouponDao.selectByCouponId(goodsCouponId);
    }

    @Override
    public long findNotUsedCount(GoodsBatchEntity batch) {
        if (batch.getBatchType() == GoodsBatchEntity.BatchTypeNormal) {
            Integer count = goodsCouponDao.selectCountByBatchNotUsed(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()),
                                                                     batch.getGid(), batch.getId());
            if (count == null) {
                return 0L;
            }
            return count;
        }else{
            if(batch.getStockId()==null){
                //如果没有填写库存id，认为0库存
                return 0L;
            }
            DubboResult<Long> ret=remoteStockService.find(batch.getStockId());
            if(ret.isSuccess()){
                return ret.getResult();
            } else {
                log.error("findNotUsedCount error: code=" + ret.getReturnCode() + ",msg=" + ret.getMsg() + ",stockId="
                          + batch.getStockId());
                throw new RuntimeGoodsException(ErrorCode.E0203003);
            }
        }

    }

    @Override
    public Boolean updateLinkBatch(GoodsTypeEnum gtype, long gid, Long batchId, String link) {
        int ret = goodsCouponDao.updateLinkCoupon(gtype, gid, batchId, link);
        if (ret == 1) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public void importLinkCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, String link) {
        goodsCouponDao.insertLinkCoupon(gtype, gid, goodsBatchId, goodsCouponId, link);
    }

    @Override
    public void importRepeatCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId, String code,
                                   String password) {
        goodsCouponDao.insertRepeatCoupon(gtype, gid, goodsBatchId, goodsCouponId, code, password);
    }

    @Override
    public Boolean updateRepeatBatch(GoodsTypeEnum gtype, long gid, Long batchId, String code, String password) {
        int ret = goodsCouponDao.updateRepeatCoupon(gtype, gid, batchId, code, password);
        if (ret == 1) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public Integer importNormalCoupons(GoodsTypeEnum gtype, long gid, Long batchId, List<CouponFormat> coupons) {
        if (coupons.size() > 1000000) {
            throw new RuntimeException("超出100W，拒绝导入");
        }

        final int MAX_RECORDS = 20000;

        int size = coupons.size();
        int batchs = size % MAX_RECORDS == 0 ? (size / MAX_RECORDS) : (size / MAX_RECORDS + 1);
        List<List<CouponFormat>> group = new ArrayList<>(batchs);
        Map<Integer, AtomicInteger> countMap = new HashMap<Integer, AtomicInteger>();

        Set<String> set = new HashSet<>();

        for (int i = 0; i < batchs; i++) {
            List<CouponFormat> l = new ArrayList<>(MAX_RECORDS);
            group.add(l);
            countMap.put(i, new AtomicInteger());
        }

        // 先分组
        for (CouponFormat cf : coupons) {
            if (set.add(cf.getCode())) {
                for (int i = 0; i < group.size(); i++) {
                    List<CouponFormat> cflist = group.get(i);
                    if (countMap.get(i).get() < MAX_RECORDS) {
                        countMap.get(i).incrementAndGet();
                        cflist.add(cf);
                        break;
                    }
                }
            } else {
                log.debug("重复券码 code=" + cf.getCode());
            }
        }

        int totalSuccess = 0;
        // 对每一个分组进行一个批量插入动作
        for (List<CouponFormat> list : group) {
            if (!list.isEmpty()) {
                totalSuccess += goodsCouponDao.insertBatchImport(gtype, gid, batchId, list);
            }
        }

        return totalSuccess;
    }

    @Override
    public Page<GoodsCouponEntity> findPage(GoodsTypeEnum gtype, long gid, long batchId, int pageSize, int pageIndex) {
        Page<GoodsCouponEntity> page = new Page<>(pageSize, pageIndex);
        int count = goodsCouponDao.selectCountByBatch(gtype, gid, batchId);
        int start = (pageIndex - 1) * pageSize;
        int limit = pageSize;
        List<GoodsCouponEntity> list = goodsCouponDao.selectPageByBatchId(gtype, gid, batchId, start, limit);
        page.setTotalPages(count % pageSize == 0 ? (count / pageSize) : (count / pageSize + 1));
        page.setList(list);
        page.setTotalCount(count);
        return page;
    }
    /**
     * 在领券前先减其他库存
     *
     * @author xuhengfei
     * @param gtype
     * @param batch
     * @param consumerId
     * @param bizNum
     * @param c
     * @return
     * @since JDK 1.6
     */
    protected boolean consumeStocks(GoodsTypeEnum gtype, GoodsBatchEntity batch,long consumerId,String bizNum,ConsumerDO c){
        //查询其他库存限制，调用库存中心进行减库存
        List<Long> stockIds=stockService.getNeedCountDownOtherStockIds(gtype.getGtype(), batch.getGid(),c.getAppId());
        if(!stockIds.isEmpty()){
            DubboResult<Boolean> ret=remoteStockService.consumeStock(ConsumeStockTypeUtil.getConsumeStockTypes(gtype).getType(), bizNum, stockIds);
            if(!ret.isSuccess() || !ret.getResult()){
                //减库存失败，拒绝领券
                return false;
            }
        }
        return true;
    }
    
    @Override
    public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId,
                                              String bizNum) {

        ConsumerDO c = remoteConsumerService.find(consumerId);
        try {
            DBTimeProfile.enter("GoodsCouponServiceImpl.takeNormalCoupon");
            
            if(!consumeStocks(gtype, batch, consumerId, bizNum, c)){
                //如果减库存失败，返还null
                return null;
            }

            GoodsCouponEntity entity = goodsCouponDao.selectOneCouponNotUsed(gtype, batch.getGid(), batch.getId());
            boolean success = takeCoupon4Point(gtype, batch.getGid(), batch.getId(), entity.getGoodsCouponId(),
                                               consumerId);
            if (success) {
                return entity;
            }
            return null;
        } finally {
            DBTimeProfile.release();
        }
    }

    /**
     * 如果是普通券判断是否有库存，会进行count，有性能问题，此处如果有大量并发查询，需要进行性能优化<br/>
     * 如果是链接券或者重复券，通过库存中心查询库存，由库存中心保证性能
     * 
     * @see cn.com.duiba.goods.center.biz.service.GoodsCouponService#isBatchHaveStock(cn.com.duiba.dcommons.enums.GoodsTypeEnum,
     * cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity)
     * @Override public boolean isBatchHaveStock(GoodsTypeEnum gtype,GoodsBatchEntity batch) { try{
     * DBTimeProfile.enter("GoodsCouponServiceImpl.isBatchHaveStock");
     * if(batch.getBatchType()==GoodsBatchEntity.BatchTypeLink){ // 链接类型 DubboResult<Long>
     * stockRet=remoteStockService.find(batch.getStockId()); if(stockRet.isSuccess()){ return stockRet.getResult()>0; }
     * }else if(batch.getBatchType()==GoodsBatchEntity.BatchTypeRepeat){ // 重复类型 DubboResult<Long>
     * stockRet=remoteStockService.find(batch.getStockId()); if(stockRet.isSuccess()){ return stockRet.getResult()>0; }
     * }else if(batch.getBatchType()==GoodsBatchEntity.BatchTypeNormal){ // 普通类型 int
     * count=goodsCouponDao.selectCountByBatchNotUsed(gtype, batch.getGid(), batch.getId()); if(count>0){ return true;
     * }else{ return false; } } return false; }finally{ DBTimeProfile.release(); } }
     */

    @Override
    public GoodsCouponEntity takeLinkCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, String bizNum) {
        DubboResult<Long> batchStock=remoteStockService.find(batch.getStockId());
        if(batchStock.isSuccess()&& batchStock.getResult()<=0){
            throw new RuntimeGoodsException(ErrorCode.E0202005);
        }
        
        ConsumerDO c = remoteConsumerService.find(consumerId);
        List<Long> stockIds = null;
        try {
            DBTimeProfile.enter("getNeedCountDownOtherStockIds");
            stockIds = stockService.getNeedCountDownOtherStockIds(gtype.getGtype(), batch.getGid(), c.getAppId());
        } finally {
            DBTimeProfile.release();
        }

        stockIds.add(batch.getStockId());

        try {
            DBTimeProfile.enter("remoteStockService.decreaseGoodStock");
            DubboResult<Boolean> stockRet = remoteStockService.consumeStock(ConsumeStockTypeUtil.getConsumeStockTypes(gtype).getType(),
                                                                            bizNum, stockIds);
            if (!stockRet.isSuccess() || !stockRet.getResult()) {
                return null;
            }
        } finally {
            DBTimeProfile.release();
        }

        // TODO 缓存优化
        try {
            DBTimeProfile.enter("selectOneByGoodsBatchId");
            return goodsCouponDao.selectOneByGoodsBatchId(gtype, batch.getGid(), batch.getId());
        } finally {
            DBTimeProfile.release();
        }
    }

    @Override
    public GoodsCouponEntity takeRepeatCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId,
                                              String bizNum) {
        DubboResult<Long> batchStock=remoteStockService.find(batch.getStockId());
        if(batchStock.isSuccess()&& batchStock.getResult()<=0){
            throw new RuntimeGoodsException(ErrorCode.E0202005);
        }
        
        
        ConsumerDO c = remoteConsumerService.find(consumerId);

        List<Long> stockIds = stockService.getNeedCountDownOtherStockIds(gtype.getGtype(), batch.getGid(), c.getAppId());
        stockIds.add(batch.getStockId());

        try {
            DBTimeProfile.enter("remoteStockService.decreaseGoodStock");
            DubboResult<Boolean> stockRet = remoteStockService.consumeStock(ConsumeStockTypeUtil.getConsumeStockTypes(gtype).getType(),
                                                                            bizNum, stockIds);
            if (!stockRet.isSuccess() || !stockRet.getResult()) {
                return null;
            }
        } finally {
            DBTimeProfile.release();
        }

        // TODO 缓存优化
        return goodsCouponDao.selectOneByGoodsBatchId(gtype, batch.getGid(), batch.getId());
    }

    @Override
    public void completeCoupon(Long couponId, long orderId) {
        goodsCouponDao.updateCoupnoMarkUsed(couponId, orderId);
    }

    @Override
    public Boolean rollbackNormalCoupon(Long couponId) {
        int ret = goodsCouponDao.updateCouponMarkRollback(couponId);
        if (ret == 1) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 从数据库加载一定的未使用的优惠券
     *
     * @author xuhengfei
     * @param gtype
     * @param gid
     * @param batchId
     * @param limit
     * @return
     * @since JDK 1.6
     */
    protected List<GoodsCouponEntity> loadCouponByBatchId(GoodsTypeEnum gtype, long gid, long batchId, int limit) {
        return goodsCouponDao.selectBatchNotUsed(gtype, gid, batchId, limit);
    }

    protected boolean takeCoupon4Point(GoodsTypeEnum gtype, long gid, long goodsBatchId, long goodsCouponId,
                                       long consumerId) {
        try {
            DBTimeProfile.enter("GoodsCouponServiceImpl.takeCoupon4Point");
            int ret = goodsCouponDao.updateCouponMarkLocked(gtype, gid, goodsBatchId, goodsCouponId, consumerId);
            if (ret == 1) {
                return true;
            } else {
                return false;
            }
        } finally {
            DBTimeProfile.release();
        }
    }

    @Override
    public GoodsCouponEntity findOneByGoodsBatchId(GoodsTypeEnum gtype, long gid, long goodsBatchId) {
        return goodsCouponDao.selectOneByGoodsBatchId(gtype, gid, goodsBatchId);
    }

    @Override
    public List<GoodsCouponEntity> searchByCode(GoodsTypeEnum gtype, long gid, long goodsBatchId, String code) {
        return goodsCouponDao.selectSearchByCode(gtype, gid, goodsBatchId, code);
    }

    @Override
    public long findTotalAllCount(GoodsTypeEnum gtype, long gid, long batchId) {
        return goodsCouponDao.selectCountByBatch(gtype, gid, batchId);
    }
    
    private Lock lock=new ReentrantLock();
    @Override
    public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch) {
        try {
        	if(!lock.tryLock()){
        		return false;
        	}
			if(batch.getBatchType()==GoodsBatchEntity.BatchTypeNormal){
				//普通券删除未使用的券码
				long count=findNotUsedCount(batch);
				if(count>50000){
				    return false;
				}
				int ret=goodsCouponDao.deleteBatchUnusedCoupons(batch.getGtype(),batch.getGid(),batch.getId());
				if(ret>0){
					return true;
				}
			}else{
				//重复券，连接券 库存减少为0 
				DubboResult<Long> stockRet = remoteStockService.find(batch.getStockId());
				DubboResult<Boolean> dret = remoteStockBackendService.decreaseItemStock(batch.getStockId(), stockRet.getResult());
				return true;
			}
			return false;
		} finally {
			lock.unlock();
		}
    }
    
	@Override
	public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, long couponId, long batchId) {
		int ret = goodsCouponDao.deleteUnusedCoupon(gtype.getGtype(), gid, couponId);
		if (ret > 0) {
			return true;
		}
		return false;
	}
	
	@Override
	public Boolean findBatchExsitUsedCoupon(GoodsTypeEnum gtype, long gid, long goodsBatchId) {
		Long ret = goodsCouponDao.selectByBatchExistUsed(gtype.getGtype(), gid, goodsBatchId);
		if (ret != null) {
			return true;
		}
		return false;
	}
	
	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, List<Long> couponIds) {
		if (couponIds == null || couponIds.size() < 1 || couponIds.size() > 5000) {
			throw new RuntimeGoodsException(ErrorCode.E9999999);
		}
		try {
			if (!lock.tryLock()) {
				throw new RuntimeGoodsException(ErrorCode.E9999999);
			}
			goodsCouponDao.deleteBatchUnusedCouponsByIds(gtype.getGtype(), gid, batchId, couponIds);
		} finally {
			lock.unlock();
		}
		return true;
	}
	@Override
	public List<GoodsCouponEntity> findUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, int limit) {
		if(limit > 5000){
			throw new RuntimeGoodsException(ErrorCode.E9999999);
		}
		return goodsCouponDao.findUnusedCoupons(gtype.getGtype(), gid, batchId, limit);
	}
    
    @Override
    public boolean deleteGoodsCoupon(GoodsTypeEnum gtype, long gid, long batchId) {

        try {
            DBTimeProfile.enter("GoodsCouponServiceImpl.deleteGoodsCoupon");
            int ret = goodsCouponDao.deleteGoodsCoupon(gtype, gid, batchId);
            if (ret == 1) {
                return true;
            } else {
                return false;
            }
        } finally {
            DBTimeProfile.release();
        }
    }
}
