/**
 * Project Name:goods-center-biz
 * File Name:GoodsBatchRedisService.java
 * Package Name:cn.com.duiba.goods.center.biz.service.impl
 * Date:2016年6月4日下午9:32:25
 * Copyright (c) 2016, duiba.com.cn All Rights Reserved.
 *
*/

package cn.com.duiba.goods.center.biz.service.impl;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.service.GoodsBatchService;
import cn.com.duiba.goods.center.common.RedisKeyTool;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.alibaba.fastjson.JSONObject;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * ClassName:GoodsBatchRedisService <br/>
 * Function: 使用redis做缓存，提升性能 <br/>
 * Date:     2016年6月4日 下午9:32:25 <br/>
 * @author   xuhengfei
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service("goodsBatchRedisService")
public class GoodsBatchRedisService extends GoodsBatchServiceImpl implements GoodsBatchService{

    private static final Logger log=LoggerFactory.getLogger(GoodsBatchRedisService.class);
    @Autowired
    private RedisTemplate<String, String> redisTemplate;

    private RedisBatchCache redisBatchCache=new RedisBatchCache();
    /**
     * Not Delete Batchs 未删除的批次列表
     *
     * @author xuhengfei
     * @version GoodsBatchRedisService
     * @since JDK 1.6
     */
    public class RedisBatchCache{
        public List<GoodsBatchEntity> getBatchsCache(GoodsTypeEnum gtype,long gid){
            String string=redisTemplate.opsForValue().get(getRedisBatchsKey(gtype, gid));
            if(string==null){
                return null;
            }
            return JSONObject.parseArray(string, GoodsBatchEntity.class);
        }
      //删除缓存
        public void clearBatchsCache(GoodsTypeEnum gtype,long gid){
            try{
                redisTemplate.delete(getRedisBatchsKey(gtype, gid));
            }catch(Exception e){
                log.error("RedisBatchCache.clearBatchsCache gtype="+gtype.getGtype()+",gid="+gid,e);
            }
        }
        //设置缓存
        public void setBatchsCache(GoodsTypeEnum gtype,long gid,List<GoodsBatchEntity> batchs){
            String string=JSONObject.toJSONString(batchs);
            redisTemplate.opsForValue().set(getRedisBatchsKey(gtype, gid), string);
            redisTemplate.expire(getRedisBatchsKey(gtype, gid), 1, TimeUnit.HOURS);
        }

        private String getRedisBatchsKey(GoodsTypeEnum gtype,long gid){
            return RedisKeyTool.getRedisKey(getClass(), "", gtype.getGtype()+"-"+gid);
        }
    }


    @Override
    public List<GoodsBatchEntity> findNormalBatchs(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchRedisService.findNormalBatchs");
            List<GoodsBatchEntity> batchs=findNotDeletedBatchs(gtype, gid);
            List<GoodsBatchEntity> ret=new ArrayList<>();
            Date today = new DateTime().withTimeAtStartOfDay().toDate();
            for(GoodsBatchEntity e:batchs){
                // 取未失效的批次
                if(!e.getEndDay().before(today)){
                    ret.add(e);
                }
            }
            return ret;
        }finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public List<GoodsBatchEntity> findNotDeletedBatchs(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchRedisService.findNotDeletedBatchs");
            List<GoodsBatchEntity> batchs=redisBatchCache.getBatchsCache(gtype, gid);
            if(batchs==null){
                batchs=super.findNotDeletedBatchs(gtype, gid);
                redisBatchCache.setBatchsCache(gtype, gid, batchs);
            }
            return batchs;
        } finally{
            DBTimeProfile.release();
        }
    }

    @Override
    public GoodsBatchEntity getUsingBatch(GoodsTypeEnum gtype, long gid) {
        List<GoodsBatchEntity> normals=findNormalBatchs(gtype, gid);
        for(GoodsBatchEntity e:normals){
            if(e.getStatus()==GoodsBatchEntity.StatusUsing){
                return e;
            }
        }
        return null;
    }

    @Override
    public Long createNormalBatch(GoodsTypeEnum gtype, long gid, Date start, Date end,Long stockId) {
        Long ret= super.createNormalBatch(gtype, gid, start, end,stockId);
        redisBatchCache.clearBatchsCache(gtype, gid);
        return ret;
    }

    @Override
    public Long createLinkBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId) {
        Long ret= super.createLinkBatch(gtype, gid, start, end, stockId);
        redisBatchCache.clearBatchsCache(gtype, gid);
        return ret;
    }

    @Override
    public Long createRepeatBatch(GoodsTypeEnum gtype, long gid, Date start, Date end, Long stockId) {
        Long ret= super.createRepeatBatch(gtype, gid, start, end, stockId);
        redisBatchCache.clearBatchsCache(gtype, gid);
        return ret;
    }

    @Override
    public Boolean deleteBatch(GoodsTypeEnum gtype, long gid, Long batchId) {
        Boolean ret=super.deleteBatch(gtype, gid, batchId);
        redisBatchCache.clearBatchsCache(gtype, gid);
        return ret;
    }
    /*
    @Override
    public Boolean fillStockId(GoodsTypeEnum gtype, long gid, long goodsBatchId, long stockId) {
        Boolean ret=super.fillStockId(gtype, gid, goodsBatchId, stockId);
        redisBatchCache.clearBatchsCache(gtype, gid);
        return ret;
    }*/

    @Override
    public GoodsBatchEntity find(Long goodsBatchId) {
        return super.find(goodsBatchId);
    }

    @Override
    public Long getSumBatchStock(GoodsTypeEnum gtype, long gid) {
        try{
            DBTimeProfile.enter("GoodsBatchRedisService.getSumBatchStock");
            return super.getSumBatchStock(gtype, gid);
        }finally{
            DBTimeProfile.release();
        }

    }

    @Override
    public Long getTotalAllBatchStock(GoodsTypeEnum gtype, long gid) {
        return super.getTotalAllBatchStock(gtype, gid);
    }

    @Override
    public void markBatchStatusNotUsed(long batchId) {
        super.markBatchStatusNotUsed(batchId);
        GoodsBatchEntity batch=super.find(batchId);
        redisBatchCache.clearBatchsCache(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid());
    }
    @Override
    public void markBatchStatusUsed(long batchId) {
        super.markBatchStatusUsed(batchId);
        GoodsBatchEntity batch=super.find(batchId);
        redisBatchCache.clearBatchsCache(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid());
    }
    @Override
    public void markBatchStatusUsing(long batchId) {
        super.markBatchStatusUsing(batchId);
        GoodsBatchEntity batch=super.find(batchId);
        redisBatchCache.clearBatchsCache(GoodsTypeEnum.getGoodsTypeEnum(batch.getGtype()), batch.getGid());
    }

    @Override
    public Boolean updateValidDate(GoodsTypeEnum gtype, long gid, long batchId, Date startDay, Date endDay) {
        Boolean result = super.updateValidDate(gtype, gid, batchId, startDay, endDay);
        redisBatchCache.clearBatchsCache(gtype, gid);
        return result;
    }
}

