/**
 * Project Name:goods-center-biz File Name:GoodsCouponService4Speed.java Package
 * Name:cn.com.duiba.goods.center.biz.service.impl Date:2016年5月25日下午1:07:36 Copyright (c) 2016, duiba.com.cn All Rights
 * Reserved.
 */

package cn.com.duiba.goods.center.biz.service.impl;

import cn.com.duiba.dcommons.enums.GoodsTypeEnum;
import cn.com.duiba.goods.center.biz.entity.GoodsBatchEntity;
import cn.com.duiba.goods.center.biz.entity.GoodsCouponEntity;
import cn.com.duiba.goods.center.biz.service.GoodsCouponService;
import cn.com.duiba.goods.center.common.ErrorCode;
import cn.com.duiba.goods.center.common.RedisKeyTool;
import cn.com.duiba.goods.center.common.RuntimeGoodsException;
import cn.com.duiba.service.domain.dataobject.ConsumerDO;
import cn.com.duiba.service.remoteservice.RemoteConsumerService;
import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import com.alibaba.fastjson.JSONObject;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * ClassName:GoodsCouponService4Speed <br/>
 * Date: 2016年5月25日 下午1:07:36 <br/>
 * 
 * @author xuhengfei
 * @version
 * @since JDK 1.6
 * @see
 */
@Service("goodsCouponService4Speed")
public class GoodsCouponService4Speed extends GoodsCouponServiceImpl implements GoodsCouponService, InitializingBean {

    private static final Logger           log            = LoggerFactory.getLogger(GoodsCouponService4Speed.class);
    private static final int              RedisBatchSize = 10000;
    /** 存放队列，内部将GoodsCouponEntity使用json转换为String */
    @Autowired
    private RedisTemplate<String, String> redisStringTemplate;
    /** redis券队列缓存管理类 */
    private CouponQueueRedisCache         couponQueueRedisCache;
    /** 批次是否售罄的redis缓存管理类 */
    private BatchSaleoutRedisCache        batchSaleoutRedisCache;
    @Autowired
    private RemoteConsumerService remoteConsumerService;

    @Override
    public void afterPropertiesSet() throws Exception {
        batchSaleoutRedisCache=new BatchSaleoutRedisCache(redisStringTemplate);
        couponQueueRedisCache=new CouponQueueRedisCache(redisStringTemplate,batchSaleoutRedisCache);
        
    }
    
    @Override
    public Boolean deleteBatchUnusedCoupons(GoodsBatchEntity batch) {
    	boolean ret = super.deleteBatchUnusedCoupons(batch);
    	couponQueueRedisCache.clearCache(batch.getId());
    	return ret;
    }
    
    @Override
    public Boolean deleteUnusedCoupon(GoodsTypeEnum gtype, long gid, long couponId, long batchId) {
    	boolean ret = super.deleteUnusedCoupon(gtype, gid, couponId, batchId);
    	couponQueueRedisCache.clearCache(batchId);
    	return ret;
    }
    

    @Override
    public GoodsCouponEntity takeNormalCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId,
                                              String bizNum) {
        try {
            DBTimeProfile.enter("GoodsCouponService4Speed.takeNormalCoupon");

            ConsumerDO c = remoteConsumerService.find(consumerId);

            if (!consumeStocks(gtype, batch, consumerId, bizNum, c)) {
                // 如果减库存失败，返还null
                return null;
            }

            if (!batchSaleoutRedisCache.isSaleOut(batch.getId())) {
                return tryTakeCoupon(gtype, batch, consumerId, 1);
                // TODO 此处需要添加支持，如果redis挂掉，也能继续领券
            } else {
                // 该批次已售罄
                throw new RuntimeGoodsException(ErrorCode.E0202005);
            }
        } finally {
            DBTimeProfile.release();
        }
    }

    @Override
    public Boolean rollbackNormalCoupon(Long couponId) {
        Boolean success = super.rollbackNormalCoupon(couponId);
        // 回滚的优惠劵的批次如果已经标记为已使用，则这个优惠劵后续也无法被使用
        // 如果想优化这点，可以在这里写代码，将批次重新变更为未使用
        // 由于这种写法会有性能隐患，暂时不实现此功能，待后续判断是否需要实现
        return success;
    }

    /**
     * 在redis出错的情况下，调用super的领券方法
     * 
     * @author xuhengfei
     * @param gtype
     * @param batch
     * @param consumerId
     * @return
     * @since JDK 1.6
     */
    private GoodsCouponEntity origionTakeCouponByLock(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId,
                                                      String bizNum) {
        if (batchLocks.getUnchecked(batch.getId()).tryLock()) {
            return super.takeNormalCoupon(gtype, batch, consumerId, bizNum);
        } else {
            return null;
        }
    }

    private static LoadingCache<Long, Lock> batchLocks = CacheBuilder.newBuilder().expireAfterAccess(1, TimeUnit.HOURS).build(new CacheLoader<Long, Lock>() {

                                                                                                                                  public Lock load(Long key)
                                                                                                                                                            throws Exception {
                                                                                                                                      return new ReentrantLock();
                                                                                                                                  };
                                                                                                                              });

    /**
     * 从redis队列中取数据 执行lock动作 如果lock动作失败，递归最多尝试3次
     *
     * @author xuhengfei
     * @param gtype
     * @param batch
     * @param consumerId
     * @param tryCount
     * @return
     * @since JDK 1.6
     */
    private GoodsCouponEntity tryTakeCoupon(GoodsTypeEnum gtype, GoodsBatchEntity batch, long consumerId, int tryCount) {
        if (tryCount > 3) {
            throw new RuntimeGoodsException(ErrorCode.E0202003);
        }
        try {
            DBTimeProfile.enter("GoodsCouponService4Speed.tryTakeCoupon");

            long goodsBatchId = batch.getId();
            long gid = batch.getGid();

            long size = couponQueueRedisCache.getQueueSize(goodsBatchId);
            if (size > 0) {
                GoodsCouponEntity entity = couponQueueRedisCache.pop(goodsBatchId);
                boolean success = takeCoupon4Point(gtype, gid, goodsBatchId, entity.getGoodsCouponId(), consumerId);
                if (success) {
                    return entity;
                } else {
                    return tryTakeCoupon(gtype, batch, consumerId, tryCount++);
                }
            } else {
                couponQueueRedisCache.tryLoadCouponByBatchId(gtype, gid, goodsBatchId, RedisBatchSize);
                return tryTakeCoupon(gtype, batch, consumerId, tryCount++);
            }
        } finally {
            DBTimeProfile.release();
        }
    }

    public static class BatchSaleoutRedisCache {

        private RedisTemplate<String, String> redisTemplate;

        public BatchSaleoutRedisCache(RedisTemplate<String, String> redis) {
            this.redisTemplate = redis;
        }

        /**
         * 是否售罄，如果缓存中没有数据，默认返回还没有售罄
         *
         * @author xuhengfei
         * @param batchId
         * @return
         * @since JDK 1.6
         */
        public boolean isSaleOut(long batchId) {
            String saleoutStr = redisTemplate.opsForValue().get(getSaleOutKey(batchId));
            Boolean saleout = null;
            if (saleoutStr != null) {
                saleout = Boolean.valueOf(saleoutStr);
            }
            if (saleout == null) {
                return false;
            }
            return Boolean.TRUE.equals(saleout);
        }

        /**
         * 设置一个批次是否售罄的缓存值，1分钟后失效删除
         *
         * @author xuhengfei
         * @param batchId
         * @param saleout
         * @since JDK 1.6
         */
        public void setSaleOut(long batchId, boolean saleout) {
            redisTemplate.opsForValue().set(getSaleOutKey(batchId), saleout + "");
            redisTemplate.expire(getSaleOutKey(batchId), 1, TimeUnit.MINUTES);
        }

        private String getSaleOutKey(long batchId) {
            return "goods-" + getClass().getSimpleName() + "-saleout-" + batchId;
        }
    }

    /**
     * ClassName: CouponQueueRedisCache <br/>
     * Function: 待发放的优惠劵队列Redis管理类 <br/>
     * date: 2016年6月5日 下午2:28:14 <br/>
     *
     * @author xuhengfei
     * @version GoodsCouponService4Speed
     * @since JDK 1.6
     */
    public class CouponQueueRedisCache {

        private RedisTemplate<String, String> redisTemplate;
        private BatchSaleoutRedisCache        batchSaleoutRedisCache;

        public CouponQueueRedisCache(RedisTemplate<String, String> redis, BatchSaleoutRedisCache batchSaleoutRedisCache) {
            this.redisTemplate = redis;
            this.batchSaleoutRedisCache = batchSaleoutRedisCache;
        }

        /**
         * 查询redis中该批次的队列长度
         *
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        public long getQueueSize(long goodsBatchId) {
            return redisTemplate.opsForSet().size(getBatchCouponsKey(goodsBatchId));
        }

        /**
         * 从redis的队列中取出一条
         *
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        public GoodsCouponEntity pop(long goodsBatchId) {
            GoodsCouponEntity entity = JSONObject.parseObject(redisTemplate.opsForSet().pop(getBatchCouponsKey(goodsBatchId)),
                                                              GoodsCouponEntity.class);
            return entity;
        }

        /**
         * 清理缓存
         * @param goodsBatchId
         */
        public void clearCache(Long goodsBatchId){
        	String key=getBatchCouponsKey(goodsBatchId);
        	redisTemplate.delete(key);
        }
        
        /**
         * 尝试加载数据库中的券码列表,并放入redis
         * 
         * @author xuhengfei
         * @param gtype
         * @param gid
         * @param batchId
         * @param limit 最大捞取条数
         * @return
         * @exception 获取锁失败，也会抛异常
         * @since JDK 1.6
         */
        public void tryLoadCouponByBatchId(GoodsTypeEnum gtype, long gid, long goodsBatchId, int limit) {
            // 分布式锁，防止并发
            boolean flag = redisTemplate.opsForValue().setIfAbsent(getBatchLockKey(goodsBatchId), "1");
            if (!flag) {
                throw new RuntimeGoodsException(ErrorCode.E0202004);
            } else {
                try {
                    DBTimeProfile.enter("redis tryLoadCouponByBatchId");
                    // 设置30秒超时失效
                    redisTemplate.expire(getBatchLockKey(goodsBatchId), 30, TimeUnit.SECONDS);
                    List<GoodsCouponEntity> list = GoodsCouponService4Speed.this.loadCouponByBatchId(gtype, gid,
                                                                                                     goodsBatchId,
                                                                                                     limit);
                    if (list.isEmpty()) {
                        // 如果查不出数据，标记已经售罄
                        batchSaleoutRedisCache.setSaleOut(goodsBatchId, true);
                    } else {
                        // 如果查到数据，将数据放入redis，并且标记未售罄
                        String[] array = new String[list.size()];
                        for (int i = 0; i < list.size(); i++) {
                            array[i] = JSONObject.toJSONString(list.get(i));
                        }
                        redisTemplate.opsForSet().add(getBatchCouponsKey(goodsBatchId), array);
                        // 队列数据1小时后失效
                        redisTemplate.expire(getBatchCouponsKey(goodsBatchId), 1, TimeUnit.HOURS);

                        batchSaleoutRedisCache.setSaleOut(goodsBatchId, false);
                    }

                } finally {
                    DBTimeProfile.release();
                    // 解锁
                    redisTemplate.delete(getBatchLockKey(goodsBatchId));
                }
            }
        }

        /**
         * redis 存放队列的key
         *
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        private String getBatchCouponsKey(Long goodsBatchId) {
            return RedisKeyTool.getRedisKey(this.getClass(), "batch_coupons", goodsBatchId + "");
        }

        /**
         * 批次的分布式锁，加锁后从数据库捞取数据
         * 
         * @author xuhengfei
         * @param goodsBatchId
         * @return
         * @since JDK 1.6
         */
        private String getBatchLockKey(Long goodsBatchId) {
            return RedisKeyTool.getRedisKey(this.getClass(), "batch_lock", goodsBatchId + "");
        }

    }

	@Override
	public Boolean deleteBatchUnusedCoupons(GoodsTypeEnum gtype, long gid, long batchId, List<Long> couponIds) {
		Boolean ret = super.deleteBatchUnusedCoupons(gtype, gid, batchId, couponIds);
		couponQueueRedisCache.clearCache(batchId);
		return ret;
	}

}
