package cn.com.duiba.goods.center.biz.service.item.impl;

import java.text.SimpleDateFormat;
import java.util.Date;

import cn.com.duiba.wolf.cache.CacheClient;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;

import cn.com.duiba.goods.center.api.remoteservice.dto.item.ItemKeyDto;
import cn.com.duiba.goods.center.biz.dao.item.EverydayLimitDAO;
import cn.com.duiba.goods.center.biz.entity.EverydayLimitEntity;
import cn.com.duiba.goods.center.biz.service.item.EverydayLimitService;
import cn.com.duiba.service.domain.dataobject.ItemDO;

@Service
public class EverydayLimitServiceImpl implements EverydayLimitService {

	private static Logger LOGGER = LoggerFactory.getLogger(EverydayLimitServiceImpl.class);

	@Autowired
	private EverydayLimitDAO everydayLimitDAO;
	
	@Autowired
	private CacheClient cacheClient;
	
	@Override
	public Boolean isEverydayLimit(ItemKeyDto itemKeyDto) {
		if (itemKeyDto.isItemMode()) {
			return itemKeyDto.getItemDto().isOpTypeItem(ItemDO.OpTypeQuantityLimit);
		} else if (itemKeyDto.isDuibaAppItemMode()) {
			if (itemKeyDto.getAppItemDto().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit)) {
				return true;
			} else if (itemKeyDto.getItemDto().isOpTypeItem(ItemDO.OpTypeQuantityLimit)) {
				return true;
			}
		} else if (itemKeyDto.isSelfAppItemMode()) {
			return itemKeyDto.getAppItemDto().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit);
		}
		return false;
	}
	
	@Override
	public Long findEverydayStock(ItemKeyDto itemKeyDto) {
		if (!isEverydayLimit(itemKeyDto)) {
			return null;
		}
		Integer limit = getLimit(itemKeyDto);
		if (limit == null) {
			return null;
		}
		Long appItemId = itemKeyDto.getAppItemDto() == null ? 0 : itemKeyDto.getAppItemDto().getId();
		Long itemId = itemKeyDto.getItemDto() == null ? 0 : itemKeyDto.getItemDto().getId();
		Integer sales = findEverydayLimitSales(appItemId, itemId);
		Integer stock = 0;
		if (sales < limit) {
			stock = limit - sales;
		}
		return Long.valueOf(stock);
	}

	@Override
	public Boolean decrEeverydayStock(ItemKeyDto itemKeyDto) {
		if (!isEverydayLimit(itemKeyDto)) {
			return null;
		}
		Integer limit = getLimit(itemKeyDto);
		if (limit == null) {
			return null;
		}
		Long appItemId = itemKeyDto.getAppItemDto() == null ? 0 : itemKeyDto.getAppItemDto().getId();
		Long itemId = itemKeyDto.getItemDto() == null ? 0 : itemKeyDto.getItemDto().getId();
		boolean ret = updateEverydaySales(appItemId, itemId, limit);
		// TODO DELETE 兼容写法之前每日限量是在MEMCACHED里面计数数据容易出现不一致现在以everydat_limit表为准
		deleteEverydayCache(itemKeyDto);
		return ret;
	}

	@Override
	public Boolean incrEverydayStock(ItemKeyDto itemKeyDto) {
		if (!isEverydayLimit(itemKeyDto)) {
			return null;
		}
		Long appItemId = itemKeyDto.getAppItemDto() == null ? 0 : itemKeyDto.getAppItemDto().getId();
		Long itemId = itemKeyDto.getItemDto() == null ? 0 : itemKeyDto.getItemDto().getId();
		rollbackEverydayLimit(appItemId, itemId);
		// TODO DELETE 兼容写法之前每日限量是在MEMCACHED里面计数数据容易出现不一致现在以everydat_limit表为准
		deleteEverydayCache(itemKeyDto);
		return true;
	}
	
	/**
	 * 获取限量商品限制数
	 * 
	 * @param itemKey
	 * @return
	 */
	private Integer getLimit(ItemKeyDto itemKey) {
		Integer limit = null;
		if (itemKey.isItemMode()) {
			limit = itemKey.getItemDto().getLimitEverydayQuantity();
		} else if (itemKey.isDuibaAppItemMode()) {
			if (itemKey.getItemDto().isOpTypeItem(ItemDO.OpTypeQuantityLimit)) {
				limit = itemKey.getItemDto().getLimitEverydayQuantity();
			} else if (itemKey.getAppItemDto().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit)) {
				limit = itemKey.getAppItemDto().getLimitEverydayQuantity();
			}
		} else if (itemKey.isSelfAppItemMode()) {
			limit = itemKey.getAppItemDto().getLimitEverydayQuantity();
		}
		return limit;
	}

	@Override
	public boolean updateEverydaySales(Long appItemId, Long itemId, int limit) {
		Date day = new Date();
		EverydayLimitEntity everyday = everydayLimitDAO.find(appItemId, itemId, day);
		if (everyday == null) {
			everyday = new EverydayLimitEntity();
			everyday.setAppItemId(appItemId);
			everyday.setItemId(itemId);
			everyday.setSales(0);
			everyday.setDay(day);
			try {
				everydayLimitDAO.insert(everyday);
			} catch (DuplicateKeyException e) {
				LOGGER.info("updateEverydaySales unique:appItemId:" + appItemId + " itemId:" + itemId + " day:" + new SimpleDateFormat("yyyy-MM-dd").format(day));
			}
		}
		int ret = everydayLimitDAO.updateSales(everyday.getId(), limit);
		if (ret < 1) {
			return false;
		}
		return true;
	}

	@Override
	public void rollbackEverydayLimit(Long appItemId, Long itemId) {
		Date day = new Date();
		EverydayLimitEntity everyday = everydayLimitDAO.find(appItemId, itemId, day);
		if (everyday == null) {
			return;
		}
		everydayLimitDAO.rollbackSales(everyday.getId());
	}

	@Override
	public Integer findEverydayLimitSales(Long appItemId, Long itemId) {
		Date day = new Date();
		EverydayLimitEntity everyday = everydayLimitDAO.find(appItemId, itemId, day);
		if (everyday == null) {
			return 0;
		}
		return everyday.getSales();
	}

	@Override
	public Integer findEverydayLimitSales(ItemKeyDto itemKeyDto) {
		Long appItemId = itemKeyDto.getAppItemDto() == null ? 0 : itemKeyDto.getAppItemDto().getId();
		Long itemId = itemKeyDto.getItemDto() == null ? 0 : itemKeyDto.getItemDto().getId();
		return findEverydayLimitSales(appItemId, itemId);
	}
	
	
	private static final String ITEM_NAMESPACE = "itemcache";
	private static final String APPITEM_NAMESPACE = "appitemcache";

	private void deleteEverydayCache(ItemKeyDto itemKey) {
		try {
			String key = getKey(itemKey);
			cacheClient.remove(key);
		} catch (Exception e) {
			LOGGER.error("deleteEverydayCache", e);
		}
	}

	private String getKey(ItemKeyDto itemKey) throws Exception {
		String key = null;
		if (itemKey.isItemMode()) {
			key = ITEM_NAMESPACE + itemKey.getItemDto().getId();
		} else if (itemKey.isDuibaAppItemMode()) {
			if (itemKey.getItemDto().isOpTypeItem(ItemDO.OpTypeQuantityLimit)) {
				key = ITEM_NAMESPACE + itemKey.getItemDto().getId();
			} else if (itemKey.getAppItemDto().isOpTypeAppItem(ItemDO.OpTypeQuantityLimit)) {
				key = APPITEM_NAMESPACE + itemKey.getAppItemDto().getId();
			}
		} else if (itemKey.isSelfAppItemMode()) {
			key = APPITEM_NAMESPACE + itemKey.getAppItemDto().getId();
		}
		return key;
	}

}
